/*
 * threadCalls.c
 * Support for threaded ops which may block (read, write, connect, etc.).
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#include "config.h"
#include <stdlib.h>
#include <sys/types.h>
#if defined(HAVE_UNISTD_H)
#include <unistd.h>
#endif
#include "lerrno.h"
#include <assert.h>
#if defined(HAVE_SYS_SOCKET_H)
#include <sys/socket.h>
#endif
#if defined(HAVE_SYS_TIME_H)
#include <sys/time.h>
#endif
#if defined(HAVE_WINSOCK_H)
#include <winsock.h>
#endif
#if defined(HAVE_SYS_IOCTL_H)
#include <sys/ioctl.h>
#endif
#if defined(HAVE_IO_H)
#include <io.h>
#endif
#include "object.h"
#include "thread.h"
#include "md.h"

#if !defined(HAVE_MEMCPY)
#define memcpy(d, s, n) bcopy ((s), (d), (n))
#endif
void bcopy(void*, void*, size_t);

/* QUICK HACK!! */
#if defined(__WIN32__)
#define	ioctl	ioctlsocket
#endif

// sjL: 17.vii.96

#ifdef BEOS
#include <net/socket.h>
#define FD_SETSIZE sizeof(unsigned long)
#endif

#define	TH_READ		0
#define	TH_WRITE	1

static int maxFd;
static fd_set readsPending;
static fd_set writesPending;
static thread* readQ[FD_SETSIZE];
static thread* writeQ[FD_SETSIZE];
static struct timeval tm = { 0, 0 };

static void blockOnFile(int, int);
void waitOnEvents(void);

extern thread* currentThread;

/*
 * Threaded create socket.
 */
int
threadedCreateSocket(int af, int type, int proto)
{
	int fd;
	int r;
	int on = 1;
	int pid;

	fd = socket(af, type, proto);

#if defined(USE_INTERNAL_THREADS)

#if defined(FIONBIO)
	/* Make non-blocking */
	r = ioctl(fd, FIONBIO, &on);
	if (r < 0) {
		return (r);
	}
#endif
#if defined(FIOSETOWN) && defined(FIOASYNC)
	/* Allow socket to signal this process when new data is available */
	pid = getpid();
	r = ioctl(fd, FIOSETOWN, &pid);
	if (r < 0) {
		return (r);
	}
	r = ioctl(fd, FIOASYNC, &on);
	if (r < 0) {
		return (r);
	}
#endif

#endif

	return (fd);
}

/*
 * Threaded socket connect.
 */
int
threadedConnect(int fd, struct sockaddr* addr, int len)
{
	int r;

	r = connect(fd, addr, len);
#if defined(USE_INTERNAL_THREADS)
	if (r < 0 & (errno == EINPROGRESS || errno == EALREADY || errno == EWOULDBLOCK)) {
		blockOnFile(fd, TH_READ);
		r = 0; /* Assume it's okay when we get released */
	}
#endif

	return (r);
}

/*
 * Threaded socket accept.
 */
int
threadedAccept(int fd, struct sockaddr* addr, int* len)
{
	int r;
	int on = 1;

#if !defined(USE_INTERNAL_THREADS)
	r = accept(fd, addr, len);
#else
	for (;;) {
		r = accept(fd, addr, len);
		if (r >= 0 || !(errno == EINPROGRESS || errno == EALREADY || errno == EAGAIN)) {
			break;
		}
		blockOnFile(fd, TH_READ);
	}

#if defined(FIONBIO)
	/* Make non-blocking - apparently this isn't inherited! */
	if (ioctl(r, FIONBIO, &on) < 0) {
		return (-1);
	}
#endif

#endif
	return (r);
}

/*
 * Read but only if we can.
 */
int
threadedRead(int fd, char* buf, int len)
{
	int r;

	for (;;) {
		r = read(fd, buf, len);
#if defined(USE_INTERNAL_THREADS)
		if (r >= 0 || errno != EWOULDBLOCK) {
			return (r);
		}
		blockOnFile(fd, TH_READ);
#else
		return (r);
#endif
	}
}

/*
 * Write but only if we can.
 */
int
threadedWrite(int fd, char* buf, int len)
{
	int r;
	char* ptr;

	ptr = buf;
	do {
		r = write(fd, ptr, len);
#if defined(USE_INTERNAL_THREADS)
		if (r < 0 && errno == EWOULDBLOCK) {
			blockOnFile(fd, TH_WRITE);
			r = 1;
			continue;
		}
#endif
		ptr += r;
		len -= r;

	} while (len > 0 && r > 0);

	return (ptr - buf);
}

#if defined(USE_INTERNAL_THREADS)
/*
 * An attempt to access a file would block, so suspend the thread until
 * it will happen.
 */
static
void
blockOnFile(int fd, int op)
{
	intsDisable();

	if (fd > maxFd) {
		maxFd = fd;
	}
	if (op == TH_READ) {
		FD_SET(fd, &readsPending);
		suspendOnQThread(currentThread, &readQ[fd]);
		FD_CLR(fd, &readsPending);
	}
	else {
		FD_SET(fd, &writesPending);
		suspendOnQThread(currentThread, &writeQ[fd]);
		FD_CLR(fd, &writesPending);
	}

	intsRestore();
}

/*
 * Check if some file descriptor or other event to become ready.
 *  Block if required.
 */
void
checkEvents(bool block)
{
	int r;
	fd_set rd;
	fd_set wr;
	thread* tid;
	thread* ntid;
	int i;

	intsDisable();

#if defined(FD_COPY)
	FD_COPY(&readsPending, &rd);
	FD_COPY(&writesPending, &wr);
#else
	memcpy(&rd, &readsPending, sizeof(rd));
	memcpy(&wr, &writesPending, sizeof(wr));
#endif
	r = select(maxFd+1, &rd, &wr, 0, (block ? 0 : &tm));

	for (i = 0; r > 0 && i <= maxFd; i++) {
		if (readQ[i] != 0 && FD_ISSET(i, &rd)) {
			for (tid = readQ[i]; tid != 0; tid = ntid) {
				ntid = tid->next;
				resumeThread(tid);
			}
			readQ[i] = 0;
			r--;
		}
		if (writeQ[i] != 0 && FD_ISSET(i, &wr)) {
			for (tid = writeQ[i]; tid != 0; tid = ntid) {
				ntid = tid->next;
				resumeThread(tid);
			}
			writeQ[i] = 0;
			r--;
		}
	}

	intsRestore();
}
#endif
