/*
 * thread.c
 * Thread support.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#define	DBG(s) 

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include "gtypes.h"
#include "access.h"
#include "object.h"
#include "constants.h"
#include "classMethod.h"
#include "baseClasses.h"
#include "thread.h"
#include "locks.h"
#include "md.h"

#if defined(INTERPRETER)
#include <setjmp.h>
extern jmp_buf* cjbuf;
#endif

#ifdef BEOS
#pragma export on
#endif
thread* currentThread;
#ifdef BEOS
#pragma export off
#endif

thread* threadQhead[MAX_THREAD_PRIO + 1];
thread* threadQtail[MAX_THREAD_PRIO + 1];

thread* garbageman;
thread* finalman;

thread* liveThreads;
classes* ThreadClass;

int blockInts;
bool needReschedule;

static int talive;
static int tdaemon;

void reschedule(void);
static void firstStartThread(void);
static thread* startDaemon(void*);
void finaliserMan(void);
void gcMan(void);
void checkEvents(bool);

long do_execute_java_method(void*, object*, char*, char*, struct _methods*, int, ...);
object* execute_java_constructor(void*, char*, classes*, char*, ...);

/* Setup default thread stack size - this can be overwritten if required */
int threadStackSize = THREADSTACKSIZE;

#define	ALLOCTHREADSTACK()	malloc(threadStackSize)


/*
 * Initialise threads.
 */
void
initThreads(void)
{
	/* Get a handle on the thread class */
	ThreadClass = lookupClass(addString(THREADCLASS));
	assert(ThreadClass != 0);

	/* Allocate a thread to be the main thread */
	currentThread = (thread*)alloc_object(ThreadClass, false);
	assert(currentThread != 0);
	currentThread->PrivateInfo = calloc(sizeof(ctx), 1);
	liveThreads = currentThread;

	currentThread->name = 0;
	currentThread->priority = NORM_THREAD_PRIO;
	currentThread->PrivateInfo->priority = NORM_THREAD_PRIO;
	currentThread->next = 0;
	currentThread->PrivateInfo->status = THREAD_SUSPENDED;
	assert(currentThread->PrivateInfo != 0);
	/* THREADINFO(currentThread->PrivateInfo); */
	currentThread->PrivateInfo->nextlive = 0;
	currentThread->single_step = 0;
	currentThread->daemon = 0;
	currentThread->stillborn = 0;
	currentThread->target = 0;
	currentThread->interruptRequested = 0;
	currentThread->group = (threadGroup*)execute_java_constructor(0, "java.lang.ThreadGroup", 0, "()V");
	assert(currentThread->group != 0);

	talive++;

	/* Add thread into runQ */
	resumeThread(currentThread);

	/* Start any daemons we need */
	finalman = startDaemon(&finaliserMan);
	garbageman = startDaemon(&gcMan);
}

#ifdef BEOS
#pragma export on
#endif

/*
 * Start a new thread running.
 */
void
startThread(thread* tid)
{
	/* Allocate a stack context */
	assert(tid->PrivateInfo == 0);
	tid->PrivateInfo = calloc(sizeof(ctx), 1);
	assert(tid->PrivateInfo != 0);
	tid->PrivateInfo->nextlive = liveThreads;
	liveThreads = tid;
	tid->PrivateInfo->stackBase = ALLOCTHREADSTACK();
	assert(tid->PrivateInfo->stackBase != 0);
	tid->PrivateInfo->stackEnd = tid->PrivateInfo->stackBase + THREADSTACKSIZE;
	tid->PrivateInfo->status = THREAD_SUSPENDED;
	tid->PrivateInfo->priority = tid->priority;

	/* Construct the initial restore point. */
	/* THREADINIT(tid->PrivateInfo, &firstStartThread); */

	talive++;
	if (tid->daemon) {
		tdaemon++;
	}

	/* Add thread into runQ */
	resumeThread(tid);
}

#ifdef BEOS
#pragma export off
#endif

/*
 * All threads start here.
 */
static
void
firstStartThread(void)
{
DBG(	printf("firstStartThread %x\n", currentThread);		)
	/* Every thread starts with the interrupts off */
	intsRestore();

	/* Find the run()V method and call it */
	do_execute_java_method(0, (object*)currentThread, "run", "()V", 0, 0);
	do_execute_java_method(0, (object*)currentThread, "exit", "()V", 0, 0);
	killThread(0);
}

#ifdef BEOS
#pragma export on
#endif

/*
 * Resume a thread running.
 */
void
resumeThread(thread* tid)
{
DBG(	printf("resumeThread %x\n", tid);			)

	intsDisable();

	if (tid->PrivateInfo->status != THREAD_RUNNING) {
		tid->PrivateInfo->status = THREAD_RUNNING;

		/* Place thread on the end of its queue */
		if (threadQhead[tid->PrivateInfo->priority] == 0) {
			threadQhead[tid->PrivateInfo->priority] = tid;
			threadQtail[tid->PrivateInfo->priority] = tid;
			if (tid->PrivateInfo->priority > currentThread->PrivateInfo->priority) {
				needReschedule = true;
			}
		}
		else {
			threadQtail[tid->PrivateInfo->priority]->next = tid;
			threadQtail[tid->PrivateInfo->priority] = tid;
		}
	}
	intsRestore();
}

/*
 * Yield process to another thread of equal priority.
 */
void
yieldThread()
{
	intsDisable();

	if (threadQhead[currentThread->PrivateInfo->priority] != threadQtail[currentThread->PrivateInfo->priority]) {

		/* Get the next thread and move me to the end */
		threadQhead[currentThread->PrivateInfo->priority] = currentThread->next;
		threadQtail[currentThread->PrivateInfo->priority]->next = currentThread;
		threadQtail[currentThread->PrivateInfo->priority] = currentThread;
		currentThread->next = 0;
		needReschedule = true;
	}

	intsRestore();
}

/*
 * Suspend a thread.
 */
void
suspendThread(thread* tid)
{
	thread** ntid;

	intsDisable();

	if (tid->PrivateInfo->status != THREAD_SUSPENDED) {
		tid->PrivateInfo->status = THREAD_SUSPENDED;

		for (ntid = &threadQhead[tid->PrivateInfo->priority]; *ntid != 0; ntid = &(*ntid)->next) {
			if (*ntid == tid) {
				*ntid = tid->next;
				if (tid == currentThread) {
					reschedule();
				}
				break;
			}
		}
	}

	intsRestore();
}

#ifdef BEOS
#pragma export off
#endif

/*
 * Suspend a thread on a queue.
 */
void
suspendOnQThread(thread* tid, thread** queue)
{
	thread** ntid;

DBG(	printf("suspendOnQThread %x\n", tid);		)

	assert(blockInts == 1);

	if (tid->PrivateInfo->status != THREAD_SUSPENDED) {
		tid->PrivateInfo->status = THREAD_SUSPENDED;

		for (ntid = &threadQhead[tid->PrivateInfo->priority]; *ntid != 0; ntid = &(*ntid)->next) {
			if (*ntid == tid) {
				*ntid = tid->next;
				/* Insert onto head of lock wait Q */
				tid->next = *queue;
				*queue = tid;
				if (tid == currentThread) {
					reschedule();
				}
				break;
			}
		}
	}
}

#ifdef BEOS
#pragma export on
#endif

/*
 * Kill thread.
 */
void
killThread(thread* tid)
{
	thread** ntid;

	intsDisable();

	/* A null tid means the current thread */
	if (tid == 0) {
		tid = currentThread;
	}

DBG(	printf("killThread %x\n", tid);			)

	if (tid->PrivateInfo->status != THREAD_DEAD) {

		/* Get thread off runq (if it needs it) */
		if (tid->PrivateInfo->status == THREAD_RUNNING) {
			for (ntid = &threadQhead[tid->PrivateInfo->priority]; *ntid != 0; ntid = &(*ntid)->next) {
				if (*ntid == tid) {
					*ntid = tid->next;
					break;
				}
			}
		}

		tid->PrivateInfo->status = THREAD_DEAD;
		talive--;
		if (tid->daemon) {
			tdaemon--;
		}

		/* If we only have daemons left, then everyone is dead. */
		if (talive == tdaemon) {
			/* Am I suppose to close things down nicely ?? */
			exit(0);
		}

		/* Notify on the object just in case anyone is waiting */
		lockMutex(&tid->obj);
		broadcastCond(&tid->obj); 
		unlockMutex(&tid->obj);

		/* Remove thread from live list to it can be garbaged */
		for (ntid = &liveThreads; *ntid != 0; ntid = &(*ntid)->PrivateInfo->nextlive) {
			if (tid == (*ntid)) {
				(*ntid) = tid->PrivateInfo->nextlive;
				break;
			}
		}

		/* Run something else */
		needReschedule = true;
	}
	intsRestore();
}

/*
 * Change thread priority.
 */
void
setPriorityThread(thread* tid, int prio)
{
	thread** ntid;

	if (tid->PrivateInfo == 0) {
		tid->priority = prio;
		return;
	}

	if (tid->PrivateInfo->status == THREAD_SUSPENDED) {
		tid->PrivateInfo->priority = prio;
		return;
	}

	intsDisable();

	/* Remove from current thread list */
	for (ntid = &threadQhead[tid->PrivateInfo->priority]; *ntid != 0; ntid = &(*ntid)->next) {
		if (*ntid == tid) {
			*ntid = tid->next;
			break;
		}
	}

	/* Insert onto a new one */
	tid->PrivateInfo->priority = prio;
	tid->priority = prio;
	if (threadQhead[prio] == 0) {
		threadQhead[prio] = tid;
		threadQtail[prio] = tid;
		if (prio > currentThread->PrivateInfo->priority) {
			needReschedule = true;
		}
	}
	else {
		threadQtail[prio]->next = tid;
		threadQtail[prio] = tid;
	}

	intsRestore();
}

/*
 * Put a thread to sleep.
 */
void
sleepThread(int64 time)
{
	abort();
}

/*
 * Is this thread alive?
 */
bool
aliveThread(thread* tid)
{
	return (tid->PrivateInfo->status != THREAD_DEAD ? true : false);
}

/*
 * How many stack frames have I invoked?
 */
long
framesThread(thread* tid)
{
	long count;
	/* THREADFRAMES(tid, count); */
	return (count);
}

#ifdef BEOS
#pragma export off
#endif

/*
 * Reschedule the thread.
 * Called whenever a change in the running thread is required.
 */
void
reschedule(void)
{
	int i;
	thread* lastThread;
	int b;

#if defined(USE_INTERNAL_THREADS)
	/* Check events - we may release a high priority thread */
	checkEvents(false);
#endif

	for (;;) {
		for (i = MAX_THREAD_PRIO; i >= MIN_THREAD_PRIO; i--) {
			if (threadQhead[i] != 0) {
				if (threadQhead[i] != currentThread) {
					b = blockInts;
#if defined(INTERPRETER)
					currentThread->PrivateInfo->exceptPtr = (void*)cjbuf;
					cjbuf = 0;
#endif
					lastThread = currentThread;
					currentThread = threadQhead[i];
DBG( printf("ctxswitch %x -> %x\n", lastThread, currentThread); )
/*					THREADSWITCH(currentThread->PrivateInfo, lastThread->PrivateInfo); */
#if defined(INTERPRETER)
					cjbuf = (jmp_buf*)currentThread->PrivateInfo->exceptPtr;
#endif
					blockInts = b;
				}
				/* Now we kill the schedule and turn ints
				   back on */
				needReschedule = false;
				return;
			}
		}
#if defined(USE_INTERNAL_THREADS)
		/* Nothing to run - wait for external event */
		checkEvents(true);
#endif
	}
}

/*
 * Start a daemon thread.
 */
static
thread*
startDaemon(void* func)
{
	thread* tid;

	/* Keep daemon threads as root objects */
	tid = (thread*)alloc_object(ThreadClass, true);
	assert(tid != 0);
	tid->PrivateInfo = calloc(sizeof(ctx), 1);

	tid->name = 0;
	tid->priority = MAX_THREAD_PRIO;
	tid->PrivateInfo->priority = MAX_THREAD_PRIO;
	tid->next = 0;
	tid->PrivateInfo->status = THREAD_SUSPENDED;
	assert(tid->PrivateInfo != 0);

	/* Insert into live list as long as it's not the finaliser */
	if (func != &finaliserMan) {
		tid->PrivateInfo->nextlive = liveThreads;
		liveThreads = tid;
	}

	tid->PrivateInfo->stackBase = ALLOCTHREADSTACK();
	assert(tid->PrivateInfo->stackBase != 0);
	tid->PrivateInfo->stackEnd = tid->PrivateInfo->stackBase + THREADSTACKSIZE;
	tid->single_step = 0;
	tid->daemon = 1;
	tid->stillborn = 0;
	tid->target = 0;
	tid->interruptRequested = 0;
	tid->group = 0;

	/* Construct the initial restore point. */
	/* THREADINIT(tid->PrivateInfo, func); */

	talive++;
	tdaemon++;

	/* Add thread into runQ */
	resumeThread(tid);
	return (tid);
}
