/*
 * locks.c
 * Manage locking system
 * This include the mutex's and cv's.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#include <assert.h>
#include "object.h"
#include "baseClasses.h"
#include "thread.h"
#include "locks.h"
#include "errors.h"
#include "exception.h"

extern thread* currentThread;

#if defined(HAVE_NATIVE_INT64)
jlong	waitforever = -1;
#else
jlong	waitforever = { -1, -1 };
#endif

/*
 * Lock a mutex.
 */
void
lockMutex(object* obj)
{
	syncObject* lk = &obj->lock;

	intsDisable();

	if (lk->holder == 0) {
		lk->holder = currentThread;
		lk->count = 1;
	}
	else if (lk->holder == currentThread) {
		lk->count++;
	}
	else {
		while (lk->holder != 0) {
			suspendOnQThread(currentThread, &lk->muxWaiters);
		}
		lk->holder = currentThread;
		lk->count = 1;
	}

	intsRestore();
}

/*
 * Release a mutex.
 */
void
unlockMutex(object* obj)
{
	thread* tid;
	syncObject* lk = &obj->lock;

	intsDisable();

	assert(lk->holder == currentThread);

	lk->count--;
	if (lk->count == 0) {
		lk->holder = 0;
		if (lk->muxWaiters != 0) {
			tid = lk->muxWaiters;
			lk->muxWaiters = tid->next;
			resumeThread(tid);
		}
	}

	intsRestore();
}

#ifdef BEOS
#pragma export on
#endif

/*
 * Wait on a conditional variable.
 *  (timeout currently ignored)
 */
void
waitCond(object* obj, jlong timeout)
{
	int count;
	syncObject* lk = &obj->lock;

	if (lk->holder != currentThread) {
		throwException(IllegalMonitorStateException);
	}

	intsDisable();

	count = lk->count;
	lk->holder = 0;
	lk->count = 0;

	/* Suspend, and keep suspended until I re-get the lock */
	suspendOnQThread(currentThread, &lk->cvWaiters);
	while (lk->holder != 0) {
		suspendOnQThread(currentThread, &lk->muxWaiters);
	}

	lk->holder = currentThread;
	lk->count = count;

	intsRestore();
}

/*
 * Wake one thread on a conditional variable.
 */
void
signalCond(object* obj)
{
	thread* tid;
	syncObject* lk = &obj->lock;

	if (lk->holder != currentThread) {
		throwException(IllegalMonitorStateException);
	}

	intsDisable();

	/* Remove one thread from cv list */
	if (lk->cvWaiters != 0) {
		tid = lk->cvWaiters;
		lk->cvWaiters = tid->next;

		/* Place it on mux list */
		tid->next = lk->muxWaiters;
		lk->muxWaiters = tid;
	}

	intsRestore();
}

/*
 * Wake all threads on a conditional variable.
 */
void
broadcastCond(object* obj)
{
	thread** tidp;
	syncObject* lk = &obj->lock;

	if (lk->holder != currentThread) {
		throwException(IllegalMonitorStateException);
	}

	intsDisable();

	/* Find the end of the cv list */
	if (lk->cvWaiters) {
		for (tidp = &lk->cvWaiters; *tidp != 0; tidp = &(*tidp)->next)
			;

		/* Place entire cv list on mux list */
		(*tidp) = lk->muxWaiters;
		lk->muxWaiters = lk->cvWaiters;
		lk->cvWaiters = 0;
	}

	intsRestore();
}

#ifdef BEOS
#pragma export off
#endif