/*
 * exception.c
 * Handle exceptions for the interpreter.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, May 1996.
 */

#include "config.h"
#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>
#include "gtypes.h"
#include "access.h"
#include "object.h"
#include "constants.h"
#include "classMethod.h"
#include "exception.h"
#include "baseClasses.h"
#include "lookup.h"
#include "thread.h"
#include "md.h"

object*	ClassFormatError;
object*	LinkageError;
object*	ClassNotFoundException;
object*	NoSuchFieldError;
object*	NoSuchMethodError;
object*	OutOfMemoryError;
object*	UnsatisfiedLinkError;
object*	VirtualMachineError;
object*	ClassCircularityError;
object*	NegativeArraySizeException;
object*	ClassCastException;
object*	IllegalMonitorStateException;
object*	NullPointerException;
object*	ArithmeticException;
object*	ArrayIndexOutOfBoundsException;

static void dispatchException(object*);

static void nullException(int);
static void arithmeticException(int);

extern object* exceptionObject;
extern jmp_buf* cjbuf;

#if defined(__WIN32__)
#define	SIG_T	void(*)()
#else
#define	SIG_T	void*
#endif

/*
 * Setup the internal exceptions.
 */
void
initExceptions(void)
{
#define	EXCEPTION(s) alloc_object(lookupClass(addString(#s)), true)

	ClassFormatError = EXCEPTION(java/lang/ClassFormatError);
	LinkageError = EXCEPTION(java/lang/LinkageError);
	ClassNotFoundException = EXCEPTION(java/lang/ClassNotFoundException);
	NoSuchFieldError = EXCEPTION(java/lang/NoSuchFieldError);
	NoSuchMethodError = EXCEPTION(java/lang/NoSuchMethodError);
	OutOfMemoryError = EXCEPTION(java/lang/OutOfMemoryError);
	UnsatisfiedLinkError = EXCEPTION(java/lang/UnsatisfiedLinkError);
	VirtualMachineError = EXCEPTION(java/lang/VirtualMachineError);
	ClassCircularityError = EXCEPTION(java/lang/ClassCircularityError);
	NegativeArraySizeException = EXCEPTION(java/lang/NegativeArraySizeException);
	ClassCastException = EXCEPTION(java/lang/ClassCastException);
	IllegalMonitorStateException = EXCEPTION(java/lang/IllegalMonitorStateException);
	NullPointerException = EXCEPTION(java/lang/NullPointerException);
	ArithmeticException = EXCEPTION(java/lang/ArithmeticException);
	ArrayIndexOutOfBoundsException = EXCEPTION(java/lang/ArrayIndexOutOfBoundsException);

	initClasses();

#if !defined(DEBUG)
	/* Catch signal we need to convert to exceptions */
#if defined(SIGSEGV)
	signal(SIGSEGV, (SIG_T)nullException);
#endif
#if defined(SIGBUS)
	signal(SIGBUS, (SIG_T)nullException);
#endif
#if defined(SIGFPE)
	signal(SIGFPE, (SIG_T)arithmeticException);
#endif
#if defined(SIGPIPE)
	signal(SIGPIPE, SIG_IGN);
#endif
#endif
}

/*
 * Throw an exception.
 */
void
throwException(object* eobj)
{
	if (eobj == 0) {
		fprintf(stderr, "Exception thrown on null object ... aborting\n");
		abort();
		exit(1);
	}
	dispatchException(eobj);
}

/*
 * Search for a matching exception.
 */
static
void
dispatchException(object* eobj)
{
	if (cjbuf != 0) {
		exceptionObject = eobj;
		longjmp(*cjbuf, 1);
	}

	fprintf(stderr, "Failed to catch exception ... aborting:\n\t%s\n", eobj->mtable->class->name);
	fflush(stderr);
	exit(1);
}

/*
 * Null exception - catches bad memory accesses.
 */
static
void
nullException(int sig)
{
#if !defined(DEBUG)
	/* Reset signal handler - necessary for SysV, does no harm for BSD */
	signal(sig, (SIG_T)nullException);
#endif
	dispatchException(NullPointerException);
}

/*
 * Division by zero.
 */
static
void
arithmeticException(int sig)
{
#if !defined(DEBUG)
        /* Reset signal handler - necessary for SysV, does no harm for BSD */
        signal(sig, (SIG_T)arithmeticException);
#endif
	dispatchException(ArithmeticException);
}
