/*
 * gc.c
 * The garbage collector.
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

/* #define	NOGC	/* Turn off garbage collection */

#define	DBG(s)
#define	MDBG(s)
#define	FDBG(s)
#define	ADBG(s)
#define	TDBG(s)

#include "config.h"
#include <stdlib.h>
#include <assert.h>
#include <stdarg.h>
#if defined(HAVE_MALLOC_H)
#include <malloc.h>
#endif
#include "gtypes.h"
#include "access.h"
#include "object.h"
#include "constants.h"
#include "classMethod.h"
#include "baseClasses.h"
#include "lookup.h"
#include "thread.h"
#include "gc.h"
#include "locks.h"
#include "slots.h"
#include "external.h"
#include "machine.h"
#include "md.h"

extern classes* ThreadClass;
extern strpair* finalpair;
extern thread* finalman;
extern thread* garbageman;
extern thread* liveThreads;
extern thread* currentThread;

static gcRef* refTable[GCREFTABLESZ];
static int refTableIdx;
static gcRef* freeRef;
static gcRef* permList;
static void* minGcMemory = (void*)-1;
static void* maxGcMemory = (void*)0;
static gcRef* garbageObjects;
static unsigned int allocCount;

/* When's a good time to run the GC? Run it every 1000 allocations */
#define	GCTRIGCOUNT			1000

static void markObject(object*);
static void garbageCollect(void);
static gcRef* validReference(object*);

/*
 * Add an object into the garbage collection scheme.
 */
void
add_object(object* o, bool perm)
{
	gcRef* table;
	gcRef* ref;
	int i;

#if defined(NOGC)
	return;
#endif

	/* Reset min and max memory pointers */
	if ((void*)o < minGcMemory) {
		minGcMemory = (void*)o;
	}
	if ((void*)o > maxGcMemory) {
		maxGcMemory = (void*)o;
	}

	ref = freeRef;
	if (ref != 0) {
		freeRef = freeRef->next;
	}
	else {
		assert(refTableIdx < GCREFTABLESZ);
		table = malloc(sizeof(gcRef) * GCREFMAX);
		assert(table != 0);

		/* Build into free list */
		for (i = 0; i < GCREFMAX; i++) {
			table[i].flags = GC_FREE;
			table[i].next = &table[i+1];
			table[i].idx = i + (refTableIdx * GCREFMAX);
		}
		table[GCREFMAX-1].next = 0;
		freeRef = &table[1];
		ref = &table[0];

		refTable[refTableIdx] = table;
		refTableIdx++;
	}

	assert(ref->flags == GC_FREE);

	/* Make ref point at object, and object point at ref */
	o->idx = ref->idx;
	ref->obj = o;
	ref->flags = GC_UNMARK;
	ref->next = 0;

	/* If object is permenant, put it on the perm list */
	if (perm == true) {
		ref->next = permList;
		permList = ref;
	}

ADBG(	printf("Adding new object 0x%x at index %d - %s\n", o, o->idx, o->mtable ? o->mtable->class->name : "<none>"); )

	if (++allocCount % GCTRIGCOUNT == 0) {
		invokeGarbageCollector();
	}
}

#ifdef BEOS
#pragma export on
#endif

/*
 * Invoke the garbage collector (if we have one)
 */
void
invokeGarbageCollector(void)
{
	if (garbageman != 0) {
		lockMutex(&garbageman->obj);
		signalCond(&garbageman->obj);
		unlockMutex(&garbageman->obj);
	}
}

#ifdef BEOS
#pragma export off
#endif

/*
 * Is object reference a real object?
 */
static
gcRef*
validReference(object* o)
{
	gcRef* ref;

	/* Validate object address.  To be a real object it must lie
	   inside the malloced memory pool, and its index must refer to
	   a GC entry which refers to it. */
	if ((void*)o < minGcMemory || (void*)o > maxGcMemory) {
		return (0);
	}
	/* Check alignment - should be pointer aligned */
	if (((uintp)o) % sizeof(void*) != 0) {
		return (0);
	}
	ref = refTable[(o->idx / GCREFMAX) % GCREFTABLESZ];
	if (ref == 0) {
		return (0);
	}
	ref = &ref[o->idx % GCREFMAX];
	if (ref->obj != o) {
		return (0);
	}

	/*
	 * If object is unmarked, then we should return if and
	 * let the system recurse though it.  If not, we should
	 * return 0 so the system will not recurse.
	 */
	if (ref->flags == GC_UNMARK || ref->flags == GC_UNMARK2) {
		return (ref);
	}
	else {
		assert(ref->flags != GC_GARBAGE);
		assert(ref->flags != GC_GARBAGE2);
		assert(ref->flags != GC_FREE);
		return (0);
	}
}

/*
 * Garbage collect objects.
 */
static
void
garbageCollect(void)
{
	int i;
	thread* tid;
	object** ptr;
	gcRef* ref;
	int j;
	gcRef* table;

#if defined(NOGC)
	return;
#endif

DBG(	printf("Garbage collecting ... \n");				)

	/* Scan the permentant object list */
	for (ref = permList; ref != 0; ref = ref->next) {
		markObject(ref->obj);
	}

        /* Scan each live thread */
	for (tid = liveThreads; tid != 0; tid = tid->PrivateInfo->nextlive) {
		markObject(&tid->obj);
		for (ptr = (object**)tid->PrivateInfo->restorePoint; ptr < (object**)tid->PrivateInfo->stackEnd; ptr++) {
			markObject(*ptr);
		}
	}

	/* Look through object table for unmarked objects which must be
	 * finalises.  Attach them to the garbage list and then reference
	 * them (so any objects the reference are not garbaged).
	 */
	for (j = 0; j < refTableIdx; j++) {
		table = refTable[j];
		for (i = 0; i < GCREFMAX; i++) {
			if (table[i].flags == GC_UNMARK &&
			    table[i].obj->mtable->class->final == true) {
				markObject(table[i].obj);
				table[i].flags = GC_GARBAGE;
				table[i].next = garbageObjects;
				garbageObjects = &table[i];
			}
		}
	}

	/* Now look through object table for objects which have not
	   been marked.  These can be freed. */
	for (j = 0; j < refTableIdx; j++) {
		table = refTable[j];
		for (i = 0; i < GCREFMAX; i++) {
			switch(table[i].flags) {
			case GC_GARBAGE:
			case GC_GARBAGE2:
			case GC_FREE:
				break;

			case GC_MARK:
				table[i].flags = GC_UNMARK; /* For next time */
				break;

			case GC_UNMARK:
				assert(table[i].obj->mtable->class->final == false);
				table[i].flags = GC_GARBAGE2;
				table[i].next = garbageObjects;
				garbageObjects = &table[i];
				break;

			case GC_UNMARK2:
				/* Object not marked - free */
				table[i].flags = GC_GARBAGE2;
				table[i].next = garbageObjects;
				garbageObjects = &table[i];
				break;
			}
		}
	}

DBG(	printf("Garbage collecting ... done.\n");			)
}

/*
 * Given an object, mark it and follow any object references it contains.
 */
static
void
markObject(object* o)
{
	object** child;
	classes* class;
	int i;
	gcRef* ref;

	/* Null object reference */
	if (o == 0) {
		return;
	}

	/* Invalid object or already visited? */
	ref = validReference(o);
	if (ref == 0) {
		return;
	}

MDBG(	printf("Marking object 0x%x at index %d - %s\n", o, o->idx, o->mtable ? o->mtable->class->name : "<none>"); )

	/* Mark this object */
	ref->flags = GC_MARK;

	class = o->mtable->class;

	/* Mark the class object */
	markObject(&class->head);

	/* If this is a class object, mark the static fields and constants */
	if (class == ClassClass) {
		class = (classes*)o;
		markObject(&class->superclass->head);
		child = (object**)class->staticFields;
		for (i = 0; i < class->sfsize; i++) {
			markObject(child[i]);
		}
		if (class->constants != 0) {
			child = (object**)class->constants->data;
			for (i = class->constants->size - 1; i >= 0; i--) {
				markObject(child[i]);
			}
		}
	}
	/* Otherwise, if a normal object ... */
	else if (o->mtable->class->sig[0] == 'L') {
		child = (object**)(o+1);
		for (i = 0; i < o->mtable->class->fsize; i++) {
			markObject(child[i]);
		}
	}
	/* Otherwise, if a array of objects ... */
	else if (o->mtable->class->sig[0] == '[' &&
		 (o->mtable->class->sig[1] == 'L' ||
		  o->mtable->class->sig[1] == '[')) {
		child = (object**)(o+1);
		for (i = 0; i < o->size; i++) {
			markObject(child[i]);
		}
	}
}

/*
 * Finaliser.
 * Finalises any objects which have been garbage collected before
 *   deleting them.
 */
void
finaliserMan(void)
{
	object* obj;
	gcRef* ref;
	void* func;

	/* All threads start with interrupts disabled */
	intsRestore();

	lockMutex(&currentThread->obj);
	for (;;) {
		while (garbageObjects) {
			ref = garbageObjects;
			garbageObjects = garbageObjects->next;
			ref->next = 0;

			assert(ref->flags == GC_GARBAGE || ref->flags == GC_GARBAGE2);
			obj = ref->obj;

			/* Finalise object if necessary */
			if (ref->flags == GC_GARBAGE) {
				func = findMethod(obj->mtable->class, finalpair);
				assert(func != 0);
FDBG(				printf("Finalise object 0x%x(%d) %s\n", obj,
				ref->idx, obj->mtable->class->name);		)
				CALL_KAFFE_FUNCTION_VARARGS(func, obj, 0, 0);
				/* Don't free it cause it might live */
				ref->flags = GC_UNMARK2;
				continue;
			}
			/* Special handling for threads */
			if (obj->mtable->class == ThreadClass) {
				thread* stiff = (thread*)obj;
				free(stiff->PrivateInfo->stackBase);
				free(stiff->PrivateInfo);
TDBG(				printf("Freeing thread 0x%x(%d)\n",
					obj, ref->idx);			)

			}
			/* Don't handle classes yet */
			else if (obj->mtable->class == ClassClass) {
				abort();
			}

			/* Put entry onto freelist */
			ref->flags = GC_FREE;
			ref->next = freeRef;
			freeRef = ref;

			/* Free the object */
FDBG(			printf("Freeing object 0x%x(%d) %s\n", obj,
				ref->idx, obj->mtable->class->name);	)
			free(obj);
		}
		waitCond(&currentThread->obj, waitforever);
	}
}

/*
 * Garbage collector.
 *  Run the garbage collector.
 */
void
gcMan(void)
{
	/* All threads start with interrupts disabled */
	intsRestore();

	lockMutex(&currentThread->obj);
	for (;;) {
                waitCond(&currentThread->obj, waitforever);

		/* Run the garbage collector */
		garbageCollect();

		/* If there's garbage, finalise it */
		if (garbageObjects != 0 && finalman != 0) {
			lockMutex(&finalman->obj);
			signalCond(&finalman->obj);
			unlockMutex(&finalman->obj);
		}
	}
}
