/*
 * Be implementation of Java threads
 */

#include <KernelKit.h>

#include "sys_api.h"
#include "monitor.h"
#include "monitor_md.h"
#include "monitor_cache.h"
#include "threads.h"
#include "threads_md.h"

#include "java_lang_Thread.h"

/*
 * Set to TRUE once threads have been bootstrapped
 */
bool_t ThreadsInitialized = FALSE;

/*
 * Queue of active Java threads
 */
static sys_thread_t *ThreadQueue = 0;

/*
 * Add thread to queue of active threads.
 */
static void
queueInsert(sys_thread_t *tid)
{
    QUEUE_LOCK();
    ActiveThreadCount++;
    tid->next = ThreadQueue;
    ThreadQueue = tid;
    if (!tid->system_thread) {
	UserThreadCount++;
    }
    QUEUE_UNLOCK();
}

/*
 * Remove thread from queue of active threads.
 */
static void
queueRemove(sys_thread_t *tid)
{
    QUEUE_LOCK();
    --ActiveThreadCount;
    if (!tid->system_thread) {
	--UserThreadCount;

	QUEUE_NOTIFY();
    }
    if (ThreadQueue == tid) {
		ThreadQueue = tid->next;
    } else {
		sys_thread_t *p;
		for (p = ThreadQueue; p->next != 0; p = p->next) {
		    if (p->next == tid) {
			p->next = tid->next;
			break;
		    }
		}
    }
	sysMonitorExitNoLookup(_queue_lock, tid);
}

/* 
 * Bootstrap the Java thread system by making the current thread the
 * "primordial" thread.
 */
int
sysThreadBootstrap(sys_thread_t **ptid)
{
    sys_thread_t *tid;
	thread_id beos_id = find_thread(NULL);
	thread_info beos_tinfo;
	
	if (get_thread_info(beos_id, &beos_tinfo) != B_NO_ERROR) {
		fprintf(stderr, "sysThreadBootstrap: get_thread_info failed\n");
		return SYS_INVALID;
	}

    /*
     * Allocate and initialize "thread private" structure.
     */
    if ((tid = malloc(sizeof(sys_thread_t))) == 0) {
        out_of_memory();
    }
    memset(tid, 0, sizeof(sys_thread_t));
    tid->stack_base = beos_tinfo.stack_base;
    tid->state = RUNNABLE;
    tid->id = beos_id;

    /* 
     * Can't call
	 * queueInsert(tid);
	 * yet because ThreadsInitialized is not yet TRUE. Paul.
	 */

    ActiveThreadCount++;
    tid->next = ThreadQueue;
    ThreadQueue = tid;
    if (!tid->system_thread) UserThreadCount++;

    ThreadsInitialized = TRUE;
    *ptid = tid;
    return SYS_OK;
}

/*
 * Return current stack pointer of specified thread.
 */
void *
sysThreadStackPointer(sys_thread_t *tid)
{
	thread_info beos_tinfo;
	
	if (get_thread_info(tid->id, &beos_tinfo) != B_NO_ERROR) {
		fprintf(stderr, "sysThreadStackPointer: get_thread_info failed\n");
		return (void *) 0;
	}

	return beos_tinfo.stack_end;
}

/*
 * Return stack base of specified thread.
 */
stackp_t
sysThreadStackBase(sys_thread_t *tid)
{
    return (stackp_t) tid->stack_base;
}

/*
 * Thread start routine for new Java threads
 */

long _start(void *p)
{
	sys_thread_t *tid = (sys_thread_t *) p;
	thread_info beos_tinfo;
	
	if (get_thread_info(tid->id, &beos_tinfo) != B_NO_ERROR) {
		fprintf(stderr, "_start[thread]: get_thread_info failed\n");
		return;
	}
	
    tid->state = RUNNABLE;
    tid->stack_base = beos_tinfo.stack_base;
    tid->start_proc(tid->start_parm);
    return 0;
}

/* 
 * Create a new Java thread. The thread is initially suspended.
 */
int
sysThreadCreate(long stack_size, unsigned long flags, void *(*proc)(void *),
		sys_thread_t **tidp, void *parm)
{
    sys_thread_t *tid;

    /* 
     * Allocate and initialize thread structure.
     */
    if ((tid = malloc(sizeof(sys_thread_t))) == 0) {
		out_of_memory();
    }
    memset(tid, 0, sizeof(sys_thread_t));
    tid->state = SUSPENDED;
    tid->start_proc = proc;
    tid->start_parm = parm;
	tid->priority = 5;
    if (flags == THR_SYSTEM) {
		tid->system_thread = TRUE;
    }

    /*
     * Start the new thread.
     */
    tid->id = spawn_thread(_start, "JavaThread", B_NORMAL_PRIORITY, tid);

    if (tid->id < B_NO_ERROR) {
		free(tid);
		fprintf(stderr, "sysThreadCreate: spawn_thread failed\n");
		return SYS_NORESOURCE;
    }
    queueInsert(tid);

    *tidp = tid;
    return SYS_OK;
}

/*
 * First thing called by a newly created Java thread.
 */
void
sysThreadInit(sys_thread_t *tid, stackp_t stack)
{

}


/* 
 * Destroy calling thread (function never returns!).
 */
void
sysThreadExit(void)
{
    sys_thread_t *tid = sysThreadSelf();
    monitor_t *mon;
    TID Hthread ;
    ExecEnv *ee;

	REGISTRY_LOCK();

    /*
     * Put the Java stack segments on the to-be-freed list.
     */
    Hthread = (TID)tid->cookie;
    ee = (ExecEnv *)THREAD(Hthread)->eetop;
    if (ee != 0) {  /* ee == 0 does happen... */
        DeleteExecEnv(ee, (JHandle *)Hthread);
    }


    /*
     * Clean up all monitors that we currently own.  monitorCleanup()
     * exits _moncache_lock but not _registry_lock
     */
    monitorCleanup(Hthread);
	REGISTRY_UNLOCK();

    sysAssert(!REGISTRY_LOCKED());


    /*
     * Notify anybody who is waiting for us to die.
     */
    THREAD(Hthread)->PrivateInfo = 0;
    mon = (monitor_t *) lookupMonitor(obj_monitor(tid->cookie));

    if (mon != 0) {
        sys_mon_t *mid = &sysmon(mon);
		sysMonitorEnter(mid);
		sysMonitorNotifyAll(mid);
		sysMonitorExit(mid);
    }

    queueRemove(tid);

    free(tid);

}


/* 
 * Yield control to another thread.
 */
void
sysThreadYield(void)
{
	snooze(100.0);
}

/* 
 * Suspend execution of the specified thread.
 */
int
sysThreadSuspend(sys_thread_t *tid)
{

	long error;
	
    if ((error = suspend_thread(tid->id)) != B_NO_ERROR) {
		if (error != B_BAD_THREAD_STATE) {
			fprintf(stderr, "sysThreadSuspend: suspend_thread failed: ");
			if (error == B_BAD_THREAD_ID) fprintf(stderr, "B_BAD_THREAD_ID\n");
			else fprintf(stderr, "unknown error\n");
			return SYS_ERR;
		}
    }
    tid->state = SUSPENDED;
    return SYS_OK;
}

/* 
 * Continue execution of the specified thread.
 */
int
sysThreadResume(sys_thread_t *tid)
{

	long error;
	
    if ((error = resume_thread(tid->id)) != B_NO_ERROR) {
		if (error != B_BAD_THREAD_STATE) {
			fprintf(stderr, "sysThreadResume: resume_thread failed: ");
			if (error == B_BAD_THREAD_ID) fprintf(stderr, "B_BAD_THREAD_ID\n");
			else fprintf(stderr, "unknown error\n");
			return SYS_ERR;
		}
    }
    tid->state = RUNNABLE;
    return SYS_OK;
}

/* 
 * Return priority of specified thread.
 */
int
sysThreadGetPriority(sys_thread_t *tid, int *pp)
{

/* Better to just let Java think it can set thread priorities :-) Paul.
 */

	*pp = tid->priority;
    return SYS_OK;
}

/*
 * Set priority of specified thread.
 */
int
sysThreadSetPriority(sys_thread_t *tid, int p)
{
	tid->priority = p;
    return SYS_OK;
}

/* 
 * Return the thread information block of the calling thread.
 */
sys_thread_t *
sysThreadSelf(void)
{
	sys_thread_t *p;
	thread_id beos_id = find_thread(NULL);

	if (ThreadQueue == 0) return 0;
	for (p = ThreadQueue; p != 0; p = p->next) {
	    if (p->id == beos_id) return p;
	}
	fprintf(stderr, "sysThreadSelf: failed to find self :-(\n");
	return 0;

}

/*
 * Enumerate over all threads in active queue calling a function for
 * each one.
 */
int
sysThreadEnumerateOver(int (*func)(sys_thread_t *, void *), void *arg)
{
    sys_thread_t *tid;
    int ret = SYS_OK;

    QUEUE_LOCK();
    for (tid = ThreadQueue; tid != 0; tid = tid->next) {
	if ((ret = (*func)(tid, arg)) != SYS_OK) {
	    break;
	}
    }
    QUEUE_UNLOCK();
    return ret;
}

/*
 * Helper function for sysThreadSingle()
 */
static int
threadSingleHelper(sys_thread_t *tid, void *self)
{
     return tid == self ? SYS_OK : sysThreadSuspend(tid);
}

/*
 * Puts each thread in the active thread queue to sleep except for the
 * calling thread. The threads must be later woken up with a corresponding
 * call to 'sysThreadMulti'. Returns SYS_OK on success, or SYS_ERR if any
 * of the threads could not be suspended.
 */
int
sysThreadSingle(void)
{
    return sysThreadEnumerateOver(threadSingleHelper, sysThreadSelf());
}

/*
 * Helper function for sysThreadMulti()
 */
static int
threadMultiHelper(sys_thread_t *tid, void *self)
{
    return tid == self ? SYS_OK : sysThreadResume(tid);
}

/*
 * Wakes up each thread in active thread queue except for the calling
 * thread. Returns SYS_ERR if not all threads could be woken up.
 */
void
sysThreadMulti(void)
{
    sysThreadEnumerateOver(threadMultiHelper, sysThreadSelf());
}

/*
 * Dump system-specific information about threads.
 */
void
sysThreadDumpInfo(sys_thread_t *tid)
{
    char *state;

    switch(tid->state) {
	case RUNNABLE: 
	    state = "RUNNABLE";
	    break;
	case SUSPENDED:
	    state = "SUSPENDED";
	    break;
	case SLEEPING:
	    state = "SLEEPING";
	    break;
	case MONITOR_WAIT:
	    state = "MONITOR_WAIT";
	    break;
	case CONDVAR_WAIT:
	    state = "CONDVAR_WAIT";
	    break;
	default:
	    state = "UNKNOWN";
    }
    fprintf(stderr, ", BeOS id:%d, state:%s", tid->id, state);

}

int
sysInterruptsPending(void)
{
    return FALSE;
}

void
sysThreadSetBackPtr(sys_thread_t *t, void *p)
{
    t->cookie = p;
}

void *
sysThreadGetBackPtr(sys_thread_t *t)
{
    return t->cookie;
}

void
sysThreadInitializeSystemThreads(void)
{
    extern void InitializeFinalizerThread(void);

    /* Create the finalization thread */
    InitializeFinalizerThread();
}

int
sysThreadCheckStack(void)
{
    return 1;
}

/*
 * The mechanics of actually signalling an exception (in the future,
 * and Alarm or Interrupt) depend upon what thread implementation you
 * are using.
 */
void
sysThreadPostException(sys_thread_t *tid, void *exc)
{
    struct execenv *ee;

    ee = (ExecEnv *) THREAD((TID)tid->cookie)->eetop;
    if (ee != 0) {
        exceptionThrow(ee, exc);
    }
}


