/*
 * Be implementation of Java monitors
 */

#include <KernelKit.h>

#include "sys_api.h"
#include "threads.h"
#include "threads_md.h"
#include "monitor.h"
#include "monitor_md.h"

/* If you don't define FAKE_MUTEX you'll use two semaphores per monitor
   which means that you will run out of sems on BeOS when compiling
   decent sized Java programs. Defining FAKE_MUTEX may make your code
   a little slower but you won't run out of sems. Paul.
*/
 
#define FAKE_MUTEX

sem_id	MutexAcquireLock,
		CondVarCreateLock;

void sysInitializeMonitors() {

	if ((MutexAcquireLock = create_sem(1, "JavaMutexAcquireLock")) < B_NO_ERROR) {
		fprintf(stderr, "sysInitializeMonitors: create_sem/MutexAcquireLock failed, serious error\n");
    }
	if ((CondVarCreateLock = create_sem(1, "JavaCondVarCreateLock")) < B_NO_ERROR) {
		fprintf(stderr, "sysInitializeMonitors: create_sem/CondVarCreateLock failed, serious error\n");
    }
	
}


void sysReleaseMonitors() {

	if (delete_sem(MutexAcquireLock) < B_NO_ERROR) {
		fprintf(stderr, "initializeMonitors: delete_sem/MutexAcquireLock failed, serious error\n");
    }
	if (delete_sem(CondVarCreateLock) < B_NO_ERROR) {
		fprintf(stderr, "initializeMonitors: delete_sem/CondVarCreateLock failed, serious error\n");
    }
	
}


#ifdef FAKE_MUTEX
int mutexLock(sys_mon_t *mid) {

	double timeout=1000.0;
	long error;
retry1:
    if ((error = acquire_sem(MutexAcquireLock)) < B_NO_ERROR) {
		fprintf(stderr, "mutexLock: acquire_sem(MutexAcquireLock)/1 failed: ");
		if (error == B_BAD_SEM_ID) fprintf(stderr, "B_BAD_SEM_ID\n");
		else if (error == B_INTERRUPTED) /* fprintf(stderr, "B_INTERRUPTED\n"); */ goto retry1;
		else fprintf(stderr, "Unknown\n");
		fflush(stderr);
		return SYS_ERR;
    }

	while (mid->mutex == 0) {
	
	    if (release_sem(MutexAcquireLock) < B_NO_ERROR) {
			fprintf(stderr, "mutexLock: release_sem(MutexAcquireLock)/1 failed\n");
			return SYS_ERR;
	    }

		snooze(timeout);
				
		if (timeout >= 100000.0) timeout = 100000.0;
		else timeout *=2;
retry2:
	    if ((error = acquire_sem(MutexAcquireLock)) < B_NO_ERROR) {
			fprintf(stderr, "mutexLock: acquire_sem(MutexAcquireLock)/2 failed: ");
			if (error == B_BAD_SEM_ID) fprintf(stderr, "B_BAD_SEM_ID\n");
			else if (error == B_INTERRUPTED) /* fprintf(stderr, "B_INTERRUPTED\n");*/ goto retry2;
			else fprintf(stderr, "Unknown\n");
			fflush(stderr);
			return SYS_ERR;
	    }

	}
	
	mid->mutex = 0;
	
    if (release_sem(MutexAcquireLock) < B_NO_ERROR) {
		fprintf(stderr, "mutexLock: release_sem(MutexAcquireLock)/2 failed\n");
		return SYS_ERR;
    }

	return SYS_OK;
	
}
#else
int mutexLock(sys_mon_t *mid) {

	long error;
retry:
    if ((error = acquire_sem(mid->mutex)) < B_NO_ERROR) {
		if (error == B_INTERRUPTED) goto retry;
		fprintf(stderr, "mutexLock: acquire_sem failed: ");
		if (error == B_BAD_SEM_ID) fprintf(stderr, "B_BAD_SEM_ID\n");
		else fprintf(stderr, "Unknown\n");
		fflush(stderr);
		return SYS_ERR;
    }
	return SYS_OK;

}
#endif

#ifdef FAKE_MUTEX
int mutexUnlock(sys_mon_t *mid) {

	long thread_cnt, error;
retry3:
    if ((error = acquire_sem(MutexAcquireLock)) != B_NO_ERROR) {
		fprintf(stderr, "mutexUnlock: acquire_sem(MutexAcquireLock) failed: ");
		if (error == B_BAD_SEM_ID) fprintf(stderr, "B_BAD_SEM_ID\n");
		else if (error == B_INTERRUPTED) /* fprintf(stderr, "B_INTERRUPTED\n"); */ goto retry3;
		else fprintf(stderr, "Unknown\n");
		fflush(stderr);
		return SYS_ERR;
    }

	mid->mutex = 1;
	
    if (release_sem(MutexAcquireLock) != B_NO_ERROR) {
		fprintf(stderr, "mutexUnlock: release_sem(MutexAcquireLock) failed\n");
		return SYS_ERR;
    }

	return SYS_OK;
	
}
#else
int mutexUnlock(sys_mon_t *mid) {

	if (release_sem(mid->mutex) != B_NO_ERROR) {
		fprintf(stderr, "REAL mutexUnlock: release_sem failed\n");
		return SYS_ERR;
    }

	return SYS_OK;
	
}
#endif

#ifdef FAKE_MUTEX
int mutexUnlockNoReschedule(sys_mon_t *mid) {

	long thread_cnt, error;
retry3:
    if ((error = acquire_sem(MutexAcquireLock)) != B_NO_ERROR) {
		fprintf(stderr, "mutexUnlock: acquire_sem(MutexAcquireLock) failed: ");
		if (error == B_BAD_SEM_ID) fprintf(stderr, "B_BAD_SEM_ID\n");
		else if (error == B_INTERRUPTED) /* fprintf(stderr, "B_INTERRUPTED\n"); */ goto retry3;
		else fprintf(stderr, "Unknown\n");
		fflush(stderr);
		return SYS_ERR;
    }

	mid->mutex = 1;
	
    if (release_sem_etc(MutexAcquireLock, 1, B_DO_NOT_RESCHEDULE) != B_NO_ERROR) {
		fprintf(stderr, "mutexUnlock: release_sem(MutexAcquireLock) failed\n");
		return SYS_ERR;
    }

	return SYS_OK;
	
}
#else
int mutexUnlockNoReschedule(sys_mon_t *mid) {

	if (release_sem_etc(mid->mutex, 1, B_DO_NOT_RESCHEDULE) != B_NO_ERROR) {
		fprintf(stderr, "REAL mutexUnlock: release_sem failed\n");
		return SYS_ERR;
    }

	return SYS_OK;
	
}
#endif

int
monitorRecreate(sys_mon_t *mid) {

	long error;

retry4:
	if ((error = acquire_sem(CondVarCreateLock)) != B_NO_ERROR) {
		fprintf(stderr, "monitorRecreate: acquire_sem(CondVarCreateLock) failed: ");
		if (error == B_BAD_SEM_ID) fprintf(stderr, "B_BAD_SEM_ID\n");
		else if (error == B_INTERRUPTED) /* fprintf(stderr, "B_INTERRUPTED\n"); */ goto retry4;
		else fprintf(stderr, "Unknown\n");
		fflush(stderr);
		return SYS_ERR;
    }

    if (mid->condvar == -1) {
		if ((mid->condvar = create_sem(1, "JavaCondVar")) < B_NO_ERROR) {
			fprintf(stderr, "monitorRecreate: create_sem/condvar failed\n");
			return SYS_ERR;
    	}
	} 

    if (release_sem(CondVarCreateLock) != B_NO_ERROR) {
		fprintf(stderr, "monitorRecreate: release_sem(CondVarCreateLock) failed\n");
		return SYS_ERR;
    }

}

/*
 * Create and initialize monitor. This can be called before threads have
 * been initialized.
 */
int
sysMonitorInit(sys_mon_t *mid, bool_t in_cache)
{

    mid->owner = 0;

	/*
	 * mutex is set to 1 so the next mutexLock will acquire it
	 * condvar is set to -1 and will only be created as a semaphore
	 * if needed. Paul.
	 */

#ifdef FAKE_MUTEX	
	mid->mutex = 1;
#else
	if ((mid->mutex = create_sem(1, "JavaMutex")) < B_NO_ERROR) {
		fprintf(stderr, "sysMonitorInit: create_sem/mutex failed\n");
		return SYS_ERR;
	}
#endif

#ifdef FAKE_MUTEX
	mid->condvar = -1;
#else
	if ((mid->condvar = create_sem(1, "JavaCondVar")) < B_NO_ERROR) {
		fprintf(stderr, "sysMonitorInit: create_sem/condvar failed\n");
		return SYS_ERR;
    }
#endif

    return SYS_OK;

}

/*
 * Take ownership of monitor. This can be called before threads have
 * been initialized, in which case we do nothing since locked monitors
 * are not yet needed.
 */
int
sysMonitorEnter(sys_mon_t *mid)
{
    sys_thread_t *self = sysThreadSelf();
	sem_info sinfo;

    if (self == 0) {
		return ThreadsInitialized ? SYS_ERR : SYS_OK;
    }
    sysAssert(ThreadsInitialized);

	if (mid->owner != self) {
	
	    self->state = MONITOR_WAIT;
		
	    if (mutexLock(mid) != SYS_OK) {
			fprintf(stderr, "sysMonitorEnter: mutexLock failed\n");
			return SYS_ERR;
	    }
	    self->state = RUNNABLE;

	}
		
	/* It seems that sysMonitorEnter could be called more than
	 * once after a thread has acquired it, i.e. we could block
	 * ourselves. So first we check to see if we own the monitor
	 * already and if not, then we try to acquire it. Paul.
	 */
	if (mid->owner == 0) {
		mid->owner = self;
	} else {
		sysAssert(mid->owner == self);
		sysAssert(mid->depth > 0);
	}	
	mid->depth++;
	
    return SYS_OK;
}

/*
 * Return TRUE if this thread currently owns the monitor. This can be
 * called before threads have been initialized, in which case we always
 * return TRUE.
 */
bool_t
sysMonitorEntered(sys_mon_t *mid)
{

    sys_thread_t *self = sysThreadSelf();
    return !ThreadsInitialized || self != 0 && mid->owner == self;
}

/*
 * Release ownership of monitor. This can be called before threads have
 * been initialized, in which case we do nothing as locked monitors are
 * not yet needed.
 */
int
sysMonitorExit(sys_mon_t *mid)
{

	return sysMonitorExitNoLookup(mid, -1);

}

/*
 * This is called by queueRemove in threads_md.c because once
 * queueRemove has removed a thread from the queue sysMonitorExit's
 * sysThreadSelf won't work anymore. Kinda annoying. Paul
 */


int
sysMonitorExitNoLookup(sys_mon_t *mid, sys_thread_t *self)
{
	long thread_cnt_condvar = -1, thread_cnt_mutex = -1;

	if (self == (sys_thread_t *) -1) {
	    self = sysThreadSelf();
	}

    if (self == 0) {
		return ThreadsInitialized ? SYS_ERR : SYS_OK;
    }
    sysAssert(ThreadsInitialized);
    if (mid->owner != self) {
		puts("\tmonitor_md.c: sysMonitorExitNoLookup() mid->owner != self");
        return SYS_ERR;
    }
	
	sysAssert(mid->depth > 0);
	if (--mid->depth == 0) {
	    mid->owner = 0;
	    if (mutexUnlock(mid) != SYS_OK) {
			fprintf(stderr, "sysMonitorExitNoLookup: mutexUnlock failed\n");
			return SYS_ERR;
	    }
	}

    return SYS_OK;
}


/*
 * Notify single thread waiting on condition variable.
 */
int
sysMonitorNotify(sys_mon_t *mid)
{
	long thread_cnt;

    if (!sysMonitorEntered(mid)) {
		fprintf(stderr, "sysMonitorNotify: sysMonitorEntered failed\n");
		return SYS_ERR;
    }
	
	if (mid->condvar == -1) monitorRecreate(mid);
	
	if (get_sem_count(mid->condvar, &thread_cnt) != B_NO_ERROR) {
		fprintf(stderr, "sysMonitorNotify: get_sem_count failed\n");
		return SYS_ERR;
    }

	if (thread_cnt < 0) {
	    if (release_sem(mid->condvar) != B_NO_ERROR) {
			fprintf(stderr, "sysMonitorNotify: release_sem failed\n");
			return SYS_ERR;
	    }
	}

    return SYS_OK;
}


/*
 * Notify all threads waiting on condition variable.
 */
int
sysMonitorNotifyAll(sys_mon_t *mid)
{
	long thread_cnt;
	sem_id cond = mid->condvar;

    if (!sysMonitorEntered(mid)) {
		fprintf(stderr, "sysMonitorNotifyAll: sysMonitorEntered failed\n");
		return SYS_ERR;
    }
	
	if (mid->condvar == -1) monitorRecreate(mid);
	
	if (get_sem_count(mid->condvar, &thread_cnt) != B_NO_ERROR) {
		fprintf(stderr, "sysMonitorNotifyAll: get_sem_count failed\n");
		return SYS_ERR;
    }

	if (thread_cnt < 0) {
	    if (release_sem_etc(mid->condvar, -thread_cnt, 0) != B_NO_ERROR) {
			fprintf(stderr, "sysMonitorNotifyAll: release_sem_count failed\n");
			return SYS_ERR;
	    }
	}

    return SYS_OK;
}

/*
 * Atomically drop mutex and wait for notification.
 */
int
sysMonitorWait(sys_mon_t *mid, int millis)
{
    sys_thread_t *self = sysThreadSelf();
	int depth = mid->depth;

    if (self == 0) {
        return SYS_ERR;
    }
    sysAssert(ThreadsInitialized);
    if (mid->owner != self) {
        return SYS_ERR;
    }
    mid->owner = 0;
	mid->depth = 0;
    self->state = CONDVAR_WAIT;

	/*
	 * Since there are no condition variables in BeOS
	 * We simulate them by releasing the mutex,
	 * acquiring condvar and re-acquiring mutex.
	 * Not pretty. Paul.
	 */
	
	if (mid->condvar == -1) monitorRecreate(mid);
	
    if (mutexUnlockNoReschedule(mid) != SYS_OK) {
		fprintf(stderr, "sysMonitorWait: mutexUnlock failed\n");
		return SYS_ERR;
    }
    if (millis == SYS_TIMEOUT_INFINITY) {

retry5:
		if (acquire_sem(mid->condvar) == B_INTERRUPTED) goto retry5;

    } else {

retry6:
        if (acquire_sem_etc(mid->condvar, 1, B_TIMEOUT, ((double) millis) * 1000.0)
			== B_INTERRUPTED) goto retry6;

    }

    if (mutexLock(mid) != SYS_OK) {
		fprintf(stderr, "sysMonitorWait: mutexLock failed\n");
		return SYS_ERR;
    }
    self->state = RUNNABLE;
	mid->depth = depth;
    mid->owner = self;


    return SYS_OK;
}

/*
 * If monitor owned by thread then release monitor and notify all other
 * waiting threads. If no other threads are waiting for the monitor then
 * return SYS_DESTROY so that the monitor can be freed up from the cache.
 */

int
sysMonitorDestroy(sys_mon_t *mid, sys_thread_t *tid)
{

	long thread_cnt = 0;

    if (mid->owner == tid) {

		mid->depth = 0;
		mid->owner = 0;
		
		if (mid->condvar != -1) {
			if (get_sem_count(mid->condvar, &thread_cnt) != B_NO_ERROR) {
				fprintf(stderr, "sysMonitorDestroy: get_sem_count failed, condvar = %d\n", mid->condvar);
				return SYS_ERR;
			}
	    }
		if (thread_cnt < 0) {
		    if (release_sem_etc(mid->condvar, -thread_cnt, 0) != B_NO_ERROR) {
				fprintf(stderr, "sysMonitorDestroy: release_sem_count failed\n");
				return SYS_ERR;
		    }
		    if (mutexUnlock(mid) != SYS_OK) {
				fprintf(stderr, "sysMonitorDestroy: mutexUnlock/1 failed\n");
				return SYS_ERR;
		    }
		    return SYS_OK;
		}

	    if (mutexUnlock(mid) != SYS_OK) {
			fprintf(stderr, "sysMonitorDestroy: mutexUnlock/2 failed\n");
			return SYS_ERR;
	    }

	    return SYS_DESTROY;
    }

    return SYS_OK;
}

/*
 * Print monitor information.
 */
void
sysMonitorDumpInfo(sys_mon_t *mid)
{

    if (mid->owner != 0) {
        TID tid = mid->owner->cookie;
	fprintf(stderr, "%s\n",
	        Object2CString((JHandle *)THREAD(tid)->name));
    } else {
	fprintf(stderr, "<unowned>\n");
    }

}

/*
 * Return size of system-dependent monitor structure.
 */
int
sysMonitorSizeof(void)
{
    return sizeof(struct sys_mon);
}

