
/*
 * Implementation of primitive memory allocation.
 *
 * The only thing machine dependent about this allocator is how it 
 * initially finds all of the possible memory, and how it implements
 * mapChunk() and unmapChunk().
 *
 * This is all pretty simple stuff.  It is not likely to be banged on
 * frequently enough to be a performance issue, unless the underlying
 * primitives are.  Implementing things:
 *
 * HPI function      Solaris   "malloc"    Win32 
 * --------------------------------------------------------------------
 * sysMapMem()	     mmap()     malloc()   VirtualAlloc(...MEM_RESERVE...)
 * sysUnMapMem()     munmap()   free()     VirtualFree(...MEM_RESERVE...)
 * sysCommitMem()    no-op      no-op      VirtualAlloc(...MEM_COMMIT...)
 * sysDecommitMem()  no-op      no-op      VirtualFree(...MEM_COMMIT...)
 *
 * Memory mapping is the default, but compiling with -DUSE_MALLOC gives
 * a system based on malloc().
 */

#include <sys/types.h>
#include <KernelKit.h>

#include "log.h"
#include "sys_api.h"

static long
roundUp(long n, long m)
{
    return (n + m - 1) & ~(m - 1);
}

static long
roundDown(long n, long m)
{
    return n & ~(m - 1);
}


static long pageSize;		/* Machine page size */

void
InitializeMem(void)
{
    pageSize = B_PAGE_SIZE;
}

/* HPI Functions: */

/*
 * Map a range of virtual memory.  Note that the size asked for here
 * is literally what the upper level has asked for.  We need to do
 * any rounding, etc. here.  If mapping fails return 0, otherwise
 * return the address of the base of the mapped memory.
 */
void *
sysMapMem(long requestedSize, long *mappedSize)
{
    void *mappedAddr;

    *mappedSize = roundUp(requestedSize, pageSize);
    mappedAddr = (void *)malloc(*mappedSize);
    if (mappedAddr != NULL) {
	Log3(2, "sysMapMem: 0x%x bytes at 0x%x (request: 0x%x bytes)\n",
	     *mappedSize, mappedAddr, requestedSize);
    } else {
	Log1(2, "sysMapMem failed: (request: 0x%x bytes)\n", requestedSize);
    } 
    return mappedAddr;
}

/*
 * Unmap a range of virtual memory.  Note that the size asked for here
 * is literally what the upper level has asked for.  We need to do any
 * rounding, etc. here.  If unmapping fails return 0, otherwise return
 * the address of the base of the unmapped memory.
 */
void *
sysUnmapMem(void *requestedAddr, long requestedSize, long *unmappedSize)
{
    void *unmappedAddr;
    int ret;

    *unmappedSize = roundUp(requestedSize, pageSize);
    free(requestedAddr);
    ret = TRUE;
    if (ret) {
	unmappedAddr = requestedAddr;
        Log4(2,
	     "sysUnmapMem: 0x%x bytes at 0x%x (request: 0x%x bytes at 0x%x)\n",
	     *unmappedSize, unmappedAddr, requestedSize, requestedAddr);
    } else {
	unmappedAddr = NULL;
	Log2(2, "sysUnmapMem failed: (request: 0x%x bytes at 0x%x)\n",
	     requestedSize, requestedAddr);
    }
    return unmappedAddr;
}

/*
 * Commit/decommit backing store to a range of virtual memory.  This range
 * needs not be identical to a mapped range, but must be a subset of one.  
 * On Solaris, we remap the range to reserve swap for the space on
 * commit.  We don't strictly need to do this, as Solaris will demand
 * page pages that we've mapped when we want to access them.  But by
 * reserving swap we get reasonable error handling for free where we'd
 * otherwise end up getting a SIGBUS on a random write when we run out
 * of swap.  It also emphasizes the general need for shared code to
 * postpone committing to mapped memory for as long as is feasible.
 * When Java really needs space (the thread stacks excepted), it will
 * soon write over it (heap, markbits), so we don't really get much from
 * demand paging. 
 *
 * We do not validate that commitment requests cover already-mapped
 * memory, although in principle we could.  The size asked for here
 * is what the upper level has asked for.  We need to do any platform-
 * dependent rounding here.
 *
 * When you commit, you commit to the entire page (or whatever quantum
 * your O/S requires) containing the pointer, and return the beginning of
 * that page.  When you decommit, you decommit starting at the next page
 * *up* from that containing the pointer, except that decommitting from
 * a pointer to the beginning of the page operates on that page.
 */

/*
 * Return the address of the base of the newly committed memory, or 0
 * if committing failed.
 */
void *
sysCommitMem(void *requestedAddr, long requestedSize, long *committedSize)
{
    void *committedAddr;

    *committedSize = roundUp(requestedSize, pageSize);
    committedAddr = (void *) roundDown((long) requestedAddr, pageSize);
	Log4(2,
	     "sysCommitMem: 0x%x bytes at 0x%x (request: 0x%x bytes at 0x%x)\n",
	     *committedSize, committedAddr, requestedSize, requestedAddr);
    return committedAddr;
} 

/*
 * Return the address of the base of the newly decommitted memory, or 0
 * if decommitting failed.
 */
void *
sysDecommitMem(void *requestedAddr, long requestedSize, long *decommittedSize)
{
    void *decommittedAddr;

    *decommittedSize = roundUp(requestedSize, pageSize);
    decommittedAddr = (void *)roundUp((long)requestedAddr, pageSize);
	Log4(2,
	     "sysDecommitMem: 0x%x bytes at 0x%x (request: 0x%x bytes at 0x%x)\n",
	     *decommittedSize, decommittedAddr, requestedSize, requestedAddr);
    return decommittedAddr;
}
