#include <KernelKit.h>
#include <stdio.h>
#include "typedefs_md.h"

#define MIDFIELD	32
#define MAXLONG		4294967296.0
#define MAXSHORT	65536

#define	abs(x)		(x) < 0 ? -(x) : (x)


int64_t	ll_zero_const = { 0,0 },
		ll_one_const = { 0,1 },
		ll_minus_one_const = { 0xffffffff, 0xffffffff },
		ll_max_long = { 0x7fffffff, 0xffffffff },
		ll_min_long = { 0x80000000, 0x00000000 };



int64_t int2ll(int a) {

	int64_t b = { 0,0 };

	b.sl = a;
	if (a < 0) b.sh = 0xffffffff;
	return b;

}


int ll2int(int64_t a) {

	return a.sl;

}


asm int64_t ll_add(int64_t a, int64_t b) {

	lwz		r4, a.sl
	lwz		r5, b.sl
	addc.	r4, r4, r5
	stw		r4, 4(r3)
	lwz		r4, a.sh
	lwz		r5, b.sh
	adde.	r4, r4, r5
	stw		r4, 0(r3)
	blr

}		


int64_t ll_and(int64_t a, int64_t b) {

	int64_t c;
	
	c.sh = a.sh & b.sh;
	c.sl = a.sl & b.sl;
	return c;
}


int64_t ll_not(int64_t a) {

	a.sh = ~a.sh;
	a.sl = ~a.sl;
	return a;

}


int64_t ll_neg(int64_t a) {

	return ll_add(ll_not(a), ll_one_const);

}


asm int64_t ll_mul(int64_t a, int64_t b) {

	lwz		r4, a.sl
	lwz		r5, b.sl
	lwz		r6, a.sh
	lwz		r7, b.sh
	mullw	r8, r4, r5
	stw		r8, 4(r3)
	mulhwu	r8, r4, r5
	mullw	r9, r6, r5
	add		r8, r8, r9
	mullw	r9, r4, r7
	add		r8, r8, r9
	stw		r8, 0(r3)
	blr

}


int64_t ll_or(int64_t a, int64_t b) {

	int64_t c;
	
	c.sh = a.sh | b.sh;
	c.sl = a.sl | b.sl;
	return c;
}


int64_t ll_shl(int64_t a, int n) {

	if (n >= MIDFIELD) {
		a.sh = a.sl << (n - MIDFIELD);
		a.sl = 0;
	} else {
		unsigned long maskl = ~(0xFFFFFFFF >> n), b;
		b = (a.sl & maskl) >> (MIDFIELD - n);
		a.sl = a.sl << n;
		a.sh = (a.sh << n) | b;
	}
	return a;
}

int64_t ll_shr(int64_t a, int n) {

	if (n >= MIDFIELD) {
		a.sl = (a.sh >> (n - MIDFIELD));
		a.sh = a.sh >> n;
	} else {
		unsigned long maskl = ~(0xFFFFFFFF << n), b;
		b = (a.sh & maskl) << (MIDFIELD - n);
		a.sh = (a.sh >> n);
		a.sl = (a.sl >> n) | b;
	}
	return a;
	
}


asm int64_t ll_sub(int64_t a, int64_t b) {

	lwz		r4, a.sl
	lwz		r5, b.sl
	subfc.	r4, r5, r4
	stw		r4, 4(r3)
	lwz		r4, a.sh
	lwz		r5, b.sh
	subfe.	r4, r5, r4
	stw		r4, 0(r3)
	blr

}		


int64_t ll_ushr(int64_t a, int n) {

	if (n >= MIDFIELD) {
		a.sl = (a.sh >> (n - MIDFIELD)) & (0xFFFFFFFF >> (n - MIDFIELD));
		a.sh = 0;
	} else {
		unsigned long maskl = ~(0xFFFFFFFF << n), b;
		b = (a.sh & maskl) << (MIDFIELD - n);
		a.sh = (a.sh >> n) & (0xFFFFFFFF >> n);
		a.sl = (a.sl >> n) & (0xFFFFFFFF >> n) | b;
	}
	return a;

}


int64_t ll_xor(int64_t a, int64_t b) {

	int64_t c;
	
	c.sh = a.sh ^ b.sh;
	c.sl = a.sl ^ b.sl;
	return c;
}


int64_t uint2ll(unsigned int a) {

	int64_t b = { 0,0 };
	
	b.sl = a;
	return b;

}


// This is based on the Shift and Subtract algorithm
// described in "Assembly Language and Systems Programming
// for M68000 Family, Second Ed." by William Ford & William Topp,
// D.C. Heath, 1989. (p.340)
// Paul.

void internal_ll_div(int64_t  a, int64_t  b,
					 int64_t *q, int64_t *r) {

	int64_t c = { 0,0 }, quo = { 0,0 }, div, oldb;
	int neg_div = 0, neg_rem = 0;

	*q = ll_zero_const;
	
	if (ll_eq(a, ll_min_long)) {
		if (ll_eq(b, ll_max_long)) {
			*q = ll_minus_one_const;
			*r = ll_minus_one_const;
			return;
		}
		if (ll_eq(b, ll_min_long)) {
			*q = ll_one_const;
			*r = ll_zero_const;
			return;
		}
	}
	
	if (ll_eq(b, ll_min_long)) {
		*r = a;
		return;
	}
		
	if (ll_ltz(a)) {
		a = ll_neg(a); neg_div++; neg_rem = 1;
	}
	if (ll_ltz(b)) {
		b = ll_neg(b); neg_div++;
	}
	
	div = b;
	
	while (ll_ge(a, b)) {
	
		while (ll_ge(a,b)) {
		
			oldb = b;
			if (ll_eqz(quo)) quo = ll_one_const;
			else quo = ll_shl(quo,1);
			if (b.sh & 0x40000000) goto hack;
			b = ll_shl(b,1);
			
		}
hack:
		a = ll_sub(a, oldb);
		*q = ll_add(quo, *q);
		quo = ll_zero_const;
		b = div;
		
	}		
		
	if (neg_div == 1) *q = ll_neg(*q);
	if (neg_rem) a = ll_neg(a);
	*r = a;
	
}

int64_t ll_div(int64_t a, int64_t b) {

	int64_t q = { 0,0 }, r = { 0,0 };
	
	if (ll_eqz(b)) return ll_zero_const;
	if (ll_eq(a, b)) return ll_one_const;
	if (a.sh == 0 && b.sh == 0) {
		q.sl = a.sl / b.sl;
		return q;
	}

	internal_ll_div(a, b, &q, &r);
	
	return q;

}

int64_t ll_rem(int64_t a, int64_t b) {

	int64_t q = { 0,0 }, r = { 0,0 };
	
	internal_ll_div(a, b, &q, &r);
	
	return r;

}


int64_t float2ll(float f) {

	int64_t a;
	int b = 0;

	if (f != f) return ll_zero_const;
	if (f < 0.0) { b = 1; f = -f; }
	a.sh = f / MAXLONG;
	a.sl = (unsigned long) (f - ((float)a.sh) * MAXLONG); 
	if (b) a = ll_neg(a);
	return a;

}


float ll2float(int64_t a) {

	float f;
	
	f = a.sh; f *= MAXLONG; f += (unsigned long) a.sl;
	return f;
	
}


int64_t double2ll(double f) {

	int64_t a;
	int b = 0;

	if (f != f) return ll_zero_const;
	if (f < 0.0) { b = 1; f = -f; }
	a.sh = f / MAXLONG;
	a.sl = (unsigned long) (f - ((double)a.sh) * MAXLONG); 
	if (b) a = ll_neg(a);
	return a;

}


float ll2double(int64_t a) {

	double f;
	int b = 0;

	if (ll_ltz(a)) { b = 1; a = ll_neg(a); }
	f = a.sh; f *= MAXLONG; f += (unsigned long) a.sl;
	if (b) f = -f;
	return f;
	
}


char * myll2str (int64_t n) {

	static char s[100];
	
	ll2str (n, s, (char *) (s+100));
	return s;
	
}

/*
 * Moved here as a portability concern - not availiable on all platforms
 *
 * The API to this is historical, and isn't exactly what we want, but
 * we can check for buffer overflow regardless.  The caller should be
 * able to guarantee against overflow, but if so, truncate silently.
 */
//int jio_snprintf(char *str, size_t count, const char *fmt, ...);

void ll2str(int64_t number, char *s, char *limit)
{
#define MAGIC    10000000	/* slightly more than (2^64) ^ (1/3) */
    int64_t group1, group2, group3, magic, a, b;
	int		g1, g2, g3;
    char	*sign;
    int 	len = limit - s;

    if (ll_gez(number)) {
		sign = "";
        number = ll_neg(number);	/* this lets -2^63 still work! */
    } else {
		sign = "-";
    }

	magic = int2ll(MAGIC);
    group3 = ll_neg(ll_rem(number, magic));	/* low 7 digits */
    number = ll_div(ll_add(number , group3), magic);
    group2 = ll_neg(ll_rem(number, magic));	/* next 7 digits */
    number = ll_div(ll_add(number , group2), magic);
    group1 = ll_neg(number);		/* high 7 digits */

	g1 = ll2int(group1); g2 = ll2int(group2); g3 = ll2int(group3);
	
    if (g1 != 0) {
        sprintf(s, "%s%d%07.7d%07.7d",
			    sign, g1, g2, g3);
    } else if (g2 != 0) {
        (void) sprintf(s, "%s%d%07.7d", sign, g2, g3);
    } else {
        (void) sprintf(s, "%s%d", sign, g3);      
    }
}

char * hex_ll2str(int64_t a) {
	static char buf[100];
	sprintf(buf, "%8x:%8x", a.sh, a.sl);
	return buf;
}

