/*
 * Copyright (c) 1993-1996 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. Please refer to the file "copyright.html"
 * for further important copyright and licensing information.
 *
 * The Java source code is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You shall
 * not disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.

 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
*/
/*
 * @(#)javah.c	1.59 96/03/26 James Gosling
 *
 */

/*-
 *      produce C header files from an java object file
 *      produce C stub files from an java object file
 */

#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>
#include "tree.h"
#include "oobj.h"
#include "interpreter.h"
#include "signature.h"
#else
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>
#include "tree.h"
#include "oobj.h"
#include "interpreter.h"
#include "signature.h"
#include "sys_api.h"
#endif

#ifndef RELEASE
#  define RELEASE "Unknown version"
#endif

#ifdef WIN32
#define EXPORT "__declspec(dllexport)"
#endif

static int renameIfDiff(char *tempfn, char *newfn, char *newdir);
void makeslottable(ClassClass *);

long stubmode = 0;
long stubtable[2];

static FILE *outfp = 0;
static char *out = 0;
static char *dir = ".";
static char *tempfn = 0;
static char *tempdir = "/tmp";
static char currentclassname[1024];

/*
    We will need the class name in two forms -
	1) in a form suitable for a file name.  To get this we replace
	   the slashes between package names with F_DELIM.
	2) in the form of a C-symbol.  To get this we replace the
	   slashes between package names with C_DELIM.

    NOTE - for now we use the same deliminter for both C symbols and
	   for filenames.  It is '_' which does not guarrantee no
	   conflicts in either case but at least is consistent.
*/

#define DIR_DELIM   '/'
#define C_DELIM     '_'
#define F_DELIM     '_'

struct structdecl {
    char *class;
    struct structdecl *next;
};
static struct structdecl *structdecls;

static char *
strsub(char *src, char *dst, char sub)
{
    char c;
    char *d = dst;

    while ((c = *src++) != 0)
	if (c == DIR_DELIM)
	    *d++ = sub;
	else
	    *d++ = c;
    *d = '\0';
    return dst;
}

static char *
strnsub(char *src, char *dst, char sub, int n)
{
    char c;
    char *d = dst;

    while (n-- > 0)
	if ((c = *src++) == DIR_DELIM)
	    *d++ = sub;
	else
	    *d++ = c;
    *d = '\0';
    return dst;
}

/* Open a temporary file */
static FILE *open_output(void) 
{
    if (outfp != 0) {
	return outfp;
    }
    tempfn = tempnam(tempdir, "javah");
    if ((outfp = fopen(tempfn, "w")) == 0) {
	perror(tempfn);
    }
    /* fprintf(stderr, "writing to %s\n", tempfn); */

    fprintf(outfp, "/* DO NOT EDIT THIS FILE - it is machine generated */\n");
    if (stubmode) {
	fprintf(outfp, "#include <StubPreamble.h>\n");
    } else {
	fprintf(outfp, "#include <native.h>\n");
    }

	/* Added so that dynamic linking works on BeOS. Paul. */
	fprintf(outfp, "#pragma export on\n");
	
    return outfp;
}

/* Close the temporary file and rename it
 * to the final output file if the content
 * has changed. */
static int close_output(void) 
{
    char fname[300];

    if (outfp == 0) {
	return 0;
    }

	/* Added so that dynamic linking works on BeOS. Paul. */
	fprintf(outfp, "#pragma export off\n");
	
    fclose(outfp);
    outfp = 0;

    while (structdecls) {
	struct structdecl *p = structdecls;
	structdecls = p->next;
	free(p->class);
	free(p);
    }

    structdecls = 0;

    if (out) {
	strncpy(fname, out, sizeof(fname));
    } else {
	char tempfname[300];
	strsub(currentclassname, tempfname, F_DELIM);
	sprintf(fname, "%s/%s%s", dir, tempfname, stubmode ? ".c" : ".h");
    }

    /* fprintf(stderr, "rename %s to %s\n", tempfn, fname); */
    return renameIfDiff(tempfn, fname, dir);
}

static char *
SprintReverseType(char *s, char *fname, char *out)
{
    char buf[1000];
    register char *tn = "?";
    if (fname == 0)
	fname = "";
    switch (*s++) {
    default:
	switch (s[-1]) {
	case SIGNATURE_ANY:
	    tn = "void *";
	    break;
	case SIGNATURE_CHAR:
	    tn = "unicode";
	    break;
	case SIGNATURE_BYTE:
	    tn = "char";
	    break;
	case SIGNATURE_ENUM:
	    tn = "enum";
	    break;
	case SIGNATURE_FLOAT:
	    tn = "float";
	    break;
	case SIGNATURE_DOUBLE:
	    tn = "double";
	    break;
	case SIGNATURE_BOOLEAN:
	    tn = "/*boolean*/ long";
	    break;
	case SIGNATURE_INT:
	    tn = "long";
	    break;
	case SIGNATURE_LONG:
	    tn = "int64_t";
	    break;
	case SIGNATURE_SHORT:
	    tn = "short";
	    break;
	case SIGNATURE_CLASS:
	    tn = buf;
	    {
		register char *sp = "struct H";
		while (*sp)
		    *tn++ = *sp++;
	    }
	    for (; *s && *s != SIGNATURE_ENDCLASS;) {
		char c = *s++;

		if (c == DIR_DELIM)
		    *tn++ = C_DELIM;
		else
		    *tn++ = c;
	    }
	    *tn++ = ' ';
	    *tn++ = '*';
	    *tn = 0;
	    if (*s == SIGNATURE_ENDCLASS)
		s++;
	    tn = buf;
	    break;
	case SIGNATURE_VOID:
	    tn = "void";
	    break;
	}
	sprintf(out, (*fname
	      ? (tn[strlen(tn) - 1] == '*'
		 ? "%s%s"
		 : "%s %s")
	      : "%s"),
	tn, fname);
	break;
    case 0:
	strcpy(out, fname);
	s--;
	break;
    case SIGNATURE_ARRAY:
	switch (*s++) {
	  case SIGNATURE_BYTE:
	    sprintf(out, *fname ? "HArrayOfByte *%s" : "HArrayOfByte *", fname);
	    break;
	  case SIGNATURE_CHAR:
	    sprintf(out, *fname ? "HArrayOfChar *%s" : "HArrayOfChar *", fname);
	    break;
	  case SIGNATURE_SHORT:
	    sprintf(out, *fname ? 
		    "HArrayOfShort *%s" : "HArrayOfShort *", fname);
	    break;
	  case SIGNATURE_INT:
	    sprintf(out, *fname ? "HArrayOfInt *%s" : "HArrayOfInt *", fname);
	    break;
	  case SIGNATURE_BOOLEAN:
	    sprintf(out, *fname ? "/*boolean*/ HArrayOfInt *%s" :
	        "HArrayOfInt *", fname);
	    break;
	  case SIGNATURE_LONG:
	    sprintf(out, *fname ? "HArrayOfLong *%s" : "HArrayOfLong *", fname);
	    break;
	  case SIGNATURE_FLOAT:
	    sprintf(out, 
		    *fname ? "HArrayOfFloat *%s" : "HArrayOfFloat *", fname);
	    break;
	  case SIGNATURE_DOUBLE:
	    sprintf(out, 
		    *fname ? "HArrayOfDouble *%s" : "HArrayOfDouble *", fname);
	    break;
	  case SIGNATURE_CLASS:
	    if (!strncmp(s,"java/lang/String;", sizeof("java/lang/String;")-1)) {
	        s += sizeof("java/lang/String;") - 1;
		sprintf(out, *fname ? "HArrayOfString *%s" : "HArrayOfString *",
			fname);
	    } else {
		while (*s++ != SIGNATURE_ENDCLASS);
		sprintf(out, *fname ? "HArrayOfObject *%s" : "HArrayOfObject *",
			fname);
	    }
	    break;
	  case SIGNATURE_ARRAY:
	    while (*s == SIGNATURE_ARRAY) s++;
	    if (*s++ == SIGNATURE_CLASS)
		while (*s++ != SIGNATURE_ENDCLASS);
	    sprintf(out, 
 		    *fname ? "HArrayOfArray *%s" : "HArrayOfArray *", fname);
	    break;
	}
	break;
    case SIGNATURE_FUNC:
	{
	    int constructor = strcmp(fname, "<init>") == 0;
	    char cname[300];

	    strsub(currentclassname, cname, C_DELIM);
	    sprintf(buf, "%s_%s(",
		    cname,
		    constructor ? "Initializor" : fname);
	    fname = buf + strlen(buf);
	    sprintf(fname, "struct H%s *,", cname);
	    fname += strlen(fname);
	    while (*s != SIGNATURE_ENDFUNC && *s) {
		s = SprintReverseType(s, (char *) 0, fname);
		fname += strlen(fname);
		*fname++ = ',';
	    }
	    if (fname[-1] == ',') fname--;
	    *fname++ = SIGNATURE_ENDFUNC;
	    *fname++ = 0;
	    s = SprintReverseType(constructor ? "" : *s ? s + 1 : s, buf, out);
	}
	break;
    }
    return s;
}

static char *
PrintType(FILE *f, char *s, char *fname)
{
    char buf[1000];
    char fnb[100];
    register char *src, *dst;
    dst = fnb;
    for (src = fname; *src && dst - fnb < sizeof fnb - 1 &&
        *src != SIGNATURE_FUNC;) {
	char c = *src++;
	if (c == DIR_DELIM)
	    *dst++ = C_DELIM;
	else
	    *dst++ = c;
    }
    *dst = 0;
    s = SprintReverseType(s, fnb, buf);
    fwrite(buf, strlen(buf), 1, f);
    return s;
}


/*
 * Returns 1 if diff, 0 if same, -1 if error
 */
static int
renameIfDiff(char *tempfn, char *newfn, char *newdir)
{
    FILE *f, *nf;
    char buf[1024];
    long c;

    if ((f = fopen(tempfn, "r")) == 0) {
	perror(tempfn);
	return -1;
    } else if (nf = fopen(newfn, "r")) {
	while ((c = getc(f)) == getc(nf)) {
	    if (c == -1) {
		fclose(f);
		fclose(nf);
		unlink(tempfn);
		if (verbose)
		    fprintf(stderr, "  (no change)\n");
		return 0;
	    }
	}
    }

    rewind(f);
    if (nf) {
	fclose(nf);
    }
    if ((nf = fopen(newfn, "w")) == 0) {
	if (mkdir(newdir, 0777) != 0) {
	    perror(newdir);
	    return -1;
	}
	nf = fopen(newfn, "w");
    }
    if (nf == 0) {
	perror(newfn);
	return -1;
    }
    while ((c = fread(buf, sizeof(char), sizeof(buf), f)) > 0) {
	fwrite(buf, sizeof(char), c, nf);
    }
    fclose(f);
    fclose(nf);
    unlink(tempfn);
    if (verbose)
	fprintf(stderr, "  %s => %s\n", tempfn, newfn);
    return 1;
}

static void
generateForwardDeclaration(FILE *f, char *p, int len) {
    char buf[128];
    struct structdecl *s;
    strnsub(p, buf, '_', len);
    for (s = structdecls ; s ; s = s->next) {
	if (!strcmp(s->class, buf)) {
	    return;
	}
    }
    fprintf(f, "struct H%s;\n", strnsub(p, buf, '_', len));
    s = (struct structdecl *)malloc(sizeof(struct structdecl));
    s->next = structdecls;
    s->class = strdup(buf);
    structdecls = s;
}

static void
forwardDeclare(FILE *f, struct fieldblock *fb) 
{
    char *fs = fieldsig(fb);
    if (*fs == SIGNATURE_FUNC) {
	for(fs++ ; *fs != '\0' ; fs++) {
	    if (*fs == SIGNATURE_CLASS) {
		char *p = ++fs;
		while (*fs != SIGNATURE_ENDCLASS) fs++;
		generateForwardDeclaration(f, p, fs - p);
	    }
	}
    } else {
	while (*fs == SIGNATURE_ARRAY) fs++;
	if (*fs == SIGNATURE_CLASS) {
	    char *p = ++fs;
	    while (*fs != SIGNATURE_ENDCLASS) fs++;
	    generateForwardDeclaration(f, p, fs - p);
	}
    }
}

static int
DumpClassHeader(ClassClass * cb)
{
    FILE *f;
    register struct fieldblock *fb;
    register struct methodblock *mb;
    register nslots;
    int offset = 0;
    char hfname[300];   /* header file name */
    char hcname[300];   /* C symbol name.  */
    char mangledFieldName[1024];

    mangleUTFString(classname(cb), currentclassname,
		    sizeof(currentclassname), MangleUTF_Class);
    strsub(currentclassname, hfname, F_DELIM);
    strsub(currentclassname, hcname, C_DELIM);


    if ((f = open_output()) == 0) {
	return -1;
    }

    if (verbose)
	fprintf(stderr, "=> %s\n", tempfn);
    fprintf(f, "/* Header for class %s */\n\n", hfname);
    fprintf(f, "#ifndef _Included_%s\n#define _Included_%s\n",
	    hcname, hcname);
    makeslottable(cb);
    if (cb != classJavaLangClass && cb != classJavaLangObject) { 
	int NumFieldsWritten = 0;

	/* first generate forward declarations */
	for (nslots = 0; (unsigned)nslots < cbSlotTableSize(cb); nslots++) {
	    fb = (cbSlotTable(cb))[nslots];
	    if ((fb->access & ACC_STATIC) == 0) {
		forwardDeclare(f, fb);
	    }
	}

	fprintf(f, "\ntypedef struct Class%s {\n",
		hcname);
	for (nslots = 0; (unsigned)nslots < cbSlotTableSize(cb); nslots++) {
	    char mangledFieldName[1024];
	    fb = (cbSlotTable(cb))[nslots];
	    mangleUTFString(fb->name, mangledFieldName, 
			    sizeof(mangledFieldName), MangleUTF_FieldStub);
	    if ((fb->access & ACC_STATIC) == 0) {
		char *fs = fieldsig(fb);
		NumFieldsWritten++;
		if (fb->u.offset != offset)
		    fprintf(f, "    char _PAD%d_[%d];\n",
			    fb->u.offset, fb->u.offset - offset);
		fprintf(f, "    ");
		switch (fs[0]) {
		  case SIGNATURE_ARRAY:
		    switch (fs[1]) {
		      case SIGNATURE_BYTE:
			fprintf(f, "struct HArrayOfByte *%s;\n", mangledFieldName);
			break;
		      case SIGNATURE_CHAR:
			fprintf(f, "struct HArrayOfChar *%s;\n", mangledFieldName);
			break;
		      case SIGNATURE_BOOLEAN:
			fprintf(f, "struct HArrayOfInt *%s;\n", mangledFieldName);
			break;
		      case SIGNATURE_SHORT:
			fprintf(f, "struct HArrayOfShort *%s;\n", mangledFieldName);
			break;
		      case SIGNATURE_INT:
			fprintf(f, "struct HArrayOfInt *%s;\n", mangledFieldName);
			break;
		      case SIGNATURE_LONG:
			fprintf(f, "struct HArrayOfLong *%s;\n", mangledFieldName);
			break;
		      case SIGNATURE_FLOAT:
			fprintf(f, "struct HArrayOfFloat *%s;\n", mangledFieldName);
			break;
		      case SIGNATURE_DOUBLE:
			fprintf(f, "struct HArrayOfDouble *%s;\n", mangledFieldName);
			break;
		      case SIGNATURE_ARRAY:
			fprintf(f, "struct HArrayOfArray *%s;\n", mangledFieldName);
			break;
		      case SIGNATURE_CLASS:
			fprintf(f, "struct HArrayOfObject *%s;\n", mangledFieldName);
			break;
		      default:
			fprintf(f, "void *%s;\n", mangledFieldName);
			break;
		    }
		    break;
		  case SIGNATURE_BYTE:
		  case SIGNATURE_CHAR:
		  case SIGNATURE_SHORT:
		    fprintf(f, "long %s;\n", mangledFieldName);
		    break;
		  default:
		    PrintType(f, fs, mangledFieldName);
		    fprintf(f, ";\n");
		}
		switch (fs[0]) {
		case SIGNATURE_ANY:
		    offset = fb->u.offset + sizeof(void *);
		    break;
		case SIGNATURE_BYTE:
		    offset = fb->u.offset + sizeof(/*char*/long);
		    break;
		case SIGNATURE_CHAR:
		    offset = fb->u.offset + sizeof(/*unicode*/long);
		    break;
		case SIGNATURE_DOUBLE:
		    offset = fb->u.offset + sizeof(double);
		    break;
		case SIGNATURE_ENUM:
		    offset = fb->u.offset + sizeof(/*enum*/long);
		    break;
		case SIGNATURE_FLOAT:
		    offset = fb->u.offset + sizeof(float);
		    break;
		case SIGNATURE_INT:
		    offset = fb->u.offset + sizeof(long);
		    break;
		case SIGNATURE_LONG:
		    offset = fb->u.offset + sizeof(int64_t);
		    break;
		case SIGNATURE_SHORT:
		    offset = fb->u.offset + sizeof(/*short*/long);
		    break;
		case SIGNATURE_VOID:
		    offset = fb->u.offset + sizeof(/*void*/long);
		    break;
		case SIGNATURE_BOOLEAN:
		    offset = fb->u.offset + sizeof(long);
		    break;
		default:
		    offset = fb->u.offset + sizeof(OBJECT);
		    break;
		}
            } else if (fb->access & ACC_VALKNOWN) {
                if (fb->signature[0] == SIGNATURE_DOUBLE) {
                    Java8 t1;
                    double d;
                    d = GET_DOUBLE(t1, twoword_static_address(fb));
		    fprintf(f, "#define %s_%s %gD\n", hcname,
			    mangledFieldName, d);
                } else if (fb->signature[0] == SIGNATURE_LONG) {
                    Java8 t1;
                    int64_t l;
                    char buf[40];
                    l = GET_INT64(t1, twoword_static_address(fb));
                    ll2str(l, buf, buf + sizeof(buf));
                    fprintf(f, "#define %s_%s %sLL\n", hcname,
                            mangledFieldName, buf);
                } else if (fb->signature[0] == SIGNATURE_FLOAT) {
		    float fl = *(float *)normal_static_address(fb);
		    fprintf(f, "#define %s_%s %gf\n", hcname,
			    mangledFieldName, fl);
                } else {
                    fprintf(f, "#define %s_%s %dL\n", hcname,
                            mangledFieldName, *normal_static_address(fb));
                }
            } else {
                fprintf(f, "/* Inaccessible static: %s */\n", mangledFieldName);
	    }
	}
	if (NumFieldsWritten == 0)
	    fprintf(f, "    char PAD;\t/* ANSI C requires structures to have a least one member */\n");
	fprintf(f, "} Class%s;\n", hcname);
	fprintf(f, "HandleTo(%s);\n\n", hcname);
    }
    fprintf(f, "#ifdef __cplusplus\nextern \"C\" {\n#endif\n");
    for (nslots = cb->methods_count, mb = cbMethods(cb); --nslots >= 0; mb++) {
	    fb = &mb->fb;
	    mangleUTFString(fb->name, mangledFieldName, 
			    sizeof(mangledFieldName), MangleUTF_FieldStub);
	    if (fb->access & ACC_NATIVE) {
		forwardDeclare(f, fb);
		fprintf(f, "extern ");
		PrintType(f, fieldsig(fb), mangledFieldName);
		fprintf(f, ";\n");
	    } 
    }
    fprintf(f, "#ifdef __cplusplus\n}\n#endif\n");
    fprintf(f, "#endif\n");

    return out ? 0 : close_output();
}


/*
 * Just decode those sigs that are meaningful to stubs.  We don't
 * support lots of things.
 */
static char *
SprintReverseArgs(char *sig, char *header, char *proto, char *call,
		  int argn, int *arg_size_p)
{

    int arg_size = 1;
    char *type;
    switch(*sig++) {
	case SIGNATURE_CLASS:
	    while (*sig++ != SIGNATURE_ENDCLASS);
	    type = "void *";
	    sprintf(call, ",((_P_[%d].p))", argn);
	    break;

	case SIGNATURE_BOOLEAN:
	case SIGNATURE_BYTE:
	case SIGNATURE_SHORT:
	case SIGNATURE_CHAR:
	case SIGNATURE_INT:
	    type = "long";
	    sprintf(call, ",((_P_[%d].i))", argn);
	    break;

	case SIGNATURE_FLOAT:
	    type = "float";
	    sprintf(call, ",((_P_[%d].f))", argn);
	    break;

	case SIGNATURE_LONG:
	    type = "int64_t";
	    sprintf(call, ",GET_INT64(_t%d, _P_+%d)", argn, argn);
	    arg_size = 2;
	    break;

	case SIGNATURE_DOUBLE:
	    type = "double";
	    sprintf(call, ",GET_DOUBLE(_t%d, _P_+%d)", argn, argn);
	    arg_size = 2;
	    break;

	case SIGNATURE_VOID:
	    type = "void *";
	    sprintf(call, ",((_P_[%d].p))", argn);
	    break;

	case SIGNATURE_ARRAY:
	    for (; *sig == SIGNATURE_ARRAY ; sig++);
	    if (*sig++ == SIGNATURE_CLASS) {
		while (*sig++ != SIGNATURE_ENDCLASS);
	    }
	    type = "void *";
	    sprintf(call, ",((_P_[%d].p))", argn);
	    break;

	default:
	    fprintf(stderr, "%s: illegal signature\n", currentclassname);
	    return 0;
    }
    sprintf(proto, ",%s", type);
    if (arg_size == 2) 
	sprintf(header, "\tJava8 _t%d;\n", argn);
    *arg_size_p = arg_size;
    return sig;
}

static int
PrintStub(FILE *f, char *fname, char *cname, struct methodblock *mb)
{
    struct fieldblock *fb = &mb->fb;
    char *s;
    char proto[1024], funcall[1024], header[1024];
    char mangledFieldName[1024];
    char *proto_p, *funcall_p, *header_p;
    char *obj_arg;
    int argn;
    
    if (!(fb->access & ACC_NATIVE))
	return 0;
    
    mangleMethodName(mb, funcall, 1024, "_stub");
    mangleUTFString(fb->name, mangledFieldName, sizeof(mangledFieldName),
		    MangleUTF_FieldStub);
    fprintf(f, "/* SYMBOL: \"%s/%s%s\", %s */\n",
	    currentclassname, mangledFieldName, fieldsig(fb), funcall);
#ifdef WIN32
    fprintf(f, EXPORT " stack_item *%s(stack_item *_P_,struct execenv *_EE_) {\n",
	    funcall);
#else
    fprintf(f, "stack_item *%s(stack_item *_P_,struct execenv *_EE_) {\n",
	    funcall);
#endif
    
    /* Generate function prototype args */
    s = fieldsig(fb) + 1;   /* first char is SIGNATURE_FUNC */

    proto_p = proto;
    header_p = header;
    *proto_p = *header_p = '\0';

    if (fb->access & ACC_STATIC) {
	argn = 0;		/* first argument is _P_[0] */
	strcpy(funcall, "NULL");
    } else {
	argn = 1;
	strcpy(funcall, "_P_[0].p");
    }
    funcall_p = funcall + strlen(funcall);

    while (s && *s != SIGNATURE_ENDFUNC) { 
	int argsize;
	s = SprintReverseArgs(s, header_p, proto_p, funcall_p, argn, &argsize);
	if (s == 0) return -1;
	header_p += strlen(header_p);
	proto_p += strlen(proto_p);
	funcall_p += strlen(funcall_p);
	argn += argsize;
	if (   header_p >= header + sizeof(header) || 
	       proto_p >= proto + sizeof(proto) || 
	       funcall_p >= funcall + sizeof(funcall))  { 
	    fprintf(stderr, "out of buffer space!\n");
	    return -1;
	}
    }
    switch(*++s) {
      case SIGNATURE_VOID:
	  fprintf(f, "%s", header);
	  fprintf(f, "\textern void %s_%s(void *%s);\n",
		  cname, mangledFieldName, proto);
	  fprintf(f, "\t(void) %s_%s(%s);\n", 
		  cname, mangledFieldName, funcall);
	  fprintf(f, "\treturn _P_;\n}\n");
	  break;


      case SIGNATURE_FLOAT:
	  fprintf(f, "%s", header);
	  fprintf(f, "\textern float %s_%s(void *%s);\n",
		  cname, mangledFieldName, proto);
	  fprintf(f, "\t_P_[0].f = %s_%s(%s);\n", 
		  cname, mangledFieldName, funcall);
	  fprintf(f, "\treturn _P_ + 1;\n}\n");
	  break;

      case SIGNATURE_DOUBLE:
	  fprintf(f, "\tJava8 _tval;\n");
	  fprintf(f, "%s", header);
	  fprintf(f, "\textern double %s_%s(void *%s);\n",
		  cname, mangledFieldName, proto);
	  fprintf(f, "\tSET_DOUBLE(_tval, _P_, %s_%s(%s));\n", 
		  cname, mangledFieldName, funcall);
	  fprintf(f, "\treturn _P_ + 2;\n}\n");
	  break;

      case SIGNATURE_CLASS:
      case SIGNATURE_ARRAY:
	  fprintf(f, "%s", header);
	  fprintf(f, "\textern void* %s_%s(void *%s);\n",
		  cname, mangledFieldName, proto);
	  fprintf(f, "\t_P_[0].p = %s_%s(%s);\n", 
		  cname, mangledFieldName, funcall);
	  fprintf(f, "\treturn _P_ + 1;\n}\n");
	  break;

      case SIGNATURE_BYTE:
      case SIGNATURE_SHORT:
      case SIGNATURE_CHAR:
      case SIGNATURE_INT:
	  fprintf(f, "%s", header);
	  fprintf(f, "\textern long %s_%s(void *%s);\n",
		  cname, mangledFieldName, proto);
	  fprintf(f, "\t_P_[0].i = %s_%s(%s);\n", 
		  cname, mangledFieldName, funcall);
	  fprintf(f, "\treturn _P_ + 1;\n}\n");
	  break;

      case SIGNATURE_BOOLEAN:
	  fprintf(f, "%s", header);
	  fprintf(f, "\textern long %s_%s(void *%s);\n",
		  cname, mangledFieldName, proto);
	  fprintf(f, "\t_P_[0].i = (%s_%s(%s) ? TRUE : FALSE);\n", 
		  cname, mangledFieldName, funcall);
	  fprintf(f, "\treturn _P_ + 1;\n}\n");
	  break;

      case SIGNATURE_LONG:
	  fprintf(f, "\tJava8 _tval;\n");
	  fprintf(f, "%s", header);
	  fprintf(f, "\textern int64_t %s_%s(void *%s);\n",
		  cname, mangledFieldName, proto);
	  fprintf(f, "\tSET_INT64(_tval, _P_, %s_%s(%s));\n", 
		  cname, mangledFieldName, funcall);
	  fprintf(f, "\treturn _P_ + 2;\n}\n");
	  break;

      default:
	  fprintf(stderr, "%s: unsupported function return type: %c\n",
		  currentclassname, *s);
	  return -1;
    }
    return 0;
}


int
DumpClassStub(ClassClass * cb)
{
    FILE *f;
    register struct methodblock *mb;
    register nslots;
    int offset = 0;
    char fname[300];   /* stub file name */
    char cname[300];   /* C symbol name.  */

    mangleUTFString(classname(cb), currentclassname,
		    sizeof(currentclassname), MangleUTF_Class);
    strsub(currentclassname, fname, F_DELIM);
    strsub(currentclassname, cname, C_DELIM);

    if ((f = open_output()) == 0) {
	return -1;
    }
    if (verbose)
	fprintf(stderr, "=> %s\n", tempfn);

    fprintf(f, "\n/* Stubs for class %s */\n", currentclassname);
    makeslottable(cb);

    for (nslots = cb->methods_count, mb = cbMethods(cb); --nslots >= 0; mb++) {
	if (mb->code == 0) {
	    if (PrintStub(f, fname, cname, mb)) {
		fclose(f);
		unlink(tempfn);
		sysExit(-1);
	    }
	}
    }
    return out ? 0 : close_output();
}

#ifdef WIN32
char *
classpath(void)
{
    char *cp = getenv("CLASSPATH");
    char loaddir[MAX_PATH];
    char *s;

    if (cp == NULL) cp = ".";

    GetModuleFileName(NULL, loaddir, MAX_PATH);
    *(strrchr(loaddir, '\\')) = '\0';

    s = (char *)malloc(strlen(cp) + strlen(loaddir) * 2 + 64);
    sprintf(s, "CLASSPATH=%s;%s\\..\\classes;%s\\..\\lib\\classes.zip",
		cp, loaddir, loaddir);

    return s;
}


void
InitializeClassPath(int argc, char **argv)
{
    int i;
    char *cp = classpath();
    char *s;

    for (i = 1; i < argc; i++) {
        if (strcmp(argv[i], "-classpath") == 0) {
            if (i + 1 >= argc) {
                fprintf(stderr, "Missing destination directory name after "
                                "'-classpath'\n");
                exit(1);
            }
            cp = argv[i+1];
        }
    }

    s = (char *)malloc(strlen(cp) + 16);
    sprintf(s, "CLASSPATH=%s", cp);
    putenv(s);
}
#else
static void
InitializeClassPath(int argc, char **argv)
{
    int i;
    char *c;
    const char *cp = "CLASSPATH=";

    for (i = 1; i < argc; i++) {
	if (argv[i] && strcmp(argv[i], "-classpath") == 0) {
	    if (i+1 == argc || argv[i+1] == 0) {
		fprintf(stderr,
			"Missing destination directory name after "
			"'-classpath'\n");
		sysExit(1);
	    }
	    c = (char *)malloc(strlen(argv[i+1]) + strlen(cp) + 1);
	    sprintf(c, "%s%s", cp, argv[i+1]);
	    putenv(c);
	}
    }
    return;
}
#endif

#include <path_md.h>		/* For LOCAL_DIR_SEPARATOR */

main(int argc, char **argv)
{
    int retcode = 0;
    int donesomething = 0;
    long T;
    SkipSourceChecks = 1;

    if ((progname = strrchr(argv[0], LOCAL_DIR_SEPARATOR)) != 0) {
        progname++;
    } else {
        progname = argv[0];
    }

    InhibitExecute = 1;

    InitializeClassPath(argc, argv);
    if (!DoImport("java/lang/Object", 0)) {
	fprintf(stderr, "java.lang.Object not found: aborting\n");
	sysExit(1);
    }
    while (--argc > 0) {
	if ((++argv)[0][0] == '-') {
	    if ((strcmp(*argv, "-d") == 0) && (argc > 1)) {
		dir = *++argv;
	        argc--;
	    } else if ((strcmp(*argv, "-o") == 0) && (argc > 1)) {
		out = *++argv;
		argc--;
	    } else if ((strcmp(*argv, "-td") == 0) && (argc > 1)) {
		tempdir = *++argv;
		argc--;
	    } else if (strcmp(*argv, "-stubs") == 0) {
		stubmode++;
	    } else if (strcmp(*argv, "-v") == 0) {
		verbose++;
	    } else if (strcmp(*argv, "-classpath") == 0) {
		argc--;
		argv++;
	    } else if (strcmp(*argv, "-version") == 0) {
                fprintf(stderr, "%s version \"%s\"\n", progname, RELEASE);
                donesomething++;
	    } else {
		fprintf(stderr, "%s: illegal argument\n", *argv);
		close_output();
		return 1;
	    }
	} else if (strchr(*argv, '/')) {
	    fprintf(stderr, "Invalid class name: %s\n", *argv);
	    close_output();
	    return 1;
	} else {
	    ClassClass *cb;
	    char *classcopy = strdup(*argv), *p;

	    /* Convert all periods in the classname to slashes */
	    for (p = classcopy; ((p = strchr(p, '.')) != 0); *p++ = '/');

	    cb = FindClass(0, classcopy, TRUE);
	    donesomething++;
	    if (cb == 0) {
		retcode = 1;
		fprintf(stderr, "%s: no such class\n", classcopy);
		continue;
	    }
	    if (stubmode) {
		if (DumpClassStub(cb) < 0) {
		    retcode = 1;
		}
	    } else {
		if (DumpClassHeader(cb) < 0) {
		    retcode = 1;
		}
	    }
	}
    }
    if (!donesomething) {
	retcode = 1;
	fprintf(stderr, 
            "Usage: %s [-v] [-version] [-l filename] classes...\n", progname);
    }
    if (close_output() < 0) {
	retcode = 1;
    }
    return retcode;
}

int
OpenCode(char *fn, char *sfn, char *dir, struct stat * st)
{
    long codefd;
#ifdef WIN32
    if (fn == 0 || (codefd = open(fn, O_BINARY|O_RDONLY)) < 0
	    || stat(fn, st) < 0)
#else
    if (fn == 0 || (codefd = open(fn, 0, 0644)) < 0
	    || fstat(codefd, st) < 0)
#endif
	return -2;
    return codefd;
}

HObject *
AllocHandle(struct methodtable *mptr, ClassObject *p)
{
    JHandle *h = malloc(sizeof(JHandle));

    h->methods = mptr;
    h->obj = (ClassObject *) p;
    return (HObject *) h;
}




/* Dummies */
#undef DumpThreads

void
DumpThreads()
{
}

char *
vramalloc(int n)
{
    return malloc(n);
}


long
CallInterpreted(struct methodblock *mb, void *obj, ...) {
    return 0;
}

ExecEnv *
EE() {
    static struct execenv lee;
    return &lee;
}

/* ResolveClassStringConstant is called by the classloader to resolve the
 * constant pool entry, before assigning a value to a final [i.e. constant]
 * String.
 *
 * For javah, we don't care.
 */

bool_t
ResolveClassStringConstant(ClassClass *cb, unsigned index, struct execenv *ee)
{
    return TRUE;
}

#ifndef WIN32
/*
 * Need the following definition for zip.c
 */

#ifndef sysOpen
int
sysOpen(const char *path, int oflag, int mode)
{
    return(open(path, oflag, mode));
}
#endif
#endif
