/*
BeSnow by Dean Luick, based on xsnow 1.40 by Rick Jansen (rick@sara.nl).

Version
0.82	Restructure source so besnow.c could be used as a screen saver module.

0.80	This is a port of xsnow 1.40.  Port problems:
			+ no preferences
			+ cannot find obscuring windows
			+ cannot sink to back of screen
			+ snow pileup not done due to poor implemention of BRegion
				-- no rect merging
		Problems inherent from xsnow:
			+ drawing of sleigh/snow/trees interferes with each other


TODO
+ Implement preferences.
+ Fix up snow pileup on the ground.
+ Snow built up on the bottom isn't restored when we get an expose event.
	This is correct in background mode if we have pileup on obscuring
	windows implemented.  I'm not shure what we should do in window mode.
+ Get window positions to pile up snow. Not possible with current APIs.
+ Add Santa blowing in the wind.
+ Get a zero-width border for the background window.
+ Draw so that Santa and snowflakes don't flicker when overlapping.
	Experiment with doing an off-screen draw followed by a drawbitmap.
+ Cleanup
	- Move decls from header (not bitmap decls)
	- Make code more modular
	- Take more advantage of C++.
*/
#include <Bitmap.h>
#include <Region.h>
#include <View.h>
#include "besnow.h"
#include "snowmap.h"

/* option variables */
static int gSmoothWhirl = 1;
static int gShowTrees = 1;
static int gShowSanta = 1;
static int gShowRudolf = 1;
static int gNoKeepSnow = 0;
static int gNoKeepSBot = 0;
static int gNoWind = 0;
static int gUseFillForClear = 1;

static int gMaxSnowflakes = INITIALSNOWFLAKES;
static int gMaxXStep = MAXXSTEP;
static int gMaxYStep = MAXYSTEP;
static int gWhirlFactor = WHIRLFACTOR;
static int gMaxScrSnowDepth = INITSCRSNOWDEPTH;

static int gSantaSize = 1;		/* current Santa size 0,1,2 */
static int gSantaSpeed = -1;	/* Santa's forward speed (not yet initialized) */
static int gWindTimer = 30;		/* seconds between storm checks */
static int gSnowDelay = 50000;	/* useconds between snow updates */

static rgb_color gSnowColor;
static rgb_color gTreeColor;
static rgb_color gSantaColor;
static rgb_color gSleighColor;
static rgb_color gFurColor;
static rgb_color gRudolfColor;


static int gDisplayWidth, gDisplayHeight;

/* other variables */
static bool gSnowInitialized = FALSE;
static Snow *gSnowFlakes;
static Santa gClaus;   
static bool gStopWindLoop = FALSE;		/* wind thread loop controller */
static bool gStopSnowLoop = FALSE;		/* snow thread loop controller */
static bool gSnowProcDone = FALSE;		/* TRUE when snow loop finishes */
static sem_id gWindSemaphore;
static thread_id gWindThread;
static thread_id gSnowThread;

/* for building up snow at bottom of screen and windows */
static BRegion *gSnowCatchRegion = NULL;

/* wind variables */
static int gWindState = 0;			/* 0 off, 1 slowing down, 2 windy */
static int gWindDirection = 0;		/* current gWindState gWindDirection */

/* forward decls */
static long SnowProc(void *arg);
static long WindProc(void *arg);

static BBitmap *XBMToBeBitmap(char *srcBits, int height, int width);
static int RandInt(int maxVal);
static void InitSnowflake(int rx);
static void UpdateSnowflake(BView *sv, int rx);
static void DrawSnowflake(BView *sv, int rx);
static void EraseSnowflake(BView *sv, int rx);
static void DrawTannenbaum(BView *sv, int x, int y);
static void InitSanta(void);
static void UpdateSanta(BView *sv);
static void DrawSanta(BView *sv);
static void EraseSanta(BView *sv);


static long WindProc(void *arg)
{
	sem_id sem = (sem_id) arg;
	int nextTime = gWindTimer;

	while (!gStopWindLoop) {
		if (acquire_sem_etc(sem, 1, B_TIMEOUT, nextTime*1000000.0) == B_NO_ERROR)
			release_sem(sem);

	    if (gNoWind) {
	    	gWindDirection = 0;
	    	gWindState = 0;
	    	nextTime = gWindTimer;
	    } else {
		    if (RandInt(100) > 65) {
		        if (RandInt(10) > 4)
		            gWindDirection = 1;
		        else
		            gWindDirection = -1;
		
		        gWindState = 2;
		        nextTime = RandInt(5)+1;
		    } else {
		        if (gWindState == 2) {
		            gWindState = 1;
		            nextTime = RandInt(3)+1;
		        } else {
		            gWindState = 0;
		            nextTime = gWindTimer;
		        }
		    }
	    }
    }

    return 0;
}


void SnowInit(BView *view)
{
	PictureMap *pm;
	int i;
	int maxSnowFlakeHeight = 0;		/* will be initialzed to the biggest flake */

#define SetColor(c,r,g,b) \
	(c).red=r,(c).green=g,(c).blue=b,(c).alpha=0

	SetColor(gSnowColor,  255,255,255);	/* white */
	SetColor(gTreeColor,    0,178,  0);
	SetColor(gSantaColor, 255,  0,  0);
	SetColor(gSleighColor,  0,  0,  0);	/* black */
	SetColor(gFurColor,   255,255,255);	/* white */
	SetColor(gRudolfColor,255,  0,  0);

	view->Window()->Lock();
	gDisplayWidth = view->Frame().IntegerWidth();
	gDisplayHeight = view->Frame().IntegerHeight();
	view->SetViewColor(desktop_color());
	view->SetHighColor(gSnowColor);
	view->SetLowColor(desktop_color());
	view->SetDrawingMode(B_OP_OVER);
	view->Window()->Unlock();

	/* Seed random */
	srand(time(NULL));
 
    /* Santa speed is different for different size Santa */
    if (gSantaSpeed < 0)
    	gSantaSpeed = Speed[gSantaSize];

	/* Do not let all of the display snow under */
	if (gMaxScrSnowDepth > gDisplayHeight-SNOWFREE)
		gMaxScrSnowDepth = gDisplayHeight-SNOWFREE;


	/* Create the snowflake pixmaps */
	for (i = 0; i <= NUMSNOWFLAKETYPE; i++) {
		pm = &snowPix[i];
		pm->pixmap = XBMToBeBitmap(pm->bits, pm->width, pm->height);
		/* remember the biggest flake */
		if (pm->height > maxSnowFlakeHeight) maxSnowFlakeHeight = pm->height;
	}

	/* Allocate structures containing the coordinates and things */
	gSnowFlakes = (Snow *)malloc(sizeof(Snow) * MAXSNOWFLAKES);

	/* Create the tree pixmap */
	pm = &tannenbaumPix[0];
	pm->pixmap = XBMToBeBitmap(pm->bits, pm->width, pm->height);

	/* Create the Santa pixmaps (only initialize current size) */
	for (i = 0; i < NUMSANTAFRAMES; i++) {
		pm = &sleighPix[gSantaSize][i];
		pm->pixmap = XBMToBeBitmap(pm->bits, pm->width, pm->height);
	}

	/* Pixmaps of the man in the red suit himself */
	pm = &santaPix[gSantaSize];
	pm->pixmap = XBMToBeBitmap(pm->bits, pm->width, pm->height);
	/* The fur in his hat */
	pm = &santaFurPix[gSantaSize];
	pm->pixmap = XBMToBeBitmap(pm->bits, pm->width, pm->height);

	/* Initialize all gSnowFlakes */
	for (i = 0; i < gMaxSnowflakes; i++)
		InitSnowflake(i);

	InitSanta();   

	/* The initial catch region: a little bit below the bottom of the screen */
	gSnowCatchRegion = new BRegion();
	if (!gNoKeepSBot)
		gSnowCatchRegion->Set(BRect(0,gDisplayHeight,gDisplayWidth-1,gDisplayHeight+gMaxYStep+maxSnowFlakeHeight-1));

	gWindSemaphore = create_sem(0, "WindTimer");
	if (gWindSemaphore < B_NO_ERROR) {
		BAlert *info;
	
		info = new BAlert("", "Can't allocate WindTimer semaphore", "OK");
		info->Go();	/* also deletes info */
	} else {
		gWindThread = spawn_thread(WindProc, "WindProc", B_NORMAL_PRIORITY, (void *)gWindSemaphore);
		resume_thread(gWindThread);
	}

	gSnowThread = spawn_thread(SnowProc, "SnowProc", B_NORMAL_PRIORITY, (void *)view);
	gSnowInitialized = TRUE;	/* set to true when threads start running */
}

void SnowGo(void)
{
	resume_thread(gWindThread);
	resume_thread(gSnowThread);
}

void SnowStop(void)
{
	gStopWindLoop = TRUE;
	gStopSnowLoop = TRUE;
}

bool Snowing(void)
{
	return !gSnowProcDone;
}

void SnowUnInit(void)
{
	PictureMap *pm;
	long result;
	int i;

	/* stop wind thread */
	gStopWindLoop = TRUE;
	delete_sem(gWindSemaphore);	
	wait_for_thread(gWindThread, &result);
	gStopWindLoop = FALSE;
	gWindThread = 0;
	gWindState = 0;			/* off */
	gWindDirection = 0;		/* no direction */

	/* stop snow update thread */
	gStopSnowLoop = TRUE;
	wait_for_thread(gSnowThread, &result);
	gStopSnowLoop = FALSE;
	gSnowProcDone = FALSE;
	gSnowThread = 0;


    gSantaSpeed = -1;	/* reset santa speed */
	gMaxScrSnowDepth = INITSCRSNOWDEPTH;

	/* free the snowflake pixmaps */
	for (i = 0; i <= NUMSNOWFLAKETYPE; i++) {
		pm = &snowPix[i];
		if (pm->pixmap) {
			delete pm->pixmap;
			pm->pixmap = NULL;
		}
	}

	/* free snowflakes */
	if (gSnowFlakes) {
		free(gSnowFlakes);
		gSnowFlakes = NULL;
	}

	/* free the tree pixmap */
	pm = &tannenbaumPix[0];
	if (pm->pixmap) {
		delete pm->pixmap;
		pm->pixmap;
	}

	/* free the Santa pixmaps (only free current size) */
	for (i = 0; i < NUMSANTAFRAMES; i++) {
		pm = &sleighPix[gSantaSize][i];
		if (pm->pixmap) {
			delete pm->pixmap;
			pm->pixmap = NULL;
		}
	}

	/* free pixmaps of the man in the red suit himself */
	pm = &santaPix[gSantaSize];
	if (pm->pixmap) {
		delete pm->pixmap;
		pm->pixmap = NULL;
	}
	pm = &santaFurPix[gSantaSize];
	if (pm->pixmap) {
		delete pm->pixmap;
		pm->pixmap = NULL;
	}

	/* free snow catch region */
	if (gSnowCatchRegion) {
		delete gSnowCatchRegion;
		gSnowCatchRegion = NULL;
	}
	/* after all window locks */
	gSnowInitialized = FALSE;
} 


static long SnowProc(void *arg)
{
	BView *sv = (BView *)arg;
	BWindow *sw = sv->Window();
	int i;

	while (!gStopSnowLoop) {
		snooze(gSnowDelay);
		sw->Lock();

		/* Snowflakes */
		sv->SetHighColor(gSnowColor);
		for (i = 0; i < gMaxSnowflakes; i++)
			UpdateSnowflake(sv,i);
	
		/* Draw dear Andrews */
		if (gShowTrees) {
			sv->SetHighColor(gTreeColor);
			DrawTannenbaum(sv, gDisplayWidth-150, gDisplayHeight-500);
			DrawTannenbaum(sv, gDisplayWidth-100, gDisplayHeight-200);
			DrawTannenbaum(sv, 100, gDisplayHeight-200);
			DrawTannenbaum(sv, 50, gDisplayHeight-150);
			DrawTannenbaum(sv, gDisplayWidth/2, gDisplayHeight-100);
			DrawTannenbaum(sv, 200,400);
		}
	
		/* Dear Santa */
		if (gShowSanta)
			UpdateSanta(sv);


		sv->Sync();
		sw->Unlock();
	}
	gSnowProcDone = TRUE;
	return 0;
}



/*
Convert an X bitmap into a BBitmap.
*/
static BBitmap *XBMToBeBitmap(char *srcBits, int width, int height)
{
	int x, y, srcRowBytes;
	char *dp, sb, db;
	BBitmap *bm;

	bm = new BBitmap(BRect(0,0,width-1,height-1), B_MONOCHROME_1_BIT);
	/*
	X bitmaps are aligned on 8 bit boundaries, Be bitmaps are aligned
	on 32 bit boundaries.  Thus we know the X rows are always a subset
	of the Be row.
	*/
	srcRowBytes = (width+7) / 8;

	dp = (char *)bm->Bits();
	for (y = 0; y < height; y++, srcBits += srcRowBytes, dp += bm->BytesPerRow()) {
		for (x = 0; x < srcRowBytes; x++) {
			/* reverse the bits */
			sb = *(srcBits+x);
			db = 0;
			if (sb & 0x01) db |= 0x80;
			if (sb & 0x02) db |= 0x40;
			if (sb & 0x04) db |= 0x20;
			if (sb & 0x08) db |= 0x10;
			if (sb & 0x10) db |= 0x08;
			if (sb & 0x20) db |= 0x04;
			if (sb & 0x40) db |= 0x02;
			if (sb & 0x80) db |= 0x01;
			*(dp+x) = db;
		}
	}
	return bm;
}


/*
   Generate random integer between 0 and maxVal-1.
*/
static int RandInt(int maxVal)
{
	return rand() % maxVal;
}


/*
   Give birth to a snowflake.
*/
static void InitSnowflake(int rx)
{
	Snow *r = &gSnowFlakes[rx];

	r->whatFlake = RandInt(NUMSNOWFLAKETYPE);
	
	/* Wind strikes! */
	if (gWindState) {
		if (gWindDirection == 1) 
			/* Create snow on the left of the display */
			r->x = RandInt(gDisplayWidth/3);
		else
			/* Create snow on the right of the display */
			r->x = gDisplayWidth - RandInt(gDisplayWidth/3);
		r->y =  RandInt(gDisplayHeight);
	} else {
		r->x = RandInt(gDisplayWidth - snowPix[r->whatFlake].width);
		r->y = RandInt(gDisplayHeight/10);
	}
	
	r->yStep = RandInt(gMaxYStep+1)+1;
	r->xStep = RandInt(r->yStep/4+1);
	if (RandInt(1000) > 500) r->xStep = -r->xStep;
	r->active = 1;
	r->visible = 1;
	r->insnow = 0;
}


/*
   Move a snowflake by erasing and redraw
*/
static void UpdateSnowflake(BView *sv, int rx)
{
	Snow *snow;
	int NewX;
	int NewY;
	int tmp_x;
	bool TouchDown;
	//int InVisible;
	int NoErase;

	/* move an active snowflake */

	snow = &gSnowFlakes[rx];
	NoErase = 0;  /* Always erase the flakes unless special case */


	/* Activate a new snowflake */
	if (!snow->active) {
		InitSnowflake(rx);
		/* Make sure newly created flakes aren't erased before actually drawn */
		/* Especially with gWindState, where flakes are created all over the screen */
		snow->insnow = 1; 
	}

	/* New x,y */
	if (gWindState) {
		tmp_x = abs(snow->xStep);
		if (gWindState == 1) {	/* going to stop winding */
			tmp_x += RandInt(gWhirlFactor+1) - gWhirlFactor/2;
		} else {				/* windy */
			tmp_x += RandInt(20);
		}
		snow->xStep = tmp_x * gWindDirection;

		if (snow->xStep > 50) snow->xStep = 50;
		if (snow->xStep < -50) snow->xStep = -50;
	}
	NewX = snow->x + snow->xStep;
	NewY = snow->y + snow->yStep;

	/*
	If flake disappears offscreen (below bottom) it becomes inactive.
	In case of gWindState also deactivate flakes blown offscreen sideways,
	so they'll reappear immediately. Make sure snow at bottom isn't
	disrupted.
	*/
	
	/* Snowflake at the end of it's life? */
	snow->active = (NewY < gDisplayHeight);
	/* If it is windy we need the flakes that blow offscreen! */
	if (gWindState) 
		snow->active = (snow->active && (NewX > 0) && (NewX < gDisplayWidth));


	/* If flake covered by windows don't bother drawing it */
	/* here is where we should check for a window obscuring the snowflake */
	snow->visible = 1;


	/*
	 *  Snow touches snow
	 */
	if (snow->visible) {
		/* Use middle bottom of flake */
		TouchDown = gSnowCatchRegion->Contains(BPoint(NewX+snowPix[snow->whatFlake].width/2,NewY+snowPix[snow->whatFlake].height));
		
		/* If the flake already is completely inside the region */
		if (TouchDown && snow->visible && !gNoKeepSnow) {
			/* Only add snow that's visible, else snow will grow */
			/* on the *back* of windows */

			NoErase = 1;

			/* Add a bit to the snow bottom region to make snow build up */
			/* Snow isn't built up more than about a certain amount: inside */
			/* SnowAllowedR */

			/* If flake already inside snow layer don't add it again */
#if 0
			/*
			While this works, it slows the machine down after a while.
			It looks like the BRegion class just keeps adding BRects to
			an internal list instead of consolidating solid regions.
			*/

			TouchDown = gSnowCatchRegion->Intersects(
								BRect(NewX,NewY,
									NewX+snowPix[snow->whatFlake].width-1,
									NewY+snowPix[snow->whatFlake].height-1));

			if (TouchDown) {
				gSnowCatchRegion->Include(
							BRect(NewX,NewY+snowPix[snow->whatFlake].height-2,
								NewX+snowPix[snow->whatFlake].width-1,
								NewY+snowPix[snow->whatFlake].height-1));
			}
#endif
		}
		
		/* Adjust horizontal speed to mimic whirling */
		if (gSmoothWhirl) 
			snow->xStep = snow->xStep + (RandInt(gWhirlFactor+1) - gWhirlFactor/2);
		else {
			/* Jerkier movement */
			snow->xStep = snow->xStep + RandInt(gWhirlFactor);
			if (RandInt(1000) > 500) snow->xStep = -snow->xStep;
		}
		
	}  /* if snow->visible */


	if (!gWindState) {
		if (snow->xStep > gMaxXStep) snow->xStep = gMaxXStep;
		if (snow->xStep < -gMaxXStep) snow->xStep = -gMaxXStep;
	}

	/* Limit speed inside snowlayers. 's only natural... */
	if (NoErase && !gWindState) snow->xStep = snow->xStep/2;

	/* Erase, unless something special has happened */
	if (!snow->insnow) EraseSnowflake(sv, rx);

	/* New coords, unless it's windy */
	snow->x = NewX;
	snow->y = NewY;

	/* If snowflake inside a snow layer now don't erase it next time */
	snow->insnow = NoErase;

	/* Draw if still active  and visible */
	if (snow->active && snow->visible) DrawSnowflake(sv, rx);
}



/*
	Draw snowflake.
*/
static void DrawSnowflake(BView *sv, int rx)
{
	Snow *snow = &gSnowFlakes[rx];

	sv->DrawBitmapAsync(snowPix[snow->whatFlake].pixmap, BPoint(snow->x, snow->y));
}


/*
 	Erase a snowflake.
*/
static void EraseSnowflake(BView *sv, int rx)
{
	Snow *snow = &gSnowFlakes[rx];

	if (gUseFillForClear) {
		int x, y;
		x = snow->x;
		y = snow->y;
		sv->SetDrawingMode(B_OP_COPY);
	 	sv->FillRect(BRect(x,y,x+snowPix[snow->whatFlake].width,y+snowPix[snow->whatFlake].height), B_SOLID_LOW);
		sv->SetDrawingMode(B_OP_OVER);
	} else {
		sv->SetDrawingMode(B_OP_ERASE);
		sv->DrawBitmapAsync(snowPix[snow->whatFlake].pixmap, BPoint(snow->x, snow->y));
		sv->SetDrawingMode(B_OP_OVER);
	}
}


/*
	Draw a tree
*/
static void DrawTannenbaum(BView *sv, int x, int y)
{
	sv->DrawBitmapAsync(tannenbaumPix[0].pixmap, BPoint(x, y));
}


/*
	Give birth to a Santa. (What a conception)
*/
static void InitSanta(void)      
{
	gClaus.x = -sleighPix[gSantaSize][gClaus.whatSanta].width;
	gClaus.y = RandInt(gDisplayHeight / 3)+40;
	gClaus.xStep = gSantaSpeed;
	gClaus.yStep = 1;
	gClaus.whatSanta = 0;
	gClaus.visible = 1;
}


/*
	Update Santa (How can you update the oldest icon in the world? Oh well.)
*/
static void UpdateSanta(BView *sv)
{
	//int Visible;

	if (gClaus.visible) EraseSanta(sv);

	/* Move forward */
	gClaus.x = gClaus.x + gClaus.xStep;

	/* Move down */
	if (RandInt(10) > 3) gClaus.y = gClaus.y + gClaus.yStep; 
	if (gClaus.y < 0) gClaus.y = 0;
	if (RandInt(100) > 80) gClaus.yStep = -gClaus.yStep;

	if (gClaus.x >= gDisplayWidth) InitSanta(); 

	/* If Santa entirely covered by windows don't bother drawing him... */
	gClaus.visible = 1;

	/* Next sleigh */
	if (++gClaus.whatSanta > 2) gClaus.whatSanta = 0;

	if (gClaus.visible) DrawSanta(sv);
}


/*
	Draw Santa
*/
static void DrawSanta(BView *sv)
{
	/* Draw the sleigh */
	sv->SetHighColor(gSleighColor);
	sv->DrawBitmapAsync(sleighPix[gSantaSize][gClaus.whatSanta].pixmap, BPoint(gClaus.x, gClaus.y));

	/* The Man in the Red Suit himself */
	sv->SetHighColor(gSantaColor);
	sv->DrawBitmapAsync(santaPix[gSantaSize].pixmap, BPoint(gClaus.x, gClaus.y));

	/* At last.... Rudolf! */
	if (gShowRudolf) {
		int x, y;

		x = gClaus.x + NoseX[gSantaSize];
		y = gClaus.y + NoseY[gSantaSize];
		sv->SetHighColor(gRudolfColor);
 		sv->FillRect(BRect(x,y,x+NoseXSiz[gSantaSize]-1,y+NoseYSiz[gSantaSize]-1), B_SOLID_HIGH);
	}

	/* The fur in his hat */
	/* Note: the fur in his hat is *imitation* white-seal fur, of course. */
	/* Santa is a big supporter of Greenpeace.                            */
	sv->SetHighColor(gFurColor);
	sv->DrawBitmapAsync(santaFurPix[gSantaSize].pixmap, BPoint(gClaus.x, gClaus.y));
}

/*
	Erase Santa
*/
static void EraseSanta(BView *sv)
{
	int x, y;

	x = gClaus.x;
	y = gClaus.y;
	sv->SetDrawingMode(B_OP_COPY);
 	sv->FillRect(BRect(x,y,x+sleighPix[gSantaSize][gClaus.whatSanta].width-1,y+sleighPix[gSantaSize][gClaus.whatSanta].height-1), B_SOLID_LOW);
	sv->SetDrawingMode(B_OP_OVER);
}
