/* ++++++++++

	/dev/random - true random numbers
	/dev/urandom - cryptographically strong pseudorandom numbers
	
	This code is placed into public domain. No warranties. Caveat Hacker.
	
	Questions? Mail me at <oa@iki.fi>

++++++ */

#include <kernel/OS.h>
#include <device/Drivers.h>
#include <device/KernelExport.h>
#include "random.h"

#if DEBUG
#define D(x) dprintf x
#else
#define D(x)
#endif

/* -------------------------- BEGIN MD5 --------------------------- */

/*
 * This code implements the MD5 message-digest algorithm.
 * The algorithm is due to Ron Rivest.  This code was
 * written by Colin Plumb in 1993, no copyright is claimed.
 * This code is in the public domain; do with it what you wish.
 *
 * Actually, this routine is low-endian. I figure it doesn't
 * matter, either way it's going to mix the input pretty well.
 */

typedef unsigned long uint32;

#define F1(x, y, z) (z ^ (x & (y ^ z)))
#define F2(x, y, z) F1(z, x, y)
#define F3(x, y, z) (x ^ y ^ z)
#define F4(x, y, z) (y ^ (x | ~z))

/* This is the central step in the MD5 algorithm. */
#define MD5STEP(f, w, x, y, z, data, s) \
	( w += f(x, y, z) + data,  w = w<<s | w>>(32-s),  w += x )

/*
 * The core of the MD5 algorithm, this alters an existing MD5 hash to
 * reflect the addition of 16 longwords of new data.  MD5Update blocks
 * the data and converts bytes into longwords for this routine.
 */
void
MD5Transform(uint32 buf[4], uint32 const in[16])
{
	register uint32 a, b, c, d;

	a = buf[0];
	b = buf[1];
	c = buf[2];
	d = buf[3];

	MD5STEP(F1, a, b, c, d, in[ 0]+0xd76aa478,  7);
	MD5STEP(F1, d, a, b, c, in[ 1]+0xe8c7b756, 12);
	MD5STEP(F1, c, d, a, b, in[ 2]+0x242070db, 17);
	MD5STEP(F1, b, c, d, a, in[ 3]+0xc1bdceee, 22);
	MD5STEP(F1, a, b, c, d, in[ 4]+0xf57c0faf,  7);
	MD5STEP(F1, d, a, b, c, in[ 5]+0x4787c62a, 12);
	MD5STEP(F1, c, d, a, b, in[ 6]+0xa8304613, 17);
	MD5STEP(F1, b, c, d, a, in[ 7]+0xfd469501, 22);
	MD5STEP(F1, a, b, c, d, in[ 8]+0x698098d8,  7);
	MD5STEP(F1, d, a, b, c, in[ 9]+0x8b44f7af, 12);
	MD5STEP(F1, c, d, a, b, in[10]+0xffff5bb1, 17);
	MD5STEP(F1, b, c, d, a, in[11]+0x895cd7be, 22);
	MD5STEP(F1, a, b, c, d, in[12]+0x6b901122,  7);
	MD5STEP(F1, d, a, b, c, in[13]+0xfd987193, 12);
	MD5STEP(F1, c, d, a, b, in[14]+0xa679438e, 17);
	MD5STEP(F1, b, c, d, a, in[15]+0x49b40821, 22);

	MD5STEP(F2, a, b, c, d, in[ 1]+0xf61e2562,  5);
	MD5STEP(F2, d, a, b, c, in[ 6]+0xc040b340,  9);
	MD5STEP(F2, c, d, a, b, in[11]+0x265e5a51, 14);
	MD5STEP(F2, b, c, d, a, in[ 0]+0xe9b6c7aa, 20);
	MD5STEP(F2, a, b, c, d, in[ 5]+0xd62f105d,  5);
	MD5STEP(F2, d, a, b, c, in[10]+0x02441453,  9);
	MD5STEP(F2, c, d, a, b, in[15]+0xd8a1e681, 14);
	MD5STEP(F2, b, c, d, a, in[ 4]+0xe7d3fbc8, 20);
	MD5STEP(F2, a, b, c, d, in[ 9]+0x21e1cde6,  5);
	MD5STEP(F2, d, a, b, c, in[14]+0xc33707d6,  9);
	MD5STEP(F2, c, d, a, b, in[ 3]+0xf4d50d87, 14);
	MD5STEP(F2, b, c, d, a, in[ 8]+0x455a14ed, 20);
	MD5STEP(F2, a, b, c, d, in[13]+0xa9e3e905,  5);
	MD5STEP(F2, d, a, b, c, in[ 2]+0xfcefa3f8,  9);
	MD5STEP(F2, c, d, a, b, in[ 7]+0x676f02d9, 14);
	MD5STEP(F2, b, c, d, a, in[12]+0x8d2a4c8a, 20);

	MD5STEP(F3, a, b, c, d, in[ 5]+0xfffa3942,  4);
	MD5STEP(F3, d, a, b, c, in[ 8]+0x8771f681, 11);
	MD5STEP(F3, c, d, a, b, in[11]+0x6d9d6122, 16);
	MD5STEP(F3, b, c, d, a, in[14]+0xfde5380c, 23);
	MD5STEP(F3, a, b, c, d, in[ 1]+0xa4beea44,  4);
	MD5STEP(F3, d, a, b, c, in[ 4]+0x4bdecfa9, 11);
	MD5STEP(F3, c, d, a, b, in[ 7]+0xf6bb4b60, 16);
	MD5STEP(F3, b, c, d, a, in[10]+0xbebfbc70, 23);
	MD5STEP(F3, a, b, c, d, in[13]+0x289b7ec6,  4);
	MD5STEP(F3, d, a, b, c, in[ 0]+0xeaa127fa, 11);
	MD5STEP(F3, c, d, a, b, in[ 3]+0xd4ef3085, 16);
	MD5STEP(F3, b, c, d, a, in[ 6]+0x04881d05, 23);
	MD5STEP(F3, a, b, c, d, in[ 9]+0xd9d4d039,  4);
	MD5STEP(F3, d, a, b, c, in[12]+0xe6db99e5, 11);
	MD5STEP(F3, c, d, a, b, in[15]+0x1fa27cf8, 16);
	MD5STEP(F3, b, c, d, a, in[ 2]+0xc4ac5665, 23);

	MD5STEP(F4, a, b, c, d, in[ 0]+0xf4292244,  6);
	MD5STEP(F4, d, a, b, c, in[ 7]+0x432aff97, 10);
	MD5STEP(F4, c, d, a, b, in[14]+0xab9423a7, 15);
	MD5STEP(F4, b, c, d, a, in[ 5]+0xfc93a039, 21);
	MD5STEP(F4, a, b, c, d, in[12]+0x655b59c3,  6);
	MD5STEP(F4, d, a, b, c, in[ 3]+0x8f0ccc92, 10);
	MD5STEP(F4, c, d, a, b, in[10]+0xffeff47d, 15);
	MD5STEP(F4, b, c, d, a, in[ 1]+0x85845dd1, 21);
	MD5STEP(F4, a, b, c, d, in[ 8]+0x6fa87e4f,  6);
	MD5STEP(F4, d, a, b, c, in[15]+0xfe2ce6e0, 10);
	MD5STEP(F4, c, d, a, b, in[ 6]+0xa3014314, 15);
	MD5STEP(F4, b, c, d, a, in[13]+0x4e0811a1, 21);
	MD5STEP(F4, a, b, c, d, in[ 4]+0xf7537e82,  6);
	MD5STEP(F4, d, a, b, c, in[11]+0xbd3af235, 10);
	MD5STEP(F4, c, d, a, b, in[ 2]+0x2ad7d2bb, 15);
	MD5STEP(F4, b, c, d, a, in[ 9]+0xeb86d391, 21);

	buf[0] += a;
	buf[1] += b;
	buf[2] += c;
	buf[3] += d;
}

#undef MD5STEP
#undef F1
#undef F2
#undef F3
#undef F4

/* ------------------------------ END MD5 ----------------------------- */


#ifndef MIN
#define MIN(a,b) (((a) < (b)) ? (a) : (b))
#endif

/* -----
	globals for this driver
----- */

#define POOLSIZE 128 /* 32 bit words!! must be a multiple of 16 */
#define POOLBITS (POOLSIZE*32)

/* primitive polynomial to use for stirring */
#define P1 99
#define P2 59
#define P3 31
#define P4 9
#define P5 7

#if POOLSIZE%16
#error POOLSIZE must be a multiple of 16
#endif

struct pool { ulong pos; ulong count; ulong rol; ulong words[POOLSIZE]; } pool;

/* we could use more of these, but, damnit, there aren't ways to get at the timings */
struct timer { ulong last; ulong delta; ulong delta2; } timer;

/* -----
	entropy pool management
----- */

asm ulong 
rol(register ulong word, register int i)
{
	rlwnm r3,word,i,0,31
	blr
}

/* add one word to the bitpool */
static void 
add_word_to_pool(struct pool *p, const ulong w)
{
	ulong i, x;
	
	x = rol(w, p->rol); /* roll input words so bits get spread evenly */
	i = p->pos = (p->pos-1)%POOLSIZE;
	p->rol = (p->rol+5)%32;

	/* XOR the word with positions in the pool */
	x ^= p->words[(i+P1)%POOLSIZE];
	x ^= p->words[(i+P2)%POOLSIZE];
	x ^= p->words[(i+P3)%POOLSIZE];
	x ^= p->words[(i+P4)%POOLSIZE];
	x ^= p->words[(i+P5)%POOLSIZE];
	x ^= p->words[i];

	p->words[i] ^= rol(x,1);
}

/*
 * add timebase entropy into the bitpool, and estimate the number of random
 * bits in it. 'x' is a value that somehow signifies this event.
 */
static void timed_entropy(struct pool *p, struct timer *timer, ulong x)
{
	int	delta, delta2, delta3;
	ulong time;
	double t;

	/* make a ulong of of microseconds since boot.
	 * this value wraps around every 4295 seconds 
	 * major performance decrease due to the memory access */
	t = system_time(); time = (ulong)(t*(1/4294967296.0));
	time = (ulong)(t-(time*4294967296.0));
	
	add_word_to_pool(p, x);
	add_word_to_pool(p, time);
	
	/* first, second and third order deltas to measure time variance */
	delta = time - timer->last;
	if (delta<0) delta = -delta;
	timer->last = time;

	delta2 = delta - timer->delta;
	if (delta2<0) delta2 = -delta2;
	timer->delta = delta;

	delta3 = delta2 - timer->delta2;
	if (delta3<0) delta3 = -delta3;
	timer->delta2 = delta2; 
	
	delta = MIN(MIN(delta, delta2), delta3);
	delta &= (1<<9)-1; /* limit the maximum entropy used */
	
	p->count += delta;
	if (p->count > POOLBITS)
		p->count = POOLBITS;
}

/*
 * read bits out of the pool and substract from entropy count.
 */
static int read_pool(struct pool *p, char * buf, int n)
{
	int ret, i;
	ulong tmp[4];
	char *cp,*dp;

	D(("reading %d bytes from pool\n", n));

	timed_entropy(p, &timer, n);
	
	ret = n;
	if (p->count / 8 >= n)
		p->count -= n*8;
	else
		p->count = 0;
		
	D(("left %d bits in the pool\n", p->count));

	do
	{
		/* Hash the pool to get the output */
		tmp[0] = 0x67452301;
		tmp[1] = 0xefcdab89;
		tmp[2] = 0x98badcfe;
		tmp[3] = 0x10325476;
		for (i = 0; i < POOLSIZE; i += 16)
			MD5Transform(tmp, p->words+i);

		/* stir up pool status with its own MD5 */
		add_word_to_pool(p, tmp[0]);
		add_word_to_pool(p, tmp[1]);
		add_word_to_pool(p, tmp[2]);
		add_word_to_pool(p, tmp[3]);

		/* obscure the previous */
		MD5Transform(tmp, p->words);

		/* Copy data to destination buffer */
		i = MIN(n, sizeof(tmp));
		memcpy(buf, (const char *)tmp, i);
		n -= i;
		buf += i;

		timed_entropy(p, &timer, n);
	}
	while (n);

	/* Wipe data from stack */
	memset(tmp, 0, sizeof(tmp));
	
	return ret;
}

/* ----------
	random_open - called on each open().
----- */

static long
random_open (device_info *dev, ulong flags)
{
	D(("someone opened %s\n", dev ? dev->entry->name : "random"));

	timed_entropy(&pool, &timer, flags);
	
	return B_NO_ERROR;
}


/* ----------
	random_close
----- */

static long
random_close (device_info *dev)
{
	D(("%s closed\n", dev ? dev->entry->name : "random"));

//	timed_entropy(&pool, &timer, 0);

	return B_NO_ERROR;
}


/* ----------
	random_read - read some stuff from our device.  
	This version tries to make sure the bits returned are truely random.
	Since there's no way to get input in, this is likely to block infinitely
	_often_ unless something else uses urandom at the same time.
----- */

static long
random_read (device_info *dev, void *buf, ulong count, ulong pos)
{
	int	retval = 0;
	int	c = 0;
	char *b = (char *)buf;
	
//	D(("random_read %x %x %d %d\n",dev,buf,count,pos));
	
	while (count > 0) 
	{
		int n = count;
		if (n > pool.count / 8)
			n = pool.count / 8;
		if (n == 0) 
			break;
		n = read_pool(&pool, b, n);
		if (n < 0)
		{
			if (count == 0)
				retval = n;
			break;
		}
		c += n;
		b += n;
		count -= n;
		break;		/* This break makes the device work */
					/* like a named pipe */
	}

	return (c ? c : retval);
}

/* ----------
	random_read - read some stuff from our device.  
	This version returns (hopefully) strongly random bits, but does not
	care to make sure if they're truly random (ie whether the pool contains
	real environmental data).
----- */

static long
random_read_u (device_info *dev, void *buf, ulong count, ulong pos)
{
//	D(("random_read_u %x %x %d %d\n",dev,buf,count,pos));
	
	return read_pool(&pool, buf, count);
}


/* ----------
	random_write - write some stuff to our device.  We ignore the 'pos'
	argument, as the skel is not a positionable device.
----- */

static long
random_write (device_info *dev, void *buf, ulong count, ulong pos)
{
	int i;
	ulong *p;

	if (count < 0)
		return -1;

//	D(("random_write %x %x %d %d\n",dev,buf,count,pos));

	timed_entropy(&pool, &timer, count);

	for (i = count, p = (ulong *)buf;
	     i >= sizeof(ulong);
	     i-= sizeof(ulong), p++) 
	{
		add_word_to_pool(&pool, *p);
	}
	if (i) 
	{
		ulong word = 0;
		memcpy(&word, p, i);
		add_word_to_pool(&pool, word);
	}
	pool.count += count*8;
	
	if (pool.count > POOLBITS)
		pool.count = POOLBITS;

	return count;
}


/* ----------
	random_control
----- */

static long
random_control (device_info *dev, ulong msg, void *buf)
{
	switch (msg) {
	
	case R_ZAP_COUNT:
		pool.count = 0;
		break;

	default:
		return B_ERROR;
	}

	return B_NO_ERROR;
}

/* ----------
	init_driver

	Called when driver is loaded.  One-time initializations go here.
----- */
long
init_driver (void)
{
	ulong *p;
	int i;
	double t; 
	ulong ts, tu;
	system_info info;

	D(("random device: %s %s\n", __DATE__, __TIME__));

	/* this jumping around with doubles is annoying. Almost makes me
	 * want to snatch the time info from the timebase register using
	 * assembly. Grr. Anyways, add current time to entropy pool! */
	t = system_time()*1e-6; ts = (ulong)t; tu = (ulong)((t-ts)*1e6);
	D(("init data: system_time: %d, %d\n", ts, tu));
	add_word_to_pool(&pool, ts);
	add_word_to_pool(&pool, tu);

	get_system_info(&info);
	for (p = (ulong *) &info, i = sizeof(info) / sizeof(ulong); i ; )
	{
		add_word_to_pool(&pool, p[--i]);
	}

	D(("random device initialized\n"));
	
	return B_NO_ERROR;
}

long
uninit_driver(void)
{
	return B_NO_ERROR;
}

/* -----
	driver-related structures
----- */

extern device_entry	devices[] = {
	{
		"/dev/random",	/* name */
		random_open, 		/* -> open entry point */
		random_close, 		/* -> close entry point */
		random_control, 	/* -> control entry point */
		random_read,		/* -> read entry point */
		random_write		/* -> write entry point */
	},
	{
		"/dev/urandom",	/* name */
		random_open, 		/* -> open entry point */
		random_close, 		/* -> close entry point */
		random_control, 	/* -> control entry point */
		random_read_u,		/* -> read entry point */
		random_write		/* -> write entry point */
	},
	0
};
