#!/usr/local/bin/perl
#
# perfbrowse.perl -- CGI browser for PERFORCE
#
#
# Point P4PORT and P4CLIENT differently if you're not using the defaults.
# PATH is set to pick up the Perforce client program.
#

#$ENV{P4PORT} = "p4:1666";
$ENV{P4CLIENT} = "p4browse";
$ENV{PATH} .= ":/usr/local/bin"; 

# Boilerplate

$myname = "$0";
$myname =~ s!^.*/!!;

$BLUE = qq(<font color="#0000FF">);
$GREEN = qq(<font color="#00B000">);
$RED = qq(<font color="#B00000">);
$END = qq(</font>);

$ADD = $BLUE;
$ADDEND = $END;
$DEL = "<STRIKE>$RED";
$DELEND = "$END</STRIKE>";

$MAXCONTEXT = 30;
$NCONTEXT = 10;

@HTMLHEADER = (
	"Content-type: text/html\n",
	"\n",
	"<html>\n",
	"<head>\n",
	"<body>\n" );

@HTMLERROR = (
	"Content-type: text/html\n",
	"\n",
	"<html>\n",
	"<head>\n",
	"<body>\n" );

#
# Switch on ARGV[0]
#

# handle isindex compatibility

unshift( @ARGV, "\@changes" ) if( @ARGV && @ARGV[0] !~ m!^@! );

################################
#
# No arguments. 
#
#	Put up the introductory screen.
#
################################

if (!@ARGV) {

    # Default codelines data is just a simple list of everything.
    # If $CODELINES is set in the environment, the codelines comes
    # from that.  The format is:
    #
    #	description1
    #	//path1
    #	description2
    #	//path2

    @CODELINES = (
	    "Full Depot Source\n",
	    "//depot/...\n" );

    if ($ENV{CODELINES}) {
	open( P4, "$ENV{CODELINES}" ) || &bail( "No codelines file." );
	@CODELINES = <P4>;
    }

    print 
	@HTMLHEADER,
	"<title>Perforce Change Browser</title>\n",
	"<center><h1>Perforce Change Browser</h1>\n</center>",
	"<i>This browser allows you to view the history of a Perforce depot.\n",
	"The first step is to select the files you want history for.</i>\n",
	"<isindex prompt=\"Click below or type a file pattern here: \">\n";

    foreach ( @CODELINES )
    {
	chop;

	if( m:^/: )
	{
	    print "<li>", &url( "\@changes+$_", $_ ), "\n";
	}
	elsif( !/^\s*$/ )
	{
	    print "</blockquote><p><h3>$_</h3><blockquote>\n";
	}
    }
}

################################
#
# changes [ path ]
#
#	show changes for path
#
################################

elsif( $ARGV[0] eq "\@changes" ) {

    &p4open( 'P4', "changes -l $ARGV[1]|" );

    print 
	@HTMLHEADER,
	"<title>Changes for $ARGV[1]</title>\n",
	"<center><h1>Changes for $ARGV[1]</h1></center>\n",
	"<i>This form displays the changes for the files you've selected.\n",
	"Click on the change number to see details of a change.  Changes\n",
	"are listed in reverse chronological order, so you see what's\n",
	"most recent first.</i>\n",
	"<hr><dl>\n";

    while (<P4>) {

	if( local( $change, $misc ) = /^Change (\d+) (.*)$/ ) 
	{
	    print 
		"<dt>", &url( "\@describe+$change", "Change $change" ), 
		" $misc<dd>\n";
	} 
	else 
	{
	    chop;
	    print "<tt>$_</tt><br>\n";
	}
    }

    print "</dl>\n";
}

################################
#
# describe change
#
#	describe a change
#
################################

elsif( $ARGV[0] eq "\@describe" ) {

    &p4open( 'P4', "describe -s $ARGV[1]|" );

    $_ = <P4>;

    ( local($chn, $user, $client, $date, $time) = 
	/^Change (\d+) by (\S*)@(\S*) on (\S*) (\S*)$/ )
	|| &bail( $_ );

    print 
	@HTMLHEADER,
	"<title>Change $chn</title>\n",
	"<center><h1>Change $chn</h1></center>\n",
	"<i>This form displays the details of a change.  For each of the\n",
	"files affected, you can click on:\n",
	"<ul>\n",
	"<li>Filename -- to see the complete file history\n",
	"<li>Revision Number -- to see the file text\n",
	"<li>Action -- to see the deltas (diffs)\n",
	"</ul></i>",
	"<hr><pre>\n",
	"<strong>Author        </strong>$user\n",
	"<strong>Client        </strong>$client\n",
	"<strong>Date          </strong>$time $date\n",
	"</pre><hr>\n",
	"<h2>Description</h2>\n",
	"<pre>\n";

    while(<P4>) {
	next if /^\s*$/;
	last if /^Affected files/;
	print $_;
    }

    print
	"</pre>",
	"<hr>\n",
	"<h2>Files</h2>\n",
	"<ul>\n",
	"<table cellpadding=1>",
	"<tr align=left><th>File<th>Rev<th>Action</tr>\n";

    # Sample:
    # ... //depot/main/p4/Jamrules#71 edit

    while(<P4>) {

	if( local( $file, $rev, $act ) = /^\.\.\. (\S*)#(\d*) (\S*)$/ )
	{
	    print 
		"<tr>",
		"<td>", &url( "\@filelog+$file", "$file" ),
		"<td>", &url( "\@print+$file+$rev", "$rev" ),
		"<td>", &url( "\@diff+$file+$rev+$act", "$act" ),
		"</tr>\n";
	}
    }

    print
	"</table></ul>\n";
}

################################
#
# filelog file 
#
#	show filelog of the file
#
################################

elsif ($ARGV[0] eq "\@filelog") {

    local( $name ) = $ARGV[1];

    &p4open( 'P4', "filelog $name|" );

    $name = <P4>;
    chop $name;

    print 
	@HTMLHEADER,
	"<title>Filelog $name</title>\n",
	"<center><h1>Filelog $name</h1></center>\n",
	"<i>This form shows the history of an individual file across\n",
	"changes.  You can click on the following:\n",
	"<ul>\n",
	"<li>Revision Number -- to see the file text\n",
	"<li>Action -- to see the deltas (diffs)\n",
	"<li>Change -- to see the complete change description, including\n",
	"other files.\n",
	"</ul></i>",
	"<hr>\n";

    print
	"<table cellpadding=1>",
	"<tr align=left><th>Rev<th>Action<th>Date",
	"<th>User<th>Change<th>Desc</tr>\n";

    # Sample:
    # ... #78 change 1477 edit on 04/18/1996 by user@client 'Fix NT mkdi'

    while( <P4> ) {

	if( local( $rev, $change, $act, $date, $user, $client, $desc ) =
	    /^\.\.\. \#(\d+) \S+ (\d+) (\S+) on (\S+) by (\S*)@(\S*) '(.*)'/ )
	{
	    print
		"<tr>",
		"<td>", &url( "\@print+$name+$rev", "$rev" ),
		"<td>", &url( "\@diff+$name+$rev+$act", $act ),
		"<td>$date",
		"<td>$user@$client",
		"<td>", &url( "\@describe+$change", "$change" ),
		"<td><tt>$desc</tt>",
		"</tr>\n";
	}
    }

    print "</table>\n";
}

################################
#
# print file rev action
#
#	print file text
#
################################

elsif ($ARGV[0] eq "\@print") {

    local($name, $rev) = @ARGV[1..2];

    &p4open( 'P4', "print $name#$rev|" );

    # Get header line
    # //depot/main/jam/Jamfile#39 - edit change 1749 (text)

    $_ = <P4>;
    local( $name, $rev, $type ) = m!^(\S+)\#(\d+) - \S+ \S+ \S+ \((\w+)\)!;

    print 
	@HTMLHEADER,
	"<title>File $name</title>\n",
	"<center><h1>File $name#$rev</h1></center>\n",
	"<i>This form shows you the raw contents of a file, as long as \n",
	"it isn't binary.</i>",
	"<hr>\n";

    if( $type eq "binary" || $type eq "xbinary" )
    {
	print "<h2>$type</h2>\n";
    }
    else
    {
	print "<pre>\n";

	while( <P4> ) {
	    print $_;
	}

	print "</pre>\n";
    }
}

################################
#
# diff file rev action
#
#	describe a change
#
################################

elsif ($ARGV[0] eq "\@diff") {

    local( $name, $rev, $mode ) = @ARGV[1..3];
    local( $nchunk ) = 0;

    print
	@HTMLHEADER,
	"<title>$name#$rev</title>\n",
	"<center><h1>$name#$rev - $mode</h1></center>\n",
	"<i>This form shows you the deltas (diffs) that lead from the\n",
	"previous to the current revision.</i>\n",
	"<hr>\n";

    if ($mode ne 'add' && $mode ne 'delete' && $mode ne 'branch') {

	local($f1) = "$name#" . ($rev - 1);
	local($f2) = "$name#" . ($rev);

	&p4open('P4', "diff2 $f1 $f2|");
	$_ = <P4>; 

	while (<P4>) {

	    local( $dels, $adds );

	    local( $la, $lb, $op, $ra, $rb ) = 
		/(\d+),?(\d*)([acd])(\d+),?(\d*)/;

	    next unless $ra;

	    if( !$lb ) { $lb = $op ne 'a' ? $la : $la - 1; }
	    if( !$rb ) { $rb = $op ne 'd' ? $ra : $ra - 1; }

	    $start[ $nchunk ] = $op ne 'd' ? $ra : $la;
	    $dels[ $nchunk ] = $dels = $lb - $la + 1;
	    $adds[ $nchunk ] = $adds = $rb - $ra + 1;
	    @lines[ $nchunk ] = ();

	    # deletes

	    while( $dels-- ) {
	    	$_ = <P4>; 	
		s/^. //;
		if (/[&<>]/) {
		    s/&/\&amp;/g;
		    s/</\&lt;/g;
		    s/>/\&gt;/g;
		}
		@lines[ $nchunk ] .= $_;
	    }
	    
	    # separator

	    if ($op eq 'c') {	
		$_ = <P4>; 
	    }

	    # adds

	    while( $adds-- ) {
		$_ = <P4>;
	    }

	    $nchunk++;
	}
    }
   
    # Now walk through the diff chunks, reading the current rev and
    # displaying it as necessary.

    print 
    	"<center><pre>",
    	"$ADD added lines $ADDEND\n",
	"$DEL deleted lines $DELEND\n",
	"</pre></center><hr><pre>\n";

    local( $curlin ) = 1;

    &p4open('P4', "print -q $name#$rev|");

    for( $n = 0; $n < $nchunk; $n++ )
    {
	# print up to this chunk.

	&catchup( 'P4', $start[ $n ] - $curlin );

	# display deleted lines -- we saved these from the diff

	if( $dels[ $n ] )
	{
		print "$DEL";
		print @lines[ $n ];
		print "$DELEND";
	}

	# display added lines -- these are in the file stream.

	if( $adds[ $n ] )
	{
		print "$ADD";
		&display( 'P4', $adds[ $n ] );
		print "$ADDEND";
	}

	$curlin = $start[ $n ] + $adds[ $n ];
    }

    &catchup( 'P4', 999999999 );
} 


################################
#
# None of the above.
#
################################

else {
	&bail( "Invalid invocation @ARGV" );
}

# Trailer

print
	"</body>\n";

##################################################################
##################################################################
#
# Subroutines.
#
##################################################################
##################################################################

sub url {
	local( $url, $name ) = @_;
	return qq(<a HREF="$myname?$url">$name</a>) ;
}

sub bail {
	print @HTMLERROR, @_, "\n";
	die @_;
}

sub p4open {
	local( $handle, @command ) = @_;
	open( $handle, "p4 @command" ) || &bail( "p4 @command failed" );
}

# Support for processing diff chunks.
#
# skip: skip lines in source file
# display: display lines in source file, handling funny chars 
# catchup: display & skip as necessary
#

sub skip {
	local( $handle, $to ) = @_;

	while( $to > 0 && ( $_ = <$handle> ) ) {
	    $to--;
	}

	return $to;
}

sub display {
	local( $handle, $to ) = @_;

	while( $to-- > 0 && ( $_ = <$handle> ) ) {

	    if (/[&<>]/) 
	    {
		s/&/\&amp;/g;
		s/</\&lt;/g;
		s/>/\&gt;/g;
	    }
	    print $_;
	}
}

sub catchup {

	local( $handle, $to ) = @_;

	if( $to > $MAXCONTEXT )
	{
	    local( $skipped ) = $to - $NCONTEXT * 2;

	    &display( $handle, $NCONTEXT );
	    
	    $skipped -= &skip( $handle, $skipped );

	    print 
		"<hr><center><strong>",
		"$skipped lines skipped",
		"</strong></center><hr>\n" if( $skipped );

	    &display( $handle, $NCONTEXT );
	}
	else
	{
	    &display;
	}
}

