// ============================================================
//  STEngine.overrides.cpp	1996 Hiroshi Lockheimer
// ============================================================
// 	STE Version 1.0a4

#include "STEngine.h"
#include <limits.h>


// ------------------------------------------------------------
// 	 AttachedToWindow
// ------------------------------------------------------------
// Reset some member variables, recalculate the line breaks, 
// redraw the text

void
STEngine::AttachedToWindow()
{
	BView::AttachedToWindow();

	Window()->SetPulseRate(500000.0);

	mCaretVisible = FALSE;
	mCaretTime = 0.0;
	mClickCount = 0;
	mClickTime = 0.0;
	mDragOffset = -1;
	mDragOwner = FALSE;
	mActive = FALSE;

	Refresh(0, LONG_MAX, TRUE, FALSE);
}


// ------------------------------------------------------------
// 	 Draw
// ------------------------------------------------------------
// Draw any lines that need to be updated and display the 
// caret or the current selection

void
STEngine::Draw(
	BRect	inRect)
{
	// what lines need to be drawn?
	long startLine = PixelToLine(inRect.top);
	long endLine = PixelToLine(inRect.bottom);

	DrawLines(startLine, endLine);
	
	// draw the caret/hilite the selection
	if (mActive) {
		if (mSelStart != mSelEnd)
			DrawSelection(mSelStart, mSelEnd);
		else {
			if (mCaretVisible)
				DrawCaret(mSelStart);
		}
	}
}


// ------------------------------------------------------------
// 	 MouseDown
// ------------------------------------------------------------
// Move the caret and track the mouse while it's down

void
STEngine::MouseDown(
	BPoint	where)
{
	// should we even bother?
	if ((!mEditable) && (!mSelectable))
		return;
	
	// if this view isn't the focus, make it the focus and return
	if (!IsFocus()) {
		MakeFocus();
		return;
	}
	
	// hide the caret if it's visible	
	if (mCaretVisible)
		InvertCaret();
	
	long 	mouseOffset = PointToOffset(where);	
	bool	shiftDown = modifiers() & B_SHIFT_KEY;

	// should we initiate a drag?
	if ((mSelStart != mSelEnd) && (!shiftDown)) {
		BPoint	loc;
		ulong	buttons;
		GetMouse(&loc, &buttons);
		// was the secondary button clicked?
		if (buttons == B_SECONDARY_MOUSE_BUTTON) {
			// was the click within the selection range?
			if ((mouseOffset >= mSelStart) && (mouseOffset <= mSelEnd)) {
				InitiateDrag();
				return;
			}
		}
	}
	
	// get the system-wide click speed
	double clickSpeed = 0.0;
	get_click_speed(&clickSpeed);
	
	// is this a double/triple click, or is it a new click?
	if ( (clickSpeed > (system_time() - mClickTime)) &&
		 (mouseOffset == mClickOffset) ) {
		if (mClickCount > 1) {
			// triple click
			mClickCount = 0;
			mClickTime = 0.0;
		}
		else {
			// double click
			mClickCount = 2;
			mClickTime = system_time();
		}
	}
	else {
		// new click
		mClickOffset = mouseOffset;
		mClickCount = 1;
		mClickTime = system_time();
	
		if (!shiftDown)
			Select(mouseOffset, mouseOffset);
	}
	
	// no need to track the mouse if we can't select
	if (!mSelectable)
		return;
		
	// track the mouse while it's down
	long		start = 0;
	long		end = 0;
	long		anchor = (mouseOffset > mSelStart) ? mSelStart : mSelEnd;
	BPoint		curMouse = where;
	ulong		buttons = 0;
	BScrollBar	*hScroll = ScrollBar(B_HORIZONTAL);
	BScrollBar	*vScroll = ScrollBar(B_VERTICAL);
	do {
		if (mouseOffset > anchor) {
			start = anchor;
			end = mouseOffset;
		}
		else {
			start = mouseOffset;
			end = anchor;
		}
		
		switch (mClickCount) {
			case 0:
				// triple click, select line by line
				start = mLines[OffsetToLine(start)]->offset;
				end = (mLines[OffsetToLine(end)] + 1)->offset;
				break;
												
			case 2:
			{
				// double click, select word by word
				long anOffset = 0;
				FindWord(start, &start, &anOffset);
				FindWord(end, &anOffset, &end);
				
				break;
			}
				
			default:
				// new click, select char by char
				break;			
		}
		if (shiftDown) {
			if (mouseOffset > anchor)
				start = anchor;
			else
				end = anchor;
		}
		Select(start, end);
		
		// Should we scroll the view?
		BRect bounds = Bounds();
		if (!bounds.Contains(curMouse)) {	
			long hDelta = 0;
			long vDelta = 0;
			
			if (hScroll != NULL) {
				if (curMouse.x < bounds.left)
					hDelta = curMouse.x - bounds.left;
				else {
					if (curMouse.x > bounds.right)
						hDelta = curMouse.x - bounds.right;
				}
				
				if (hDelta != 0)
					hScroll->SetValue(hScroll->Value() + hDelta);
			}
			
			if (vScroll != NULL) {
				if (curMouse.y < bounds.top)
					vDelta = curMouse.y - bounds.top;
				else {
					if (curMouse.y > bounds.bottom)
						vDelta = curMouse.y - bounds.bottom;
				}
				
				if (vDelta != 0)
					vScroll->SetValue(vScroll->Value() + vDelta);
			}
			
			if ((hDelta != 0) || (vDelta != 0))
				Window()->UpdateIfNeeded();
		}
		
		// Zzzz...
		snooze(30000.0);
		
		GetMouse(&curMouse, &buttons);
		mouseOffset = PointToOffset(curMouse);		
	} while (buttons != 0);
}


// ------------------------------------------------------------
// 	 MouseMoved
// ------------------------------------------------------------
// Set the cursor to the I-Beam when it's above this view and
// track any drags that are over this view

void
STEngine::MouseMoved(
	BPoint		where,
	ulong		code, 
	BMessage	*message)
{
	switch (code) {
		case B_ENTERED_VIEW:
			if ((mActive) && (message == NULL))
				be_app->SetCursor(B_I_BEAM_CURSOR);
			break;
			
		case B_INSIDE_VIEW:
			if ((mEditable) && (message != NULL)) {
				if ((mDragOwner) || (CanDrop(message)))
					TrackDrag(where);
			}
			break;
			
		case B_EXITED_VIEW:
			DragCaret(-1);
			if (mActive)
				be_app->SetCursor(B_HAND_CURSOR);
			break;
			
		default:
			BView::MouseMoved(where, code, message);
			break;
	}
}


// ------------------------------------------------------------
// 	 WindowActivated
// ------------------------------------------------------------
// Activate this view if and only if its window is active and
// the view is the focus of that window

void
STEngine::WindowActivated(
	bool	state)
{
	BView::WindowActivated(state);
	
	if (state && IsFocus()) {
		if (!mActive)
			Activate();
	}
	else {
		if (mActive)
			Deactivate();
	} 
}


// ------------------------------------------------------------
// 	 KeyDown
// ------------------------------------------------------------
// Respond to key presses

void
STEngine::KeyDown(
	ulong	inKey)
{
	if (!mEditable)
		return;
		
	// hide the cursor and caret
	be_app->ObscureCursor();
	if (mCaretVisible)
		InvertCaret();
	
	switch (inKey) {
		case B_BACKSPACE:
			HandleBackspace();
			break;
			
		case B_LEFT_ARROW:
		case B_RIGHT_ARROW:
		case B_UP_ARROW:
		case B_DOWN_ARROW:
			HandleArrowKey(inKey);
			break;
		
		case B_DELETE:
			HandleDelete();
			break;
			
		case B_HOME:
		case B_END:
		case B_PAGE_UP:
		case B_PAGE_DOWN:
			HandlePageKey(inKey);
			break;
			
		case B_ESCAPE:
		case B_INSERT:
		case B_FUNCTION_KEY:
			// ignore, pass key up to superclass
			BView::KeyDown(inKey);
			break;
			
		default:
			HandleAlphaKey(inKey);
			break;
	}

	// draw the caret
	if (mSelStart == mSelEnd) {
		if (!mCaretVisible)
			InvertCaret();
	}
}


// ------------------------------------------------------------
// 	 Pulse
// ------------------------------------------------------------
// Flash the caret at 1/2 second intervals and track drags

void
STEngine::Pulse()
{	
	if ((mActive) && (mEditable) && (mSelStart == mSelEnd)) {
		if (system_time() > (mCaretTime + 500000.0))
			InvertCaret();
	}
	
	if (mDragOwner) {
		BPoint	where;
		ulong 	buttons;
		GetMouse(&where, &buttons);
		
		if (buttons == 0) {
			// our drag must has been dropped elsewhere
			mDragOwner = FALSE;
			Window()->SetPulseRate(500000.0);
		}
		else {
			if (mEditable)
				// assume that we know how to handle this drag 
				TrackDrag(where);
		}
	}
}


// ------------------------------------------------------------
// 	 FrameResized
// ------------------------------------------------------------
// Update the scroll bars to mirror the visible area

void
STEngine::FrameResized(
	float	width,
	float 	height)
{
	BView::FrameResized(width, height);

	UpdateScrollbars();
}


// ------------------------------------------------------------
// 	 MakeFocus
// ------------------------------------------------------------
// Activate this view if and only if its window is active and
// the view is the focus of that window

void
STEngine::MakeFocus(
	bool	focusState)
{
	BView::MakeFocus(focusState);
	
	if (focusState && Window()->IsActive()) {
		if (!mActive)
			Activate();
	}
	else {
		if (mActive)
			Deactivate();
	} 
}


// ------------------------------------------------------------
// 	 SetFontRotation
// ------------------------------------------------------------
// Gimme a break!
//
// Font rotation is disabled in STE

void
STEngine::SetFontRotation(
	float	degrees)
{
#pragma unused (degrees)

	// don't do anything
}


// ------------------------------------------------------------
// 	 MessageReceived
// ------------------------------------------------------------
// Check for dropped messages and respond to the standard 
// Cut, Copy, and Paste messages

void
STEngine::MessageReceived(
	BMessage	*message)
{
	// was this message dropped?
	if (message->WasDropped()) {
		BPoint dropLoc;
		BPoint offset;
		
		dropLoc = message->DropPoint(&offset);
		ConvertFromScreen(&dropLoc);
		ConvertFromScreen(&offset);

		if (!MessageDropped(message, dropLoc, offset))
			BView::MessageReceived(message);
		
		return;
	}

	switch (message->what) {
		case B_CUT:
			Cut();
			break;
			
		case B_COPY:
			Copy();
			break;
			
		case B_PASTE:
			Paste();
			break;
			
		default:
			BView::MessageReceived(message);
			break;
	}
}
