// ============================================================
//  STEngine.cpp	1996 Hiroshi Lockheimer
// ============================================================
// 	STE Version 1.0a4

#include "STEngine.h"
#include <limits.h>


// === Static Member Variables ===

STEWidthBuffer	STEngine::sWidths;


// ------------------------------------------------------------
// 	 STEngine
// ------------------------------------------------------------
// Constructor
//
// textRect is in local coordinates
// nullStyle is the default style, all fields must be valid
//
// Initially the text is selectable, editable, and will wrap
// The initial tab width is that of four spaces in 'Kate'
// An offscreen bitmap is not used by default

STEngine::STEngine(
	BRect				frame,
	const char			*name,
	BRect				textRect,
	ConstSTEStylePtr	nullStyle,
	ulong 				resizeMask, 
	ulong				flags)
		: BView(frame, name, resizeMask, flags),
		  mStyles(nullStyle)
{
	mTextRect = textRect;
	mSelStart = 0;
	mSelEnd = 0;
	mCaretVisible = FALSE;
	mCaretTime = 0.0;
	mClickOffset = -1;
	mClickCount = 0;
	mClickTime = 0.0;
	mDragOffset = -1;
	mDragOwner = FALSE;
	mActive = FALSE;
	mTabWidth = 28.0;
	mSelectable = TRUE;
	mEditable = TRUE;
	mWrap = TRUE;
	mOffscreen = NULL;
}


// ------------------------------------------------------------
// 	 ~STEngine
// ------------------------------------------------------------
// Destructor

STEngine::~STEngine()
{
	delete (mOffscreen);
}


// ------------------------------------------------------------
// 	 SetText
// ------------------------------------------------------------
// Replace the current text with inText
//
// Pass NULL for inStyles if there is no style data

void
STEngine::SetText(
	const char				*inText,
	long					inLength,
	ConstSTEStyleRangePtr	inStyles)
{	
	// hide the caret/unhilite the selection
	if (mActive) {
		if (mSelStart != mSelEnd)
			DrawSelection(mSelStart, mSelEnd);
		else {
			if (mCaretVisible)
				InvertCaret();
		}
	}
	
	// remove data from buffer
	if (mText.Length() > 0)
		RemoveRange(0, mText.Length());
		
	InsertAt(inText, inLength, 0, inStyles);
	
	mSelStart = mSelEnd = 0;	

	// recalc line breaks and draw the text
	Refresh(0, inLength, TRUE, TRUE);
	
	HandleModification();

	// draw the caret
	if (mActive) {
		if (!mCaretVisible)
			InvertCaret();
	}
}


// ------------------------------------------------------------
// 	 Insert
// ------------------------------------------------------------
// Copy inLength bytes from inText and insert it at the caret
// position, or at the beginning of the selection range
// 
// Pass NULL for inStyles if there is no style data
//
// The caret/selection will move with the insertion

void
STEngine::Insert(
	const char				*inText,
	long					inLength,
	ConstSTEStyleRangePtr	inStyles)
{
	// do we really need to do anything?
	if (inLength < 1)
		return;
	
	// hide the caret/unhilite the selection
	if (mActive) {
		if (mSelStart != mSelEnd)
			DrawSelection(mSelStart, mSelEnd);
		else {
			if (mCaretVisible)
				InvertCaret();
		}
	}
	
	// copy data into buffer
	InsertAt(inText, inLength, mSelStart, inStyles);

	// offset the caret/selection
	long saveStart = mSelStart;
	mSelStart += inLength;
	mSelEnd += inLength;

	// recalc line breaks and draw the text
	Refresh(saveStart, mSelStart, TRUE, TRUE);
	
	HandleModification();

	// draw the caret/hilite the selection
	if (mActive) {
		if (mSelStart != mSelEnd)
			DrawSelection(mSelStart, mSelEnd);
		else {
			if (!mCaretVisible)
				InvertCaret();
		}
	}
}


// ------------------------------------------------------------
// 	 Delete
// ------------------------------------------------------------
// Delete the current selection

void
STEngine::Delete()
{
	// anything to delete?
	if (mSelStart == mSelEnd)
		return;
		
	// hide the caret/unhilite the selection
	if (mActive) {
		if (mSelStart != mSelEnd)
			DrawSelection(mSelStart, mSelEnd);
		else {
			if (mCaretVisible)
				InvertCaret();
		}
	}
	
	// remove data from buffer
	RemoveRange(mSelStart, mSelEnd);
	
	// collapse the selection
	mSelEnd = mSelStart;
	
	// recalc line breaks and draw what's left
	Refresh(mSelStart, mSelEnd, TRUE, TRUE);
	
	HandleModification();
	
	// draw the caret
	if (mActive) {
		if (!mCaretVisible)
			InvertCaret();
	}
}


// ------------------------------------------------------------
// 	 Text
// ------------------------------------------------------------
// Return a pointer to the text, NULL if error
//
// Do not free the returned text or alter it in any way
//
// The pointer that is returned may become invalid after a
// subsequent call to any other STEngine function
// Copy it into your own buffer or use STEngine::GetText() if 
// you need the text for an extended period of time
//
// The text is null terminated

const char*
STEngine::Text()
{
	return (mText.Text());
}


// ------------------------------------------------------------
// 	 TextLength
// ------------------------------------------------------------
// Return the length of the text buffer
//
// The length does not include the null terminator

long
STEngine::TextLength() const
{
	return (mText.Length());
}


// ------------------------------------------------------------
// 	 GetText
// ------------------------------------------------------------
// Copy into buffer up to length characters of the text 
// starting at offset
//
// The text is null terminated

void
STEngine::GetText(
	char	*buffer,
	long 	offset,
	long 	length) const
{
	long textLen = mText.Length();
	
	if ((offset < 0) || (offset > (textLen - 1))) {
		buffer[0] = '\0';
		return;
	}
		
	length = ((offset + length) > textLen) ? textLen - offset : length;
	for (long i = 0; i < length; i++)
		buffer[i] = mText[offset + i];
	buffer[length] = '\0';
}


// ------------------------------------------------------------
// 	 CharAt
// ------------------------------------------------------------
// Return the character at offset

char
STEngine::CharAt(
	long	offset) const
{
	if ((offset < 0) || (offset > (mText.Length() - 1)))
		return ('\0');
		
	return (mText[offset]);
}


// ------------------------------------------------------------
// 	 CountLines
// ------------------------------------------------------------
// Return the number of lines

long
STEngine::CountLines() const
{
	long numLines = mLines.NumLines();
	
	long textLength = mText.Length();
	if (textLength > 0) {
		// add a line if the last character is a newline
		if (mText[textLength - 1] == '\n')
			numLines++;
	}
	
	return (numLines);
}


// ------------------------------------------------------------
// 	 CurrentLine
// ------------------------------------------------------------
// Return the line number of the caret or the beginning of
// the selection
//
// Line numbers start at 0

long
STEngine::CurrentLine() const
{
	long lineNum = OffsetToLine(mSelStart);
	
	long textLength = mText.Length();
	if ((textLength > 0) && (mSelStart == textLength)) {
		// add a line if the last character is a newline
		if (mText[textLength - 1] == '\n')
			lineNum++;
	}
	
	return (lineNum);
}


// ------------------------------------------------------------
// 	 GoToLine
// ------------------------------------------------------------
// Move the caret to the beginning of lineNum
//
// This function does not automatically scroll the view, 
// use ScrollToSelection() if you want to ensure that lineNum 
// is visible
//
// Line numbers start at 0

void
STEngine::GoToLine(
	long	lineNum)
{
	// hide the caret/unhilite the selection
	if (mActive) {
		if (mSelStart != mSelEnd)
			DrawSelection(mSelStart, mSelEnd);
		else {
			if (mCaretVisible)
				InvertCaret();
		}
	}
	
	long saveLineNum = lineNum;
	long maxLine = mLines.NumLines() - 1;
	lineNum = (lineNum > maxLine) ? maxLine : lineNum;
	lineNum = (lineNum < 0) ? 0 : lineNum;
	
	mSelStart = mSelEnd = mLines[lineNum]->offset;
	
	long textLength = mText.Length();
	if ((textLength > 0) && (saveLineNum > maxLine)) {
		// add a line if the last character is a newline
		if (mText[textLength - 1] == '\n')
			mSelStart = mSelEnd = textLength;
	}
	
	InvertCaret();
}


// ------------------------------------------------------------
// 	 Cut
// ------------------------------------------------------------
// Copy the current selection into the clipboard and delete it
// from the buffer

void
STEngine::Cut()
{
	be_clipboard->Lock();
	be_clipboard->Clear();
	
	// add text
	be_clipboard->AddData(B_ASCII_TYPE, mText.Text() + mSelStart, 
						  mSelEnd - mSelStart);
	
	// add corresponding styles
	long 				length = 0;
	STEStyleRangePtr	styles = GetStyleRange(mSelStart, mSelEnd, &length);			
	
	// reset the extra fields, override if you don't like this behavior
	for (long i = 0; i < styles->count; i++)
		styles->runs[i].style.extra = 0;
	
	be_clipboard->AddData(STE_STYLE_TYPE, styles, length);
	
	free(styles);
	
	be_clipboard->Commit();
	be_clipboard->Unlock();
	
	Delete();
}


// ------------------------------------------------------------
// 	 Copy
// ------------------------------------------------------------
// Copy the current selection into the clipboard
//
// The extra field in the clipboard will be reset to 0

void
STEngine::Copy()
{
	be_clipboard->Lock();
	be_clipboard->Clear();
	
	// add text
	be_clipboard->AddData(B_ASCII_TYPE, mText.Text() + mSelStart, 
						  mSelEnd - mSelStart);
	
	// add corresponding styles
	long 				length = 0;
	STEStyleRangePtr	styles = GetStyleRange(mSelStart, mSelEnd, &length);			
	
	// reset the extra fields, override if you don't like this behavior
	for (long i = 0; i < styles->count; i++)
		styles->runs[i].style.extra = 0;
	
	be_clipboard->AddData(STE_STYLE_TYPE, styles, length);
	
	free(styles);
	
	be_clipboard->Commit();
	be_clipboard->Unlock();
}


// ------------------------------------------------------------
// 	 Paste
// ------------------------------------------------------------
// Copy the contents of the clipboard into the text buffer

void
STEngine::Paste()
{
	// do we really want what's in the clipboard?
	if (!CanPaste())
		return;
	
	// unhilite and delete the selection
	if (mSelStart != mSelEnd) {
		if (mActive)
			DrawSelection(mSelStart, mSelEnd);
	
		RemoveRange(mSelStart, mSelEnd);
		mSelEnd = mSelStart;
	}
		
	be_clipboard->Lock();
	
	// get data from clipboard
	long 		textLen = 0;
	const char	*text = be_clipboard->FindText(&textLen);
	
	long 				styleLen = 0;
	STEStyleRangePtr	styles = NULL;
	styles = (STEStyleRangePtr)be_clipboard->FindData(STE_STYLE_TYPE, &styleLen);

	// copy text and styles into the buffers
	Insert(text, textLen, styles);

	be_clipboard->Unlock();
}


// ------------------------------------------------------------
// 	 Clear
// ------------------------------------------------------------
// Delete the current selection

void
STEngine::Clear()
{
	Delete();
}


// ------------------------------------------------------------
// 	 SelectAll
// ------------------------------------------------------------
// Select everything

void
STEngine::SelectAll()
{
	if (mSelectable)
		Select(0, LONG_MAX);
}


// ------------------------------------------------------------
// 	 CanPaste
// ------------------------------------------------------------
// Return true if the clipboard contains data that can inserted 

bool
STEngine::CanPaste()
{
	if (!mEditable)
		return (FALSE);
		
	bool result = FALSE;
	
	be_clipboard->Lock();
	
	result = (be_clipboard->CountEntries(B_ASCII_TYPE) > 0);
	
	be_clipboard->Unlock();
	
	return (result);
}


// ------------------------------------------------------------
// 	 CanDrop
// ------------------------------------------------------------
// Return true if the message contains data that can be dropped

bool
STEngine::CanDrop(
	BMessage	*inMessage)
{
	if (!mEditable)
		return (FALSE);
		
	if ( (inMessage->HasData("text", B_ASCII_TYPE)) ||
	 	 (inMessage->HasData("char", B_LONG_TYPE)) )
		return (TRUE);
		
	return (FALSE);
}


// ------------------------------------------------------------
// 	 Select
// ------------------------------------------------------------
// Select (hilite) a range of text

void
STEngine::Select(
	long	startOffset,
	long 	endOffset)
{
	// a negative selection?
	if (startOffset > endOffset)
		return;
		
	// pin offsets at reasonable values
	startOffset = (startOffset < 0) ? 0 : startOffset;
	endOffset = (endOffset < 0) ? 0 : endOffset;
	endOffset = (endOffset > mText.Length()) ? mText.Length() : endOffset;

	// is the new selection any different from the current selection?
	if ((startOffset == mSelStart) && (endOffset == mSelEnd))
		return;
	
	mStyles.InvalidateNullStyle();
	
	// hide the caret
	if (mCaretVisible)
		InvertCaret();
	
	if (startOffset == endOffset) {
		if (mSelStart != mSelEnd) {
			// unhilite the selection
			if ((mActive) && (mSelectable))
				DrawSelection(mSelStart, mSelEnd); 
		}
		mSelStart = mSelEnd = startOffset;
		if ((mActive) && (mEditable))
			InvertCaret();
	}
	else {
		if ((mActive) && (mSelectable)) {
			// does the new selection overlap with the current one?
			if ( ((startOffset < mSelStart) && (endOffset < mSelStart)) ||
				 ((endOffset > mSelEnd) && (startOffset > mSelEnd)) ) {
				// they don't overlap, don't bother with stretching
				// thanks to Brian Stern for the code snippet
				DrawSelection(mSelStart, mSelEnd);
				DrawSelection(startOffset, endOffset);	
			}
			else {
				// stretch the selection, draw only what's different
				long start, end;
				if (startOffset != mSelStart) {
					// start of selection has changed
					if (startOffset > mSelStart) {
						start = mSelStart;
						end = startOffset;
					}
					else {
						start = startOffset;
						end = mSelStart;
					}
					DrawSelection(start, end);
				}
				
				if (endOffset != mSelEnd) {
					// end of selection has changed
					if (endOffset > mSelEnd) {
						start = mSelEnd;
						end = endOffset;
					}
					else {
						start = endOffset;
						end = mSelEnd;
					}
					DrawSelection(start, end);
				}
			}
		}
		mSelStart = startOffset;
		mSelEnd = endOffset;
	}
}


// ------------------------------------------------------------
// 	 GetSelection
// ------------------------------------------------------------
// Pass back the current selection offsets

void
STEngine::GetSelection(
	long	*outStart,
	long 	*outEnd) const
{
	*outStart = mSelStart;
	*outEnd = mSelEnd;
}


// ------------------------------------------------------------
// 	 SetStyle
// ------------------------------------------------------------
// Set the style for the current selection
//
// inMode specifies the pertinent fields of inStyle
// It can be one (or a combination) of the following:
//
//	doFont			-	set font
//	doSize			-	set size
//	doShear			-	set shear
//	doUnderline		-	set underline
//	doColor			-	set color
//	doExtra			-	set the extra field
//	doAll			-	set everything
//	addSize			-	add size value
	
void
STEngine::SetStyle(
	ulong				inMode,
	ConstSTEStylePtr	inStyle)
{
	// hide the caret/unhilite the selection
	if (mActive) {
		if (mSelStart != mSelEnd)
			DrawSelection(mSelStart, mSelEnd);
		else {
			if (mCaretVisible)
				InvertCaret();
		}
	}
	
	// add the style to the style buffer
	mStyles.SetStyleRange(mSelStart, mSelEnd, mText.Length(),
						  inMode, inStyle, this);
						
	if ((inMode & doFont) || (inMode & doSize))
		// recalc the line breaks and redraw with new style
		Refresh(mSelStart, mSelEnd, mSelStart != mSelEnd, FALSE);
	else
		// the line breaks wont change, simply redraw
		DrawLines(OffsetToLine(mSelStart), OffsetToLine(mSelEnd), 
				  mSelStart, TRUE);
	
	HandleModification();
	
	// draw the caret/hilite the selection
	if (mActive) {
		if (mSelStart != mSelEnd)
			DrawSelection(mSelStart, mSelEnd);
		else {
			if (!mCaretVisible)
				InvertCaret();
		}
	}
}


// ------------------------------------------------------------
// 	 GetStyle
// ------------------------------------------------------------
// Get the style at inOffset

void
STEngine::GetStyle(
	long		inOffset,
	STEStylePtr	outStyle) const
{
	mStyles.GetStyle(inOffset, outStyle);
}


// ------------------------------------------------------------
// 	 SetStyleRange
// ------------------------------------------------------------
// Set the styles of a range of text

void
STEngine::SetStyleRange(
	long					startOffset,
	long 					endOffset,
	ConstSTEStyleRangePtr	inStyles,
	bool					inRefresh)
{
	long numStyles = inStyles->count;
	if (numStyles < 1)
		return;
			
	if (inRefresh) {
		// hide the caret/unhilite the selection
		if (mActive) {
			if (mSelStart != mSelEnd)
				DrawSelection(mSelStart, mSelEnd);
			else {
				if (mCaretVisible)
					InvertCaret();
			}
		}
	}
	
	// pin offsets at reasonable values
	long textLength = mText.Length();
	startOffset = (startOffset < 0) ? 0 : startOffset;
	endOffset = (endOffset < 0) ? 0 : endOffset;
	endOffset = (endOffset > textLength) ? textLength : endOffset;
	
	// loop through the style runs
	ConstSTEStyleRunPtr	theRun = &inStyles->runs[0];
	for (long index = 0; index < numStyles; index++) {
		long fromOffset = theRun->offset + startOffset;
		long toOffset = endOffset;
		if ((index + 1) < numStyles) {
			toOffset = (theRun + 1)->offset + startOffset;
			toOffset = (toOffset > endOffset) ? endOffset : toOffset;
		}

		mStyles.SetStyleRange(fromOffset, toOffset, textLength,
						  	  doAll, &theRun->style, this);
						
		theRun++;
	}
	
	mStyles.InvalidateNullStyle();
	
	if (inRefresh) {
		Refresh(startOffset, endOffset, TRUE, FALSE);
		
		// draw the caret/hilite the selection
		if (mActive) {
			if (mSelStart != mSelEnd)
				DrawSelection(mSelStart, mSelEnd);
			else {
				if (!mCaretVisible)
					InvertCaret();
			}
		}
	}
}


// ------------------------------------------------------------
// 	 GetStyleRange
// ------------------------------------------------------------
// Return the styles of a range of text
//
// You are responsible for freeing the buffer that is returned
//
// Pass NULL for outLength if you are not interested in the
// length of the buffer

STEStyleRangePtr
STEngine::GetStyleRange(
	long	startOffset,
	long	endOffset,
	long	*outLength) const
{
	STEStyleRangePtr result = mStyles.GetStyleRange(startOffset, endOffset - 1);
	
	if (outLength != NULL)
		*outLength = sizeof(long) + (sizeof(STEStyleRun) * result->count);
	
	return (result);
}


// ------------------------------------------------------------
// 	 IsContinuousStyle
// ------------------------------------------------------------
// Return whether the attributes that are set in ioMode are 
// continuous over the selection range 
//
// Any attributes in ioMode that aren't contiuous will be cleared
//
// outStyle will be set with the continuous attributes only

bool
STEngine::IsContinuousStyle(
	ulong		*ioMode,
	STEStylePtr	outStyle) const
{
	bool result = TRUE;
	
	if ((mSelStart == mSelEnd) && (mStyles.IsValidNullStyle()))
		mStyles.SetStyle(*ioMode, &mStyles.GetNullStyle(), outStyle);
	else
		result = mStyles.IsContinuousStyle(ioMode, outStyle, 
									   	   mSelStart, mSelEnd);
			
	return (result);
}


// ------------------------------------------------------------
// 	 OffsetToLine
// ------------------------------------------------------------
// Return the number of the line in the line array that 
// contains offset
// 
// Line numbers start at 0

long
STEngine::OffsetToLine(
	long	offset) const
{
	return (mLines.OffsetToLine(offset));
}


// ------------------------------------------------------------
// 	 PixelToLine
// ------------------------------------------------------------
// Return the number of the line at the vertical location
// pixel should be in local coordinates
// 
// Line numbers start at 0

long
STEngine::PixelToLine(
	float	pixel) const
{
	return (mLines.PixelToLine(pixel - mTextRect.top));
}


// ------------------------------------------------------------
// 	 OffsetToPoint
// ------------------------------------------------------------
// Return the local coordinates of the character at inOffset
// Pass back the height of the line that contains inOffset in outHeight
//
// The vertical coordinate will be at the origin (top) of the line
// The horizontal coordinate will be to the left of the character
// at offset
//
// Pass NULL for outHeight if you do not need to know the height

BPoint
STEngine::OffsetToPoint(
	long	inOffset,
	float	*outHeight)
{
	BPoint 			result;
	long			textLength = mText.Length();
	long 			lineNum = OffsetToLine(inOffset);
	ConstSTELinePtr	line = mLines[lineNum];
	float 			height = (line + 1)->origin - line->origin;
	
	result.x = 0.0;
	result.y = line->origin + mTextRect.top;
	
	if (textLength > 0) {
		// special case: go down one line if inOffset is a newline
		if ((inOffset == textLength) && (mText[textLength - 1] == '\n')) {
			float	ascent = 0.0;
			float	descent = 0.0;
	
			StyledWidth(inOffset, 1, &ascent, &descent);
			
			result.y += height;
			height = (ascent + descent);
		}
		else {
			long	offset = line->offset;
			long	length = inOffset - line->offset;
			long	numChars = length;
			bool	foundTab = FALSE;		
			do {
				foundTab = mText.FindChar('\t', offset, &numChars);
			
				result.x += StyledWidth(offset, numChars);
		
				if (foundTab) {
					long numTabs = 0;
					for (numTabs = 0; (numChars + numTabs) < length; numTabs++) {
						if (mText[offset + numChars + numTabs] != '\t')
							break;
					}
											
					float tabWidth = ActualTabWidth(result.x);
					if (numTabs > 1)
						tabWidth += ((numTabs - 1) * mTabWidth);
		
					result.x += tabWidth;
					numChars += numTabs;
				}
				
				offset += numChars;
				length -= numChars;
				numChars = length;
			} while ((foundTab) && (length > 0));
		} 
	}

	// convert from text rect coordinates
	result.x += (mTextRect.left - 1.0);

	if (outHeight != NULL)
		*outHeight = height;
		
	return (result);
}


// ------------------------------------------------------------
// 	 PointToOffset
// ------------------------------------------------------------
// Return the offset of the character that lies at point
// 
// point should be in local coordinates

long
STEngine::PointToOffset(
	BPoint	point)
{
	// should we even bother?
	if (point.y >= mTextRect.bottom)
		return (mText.Length());
	else {
		if (point.y < mTextRect.top)
			return (0);
	}

	long			lineNum = PixelToLine(point.y);
	ConstSTELinePtr	line = mLines[lineNum];
	
	// special case: if point is within the text rect and PixelToLine()
	// tells us that it's on the last line, but if point is actually  
	// lower than the bottom of the last line, return the last offset 
	// (can happen for newlines)
	if (lineNum == (mLines.NumLines() - 1)) {
		if (point.y >= ((line + 1)->origin + mTextRect.top))
			return (mText.Length());
	}
	
	// convert to text rect coordinates
	point.x -= mTextRect.left;
	point.x = (point.x < 0.0) ? 0.0 : point.x;
	
	// do a pseudo-binary search of the character widths on the line
	// that PixelToLine() gave us
	// note: the right half of a character returns its offset + 1
	long	offset = line->offset;
	long	saveOffset = offset;
	long	limit = (line + 1)->offset;
	long	length = limit - line->offset;
	float	sigmaWidth = 0.0;
	long	numChars = length;
	bool	foundTab = FALSE;
	bool	done = FALSE;
	do {		
		// any tabs?
		foundTab = mText.FindChar('\t', offset, &numChars);

		if (numChars > 0) {
			long delta = numChars / 2;
			delta = (delta < 1) ? 1 : delta;

			do {
				float deltaWidth = StyledWidth(offset, delta);
				float leftWidth = StyledWidth(offset + delta - 1, 1);
				sigmaWidth += deltaWidth;
	
				if (point.x >= (sigmaWidth - (leftWidth / 2))) {
					// we're to the left of the point
					float rightWidth = StyledWidth(offset + delta, 1);
					if (point.x < (sigmaWidth + (rightWidth / 2))) {
						// the next character is to the right, we're done!
						offset += delta;
						done = TRUE;
						break;
					}
					else {
						// still too far to the left, measure some more
						offset += delta;
						delta /= 2;
						delta = (delta < 1) ? 1 : delta;
					}
				}
				else {
					// oops, we overshot the point, go back some 
					sigmaWidth -= deltaWidth;
					
					if (delta == 1) {
						done = TRUE;
						break;
					}
						
					delta /= 2;
					delta = (delta < 1) ? 1 : delta;
	
				}
			} while (offset < (numChars + saveOffset));
		}
		
		if (done || (offset >= limit))
			break;
			
		if (foundTab) {
			float tabWidth = ActualTabWidth(sigmaWidth);
			
			// is the point in the left-half of the tab?
			if (point.x < (sigmaWidth + (tabWidth / 2)))
				break;
			else {
				// is the point in the right-half of the tab?
				if (point.x < (sigmaWidth + tabWidth)) {
					offset++;
					break;
				}
			}
				
			sigmaWidth += tabWidth;
			numChars++;
			
			// maybe we have more tabs?
			bool	foundPoint = FALSE;
			long	numTabs = 0;
			for (numTabs = 0; (numChars + numTabs) < length; numTabs++) {
				if (mText[saveOffset + numChars + numTabs] != '\t')
					break;
				
				if (point.x < (sigmaWidth + (mTabWidth / 2))) {
					foundPoint = TRUE;
					break;
				}
				else {
					if (point.x < (sigmaWidth + mTabWidth)) {
						foundPoint = TRUE;
						numTabs++;
						break;
					}
				}
				
				sigmaWidth += mTabWidth;
			}

			if (foundPoint) {
				offset = saveOffset + numChars + numTabs;
				break;
			}

			numChars += numTabs;
		}

		offset = saveOffset + numChars;
		saveOffset = offset;
		length -= numChars;
		numChars = length;
	} while (length > 0);
	
	if (offset == (line + 1)->offset) {
		// special case: newlines aren't visible
		// return the offset of the character preceding the newline
		if (mText[offset - 1] == '\n')
			return (offset - 1);

		// special case: return the offset preceding any spaces that 
		// aren't at the end of the buffer
		if ((offset != mText.Length()) && (mText[offset - 1] == ' '))
			return (offset - 1);
	}
	
	return (offset);
}


// ------------------------------------------------------------
// 	 FindWord
// ------------------------------------------------------------
// Return a pair of offsets that describe a word at inOffset
//
// Override STEngine::IsWordBreakChar() to define a 'word'

void
STEngine::FindWord(
	long	inOffset, 
	long	*outFromOffset,
	long 	*outToOffset)
{
	long offset;
	
	// check to the left
	for (offset = inOffset; offset > 0; offset--) {
		if (IsWordBreakChar(mText[offset - 1]))
			break;
	}
	*outFromOffset = offset;

	// check to the right
	long textLen = mText.Length();
	for (offset = inOffset; offset < textLen; offset++) {
		if (IsWordBreakChar(mText[offset]))
			break;
	}
	*outToOffset = offset;
}


// ------------------------------------------------------------
// 	 GetHeight
// ------------------------------------------------------------
// Return the height from startLine to endLine
						
float
STEngine::GetHeight(
	long	startLine,
	long	endLine)
{
	long lastChar = mText.Length() - 1;
	long lastLine = mLines.NumLines() - 1;
	startLine = (startLine < 0) ? 0 : startLine;
	endLine = (endLine > lastLine) ? lastLine : endLine;
	
	float height = mLines[endLine + 1]->origin - 
				   mLines[startLine]->origin;
				
	if ((endLine == lastLine) && (mText[lastChar] == '\n')) {
		float ascent = 0.0;
		float descent = 0.0;

		StyledWidth(lastChar, 1, &ascent, &descent);
		
		height += (ascent + descent);
	}
	
	return (height);
}


// ------------------------------------------------------------
// 	 GetHiliteRegion
// ------------------------------------------------------------
// Return the region that encompasses the characters in the
// range of startOffset to endOffset

void
STEngine::GetHiliteRegion(
	long	startOffset,
	long	endOffset,
	BRegion	*outRegion)
{
	outRegion->MakeEmpty();

	// return an empty region if the range is invalid
	if (startOffset >= endOffset)
		return;

	float	startLineHeight = 0.0;
	float	endLineHeight = 0.0;
	BPoint	startPt = OffsetToPoint(startOffset, &startLineHeight);
	startPt.x = ceil(startPt.x);
	startPt.y = ceil(startPt.y);
	BPoint	endPt = OffsetToPoint(endOffset, &endLineHeight);
	endPt.x = ceil(endPt.x);
	endPt.y = ceil(endPt.y);
	BRect	selRect;

	if (startPt.y == endPt.y) {
		// this is a one-line region
		selRect.left = (startPt.x < mTextRect.left) ? mTextRect.left : startPt.x;
		selRect.top = startPt.y;
		selRect.right = endPt.x - 1.0;
		selRect.bottom = endPt.y + endLineHeight - 1.0;
		outRegion->Include(selRect);
	}
	else {
		// more than one line in the specified offset range
		selRect.left = (startPt.x < mTextRect.left) ? mTextRect.left : startPt.x;
		selRect.top = startPt.y;
		selRect.right = mTextRect.right;
		selRect.bottom = startPt.y + startLineHeight - 1.0;
		outRegion->Include(selRect);
		
		if ((startPt.y + startLineHeight) < endPt.y) {
			// more than two lines in the range
			selRect.left = mTextRect.left;
			selRect.top = startPt.y + startLineHeight;
			selRect.right = mTextRect.right;
			selRect.bottom = endPt.y - 1.0;
			outRegion->Include(selRect);
		}
		
		selRect.left = mTextRect.left;
		selRect.top = endPt.y;
		selRect.right = endPt.x - 1.0;
		selRect.bottom = endPt.y + endLineHeight - 1.0;
		outRegion->Include(selRect);
	}
}


// ------------------------------------------------------------
// 	 ScrollToOffset
// ------------------------------------------------------------
// Scroll the view so that inOffset is fully visible
//
// This function does nothing if there are no scroll bars 
// that target this view

void
STEngine::ScrollToOffset(
	long	inOffset)
{
	BRect	bounds = Bounds();
	float	lineHeight = 0.0;
	BPoint	point = OffsetToPoint(inOffset, &lineHeight);
	bool	update = FALSE;
	
	if ((point.x < bounds.left) || (point.x >= bounds.right)) {
		BScrollBar *hScroll = ScrollBar(B_HORIZONTAL);
		if (hScroll != NULL) {
			update = TRUE;
			hScroll->SetValue(point.x - (bounds.IntegerWidth() / 2));
		}
	}
	
	if ((point.y < bounds.top) || ((point.y + lineHeight) >= bounds.bottom)) {
		BScrollBar *vScroll = ScrollBar(B_VERTICAL);
		if (vScroll != NULL) {
			update = TRUE;
			vScroll->SetValue(point.y - (bounds.IntegerHeight() / 2));
		}
	}
	
	if (update)
		Window()->UpdateIfNeeded();
}


// ------------------------------------------------------------
// 	 ScrollToSelection
// ------------------------------------------------------------
// Scroll the view so that the beginning of the selection is
// within visible range
//
// This function does nothing if there are no scroll bars 
// that target this view

void
STEngine::ScrollToSelection()
{
	ScrollToOffset(mSelStart);
}


// ------------------------------------------------------------
// 	 SetTextRect
// ------------------------------------------------------------
// Set the text rect
//
// This function will also resize the offscreen bitmap 
// (if there is one)

void
STEngine::SetTextRect(
	BRect	rect)
{
	if (rect == mTextRect)
		return;
		
	mTextRect = rect;
	mTextRect.bottom = mTextRect.top;

	if (DoesUseOffscreen())
		UseOffscreen(mOffscreen->ColorSpace());
	
	if (Window() != NULL) {
		if (mActive) {
			// hide the caret, unhilite the selection
			if (mSelStart != mSelEnd)
				DrawSelection(mSelStart, mSelEnd);
			else {
				if (mCaretVisible)
					InvertCaret();
			}
		}
		
		Refresh(0, LONG_MAX, TRUE, FALSE);
		
		// invalidate and immediately redraw in case the 
		// text rect has gotten smaller 
		Invalidate();
		Window()->UpdateIfNeeded();
	}
}


// ------------------------------------------------------------
// 	 TextRect
// ------------------------------------------------------------
// Get the text rect

BRect
STEngine::TextRect() const
{
	return (mTextRect);
}

	
// ------------------------------------------------------------
// 	 SetTabWidth
// ------------------------------------------------------------
// Set the width of the tab character

void
STEngine::SetTabWidth(
	float	width)
{
	if (width == mTabWidth)
		return;
		
	mTabWidth = width;
	
	if (Window() != NULL) {
		if (mActive) {
			// hide the caret, unhilite the selection
			if (mSelStart != mSelEnd)
				DrawSelection(mSelStart, mSelEnd);
			else {
				if (mCaretVisible)
					InvertCaret();
			}
		}
	
		Refresh(0, LONG_MAX, TRUE, FALSE);
		
		if (mActive) {
			// show the caret, hilite the selection
			if (mSelStart != mSelEnd)
				DrawSelection(mSelStart, mSelEnd);
			else {
				if (!mCaretVisible)
					InvertCaret();
			}
		}
	}
}


// ------------------------------------------------------------
// 	 TabWidth
// ------------------------------------------------------------
// Return the width of the tab character

float
STEngine::TabWidth() const
{
	return (mTabWidth);
}


// ------------------------------------------------------------
// 	 MakeSelectable
// ------------------------------------------------------------
// Set whether the caret/selection range will be displayed

void
STEngine::MakeSelectable(
	bool	selectable)
{
	if (selectable == mSelectable)
		return;
		
	mSelectable = selectable;
	
	if (Window() != NULL) {
		if (mActive) {
			// show/hide the caret, hilite/unhilite the selection
			if (mSelStart != mSelEnd)
				DrawSelection(mSelStart, mSelEnd);
			else
				InvertCaret();
		}
	}
}


// ------------------------------------------------------------
// 	 IsSelectable
// ------------------------------------------------------------
// Return whether the caret/selection range will be displayed

bool
STEngine::IsSelectable() const
{
	return (mSelectable);
}


// ------------------------------------------------------------
// 	 MakeEditable
// ------------------------------------------------------------
// Set whether the text can be edited

void
STEngine::MakeEditable(
	bool	editable)
{
	if (editable == mEditable)
		return;
		
	mEditable = editable;
	
	if (Window() != NULL) {
		if (mActive) {
			if ((!mEditable) && (mCaretVisible))
				InvertCaret();
		}
	}
}


// ------------------------------------------------------------
// 	 IsEditable
// ------------------------------------------------------------
// Return whether the text can be edited

bool
STEngine::IsEditable() const
{
	return (mEditable);
}


// ------------------------------------------------------------
// 	 SetWordWrap
// ------------------------------------------------------------
// Set whether the text should be wrapped 

void
STEngine::SetWordWrap(
	bool	wrap)
{
	if (wrap == mWrap)
		return;
		
	mWrap = wrap;
	
	if (Window() != NULL) {
		if (mActive) {
			// hide the caret, unhilite the selection
			if (mSelStart != mSelEnd)
				DrawSelection(mSelStart, mSelEnd);
			else {
				if (mCaretVisible)
					InvertCaret();
			}
		}
		
		Refresh(0, LONG_MAX, TRUE, FALSE);
		
		if (mActive) {
			// show the caret, hilite the selection
			if (mSelStart != mSelEnd)
				DrawSelection(mSelStart, mSelEnd);
			else {
				if (!mCaretVisible)
					InvertCaret();
			}
		}
	}
}


// ------------------------------------------------------------
// 	 DoesWordWrap
// ------------------------------------------------------------
// Return whether the text is wrapped 

bool
STEngine::DoesWordWrap() const
{
	return (mWrap);
}


// ------------------------------------------------------------
// 	 UseOffscreen
// ------------------------------------------------------------
// Use an offscreen bitmap that is colors deep
//
// The bitmap is used only for drawing the current line

void
STEngine::UseOffscreen(
	color_space	colors)
{
	// delete in case we're resizing or changing color spaces
	delete (mOffscreen);
	
	BRect offBounds;
	offBounds.left = 0.0;
	offBounds.top = 0.0;
	offBounds.right = mTextRect.Width();
	offBounds.bottom = mLines[1]->origin - mLines[0]->origin;
	mOffscreen = new BBitmap(offBounds, colors, TRUE);
	mOffscreen->AddChild(new BView(offBounds, B_EMPTY_STRING, B_FOLLOW_NONE, 0));
}


// ------------------------------------------------------------
// 	 DontUseOffscreen
// ------------------------------------------------------------
// Delete the offscreen bitmap and settle with the ugly 
// flickers

void
STEngine::DontUseOffscreen()
{
	delete (mOffscreen);
	mOffscreen = NULL;
}


// ------------------------------------------------------------
// 	 DoesUseOffscreen
// ------------------------------------------------------------
// Return whether an offscreen bitmap is being used

bool
STEngine::DoesUseOffscreen() const
{
	return (mOffscreen != NULL);
}
































