#include <stdlib.h>
#include <string.h>
#include <Be.h>
#include <Debug.h>
#include "PrefsSection.h"


#define kTimeOut	500000			// half a sec time out

//***************************************************************
//***************************************************************
			PrefsSection::PrefsSection(PrefsServerType serverType, const char* path)
{
		status_t			result;
		BMessage			msg(PrefServ_OpenSectionMsg);
		BMessage			reply;
		thread_info			info;

	server = new BMessenger("application/x-be-prefs_server", -1, &result);
	if (result != B_NO_ERROR || ! server->IsValid())
		goto err;
		
	get_thread_info(find_thread(NULL), &info);
	msg.AddInt32("team_id", info.team);
	msg.AddString("path", path);
	msg.AddInt32("server", serverType);
	server->SendMessage(&msg, &reply, kTimeOut);
	id = reply.FindInt32("sect_id");
	validator = reply.FindInt32("sect_validator");
	readOnly = reply.FindBool("read_only");
	
	if (validator != 0)
		return;
		
err:	
	delete server;
	server = NULL;
}


//***************************************************************
//***************************************************************
			PrefsSection::~PrefsSection()
{
		BMessage			msg(PrefServ_CloseSectionMsg);

	if (server == NULL)
		return;

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	server->SendMessage(&msg);
	delete server;
}


//***************************************************************
//***************************************************************
void			PrefsSection::Flush()
{
		BMessage			msg(PrefServ_FlushSectionMsg);
		
	if (server == NULL)
		return;

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	server->SendMessage(&msg);
}


//***************************************************************
//***************************************************************
bool			PrefsSection::IsValid()
{
	return(server != NULL);
}


//***************************************************************
//***************************************************************
bool			PrefsSection::ReadOnly()
{
	return(readOnly);
}

//***************************************************************
//***************************************************************
void		PrefsSection::FindString(const char* entry, char* value, const uint32 sizeOfValue, const char* defValue )
{
		BMessage			msg(PrefServ_ReadEntryMsg);
		BMessage			reply;
		void*				ptr;
		ssize_t				size;
		
	if (server != NULL)
	{
		msg.AddInt32("sect_id", id);
		msg.AddInt32("sect_validator", validator);
		msg.AddString("entry_name", entry);
		server->SendMessage(&msg, &reply, kTimeOut);	
		if (reply.FindData("result", B_STRING_TYPE, &ptr, &size) != B_NO_ERROR)
			ptr = NULL;
		else
			if (size < sizeof(uint32) || *(uint32*)ptr != PS_StringType)
				ptr = NULL;
			else
			{
				ptr = (char*)ptr + 4;
				size -= sizeof(uint32);
			}
	}
	else
		ptr = NULL;
	
	if (ptr == NULL)
	{
		size = strlen(defValue);
		ptr = (void*)defValue;
	}

	if (size >= sizeOfValue)
		size = sizeOfValue - 1;
	memcpy(value, ptr, size);
	value[size] = 0;
}


//***************************************************************
//***************************************************************
uint32		PrefsSection::FindData(const char* entry, void* buffer, const uint32 bufferSize)
{
		BMessage			msg(PrefServ_ReadEntryMsg);
		BMessage			reply;
		void*				ptr;
		ssize_t				size;
		
	if (server == NULL)
		return(0);

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	msg.AddString("entry_name", entry);
	server->SendMessage(&msg, &reply, kTimeOut);
	if (reply.FindData("result", B_STRING_TYPE, &ptr, &size) == B_NO_ERROR)
	{
		if (size < sizeof(uint32) || *(uint32*)ptr != PS_StringType)
			size = 0;
		else
		{
			ptr = (char*)ptr + 4;
			size -= sizeof(uint32);
			if (size > bufferSize)
				memcpy(buffer, ptr, bufferSize);
			else
				memcpy(buffer, ptr, size);
		}
	}
	else
		size = 0;
	
	return(size);
}


//***************************************************************
//***************************************************************
bool		PrefsSection::Find(const uint32 type, const char* entry, void* data, const uint32 len)
{
		BMessage			msg(PrefServ_ReadEntryMsg);
		BMessage			reply;
		void*				ptr;
		ssize_t				size;
		bool				result	= false;		
		
	if (server == NULL)
		return(false);

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	msg.AddString("entry_name", entry);
	server->SendMessage(&msg, &reply, kTimeOut);
	if (reply.FindData("result", B_STRING_TYPE, &ptr, &size) == B_NO_ERROR)
	{
		ASSERT(size == len + sizeof(uint32));
		ASSERT(*(uint32*)ptr == type);
		if (size == len + sizeof(uint32) && *(uint32*)ptr == type)
		{
			memcpy(data, (char*)ptr + sizeof(uint32), len);
			result = true;
		}
	}
	return(result);
}


//***************************************************************
//***************************************************************
void			PrefsSection::WriteString(const char* entry, const char* value)
{
	Write(PS_StringType, entry, value, strlen(value));
}


//***************************************************************
//***************************************************************
void			PrefsSection::WriteData(const char* entry, const void* data, const uint32 len)
{
	Write(PS_DataType, entry, data, len);
}


//***************************************************************
//***************************************************************
void		PrefsSection::Write(const uint32 type, const char* entry, const void* value, const uint32 len)
{
		BMessage			msg(PrefServ_WriteEntryMsg);
		char				buffer[1024];
		void*				ptr;
		
	if (server == NULL || *entry == 0)
		return;

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	msg.AddString("entry_name", entry);
	
	if (len + sizeof(uint32) > sizeof(buffer))
		ptr = malloc(len + sizeof(uint32));
	else
		ptr = buffer;
	memcpy((char*)ptr + sizeof(uint32), value, len);
	*(uint32*)ptr = type;

	msg.AddData("value", B_STRING_TYPE, ptr, len + sizeof(uint32));

	if (ptr != buffer)
		free(ptr);
		
	server->SendMessage(&msg);
}




#define DEF(b,B)	b	PrefsSection::Find##B(const char* entry, b defValue)	\
					{															\
							b		value;										\
																				\
						if (Find(PS_##B##Type, entry, &value, sizeof(b)))		\
							return(value);										\
						else													\
							return(defValue);									\
					}															\
					void PrefsSection::Write##B(const char* entry, b value)	\
					{															\
						Write(PS_##B##Type, entry, &value, sizeof(value));		\
					}															\


DEF(bool,Bool)
DEF(int32,Int32)
DEF(float,Float)
DEF(BRect,Rect)
DEF(BPoint,Point)


//***************************************************************
//***************************************************************
void		PrefsSection::DeleteEntry(const char* entry)
{
		BMessage			msg(PrefServ_DeleteEntryMsg);
		
	if (server == NULL)
		return;

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	msg.AddString("entry_name", entry);
	server->SendMessage(&msg);
}


//***************************************************************
//***************************************************************
bool		PrefsSection::EntryExists(const char* entry, PrefsType* type)
{
		BMessage			msg(PrefServ_EntryExistsMsg);
		BMessage			reply;
		int32				result;
		
	if (server == NULL)
		return(false);

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	msg.AddString("entry_name", entry);
	server->SendMessage(&msg, &reply, kTimeOut);
	result = reply.FindInt32("result");
	if (type != NULL)
		*type = (PrefsType)result;
	
	return(result != 0);
}

//***************************************************************
//***************************************************************
char*		PrefsSection::MallocList(const uint32 msgID)
{
		BMessage			msg(msgID);
		BMessage			reply;
		void*				ptr;
		ssize_t				size;
		void*				result;
		
	if (server == NULL)
		return(NULL);

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	server->SendMessage(&msg, &reply, kTimeOut * 10); // 5 secs
	if (reply.FindData("result", B_STRING_TYPE, &ptr, &size) != B_NO_ERROR)
		return(NULL);

	if (size == 0)
		return(NULL);

	result = malloc(size);
	memcpy(result, ptr, size);
	
	return((char*)result);
}


//***************************************************************
//***************************************************************
char*		PrefsSection::MallocSectionList()
{
	return(MallocList(PrefServ_SectionListMsg));
}

//***************************************************************
//***************************************************************
char*		PrefsSection::MallocEntryList()
{
	return(MallocList(PrefServ_EntryListMsg));
}


//***************************************************************
//***************************************************************
bool		PrefsSection::DeleteSection(const char* sectionName)
{
		BMessage			msg(PrefServ_DeleteSectionMsg);
		BMessage			reply;
		
	if (server == NULL)
		return(false);

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	msg.AddString("section", sectionName);
	server->SendMessage(&msg, &reply, kTimeOut);	
	return(reply.FindBool("result"));
}


//***************************************************************
//***************************************************************
uint32		PrefsSection::IsSectionInUse(const char* sectionName)
{
		BMessage			msg(PrefServ_SectionInUseMsg);
		BMessage			reply;
		
	if (server == NULL)
		return(false);

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	msg.AddString("section", sectionName);
	server->SendMessage(&msg, &reply, kTimeOut);
	return(reply.FindInt32("result"));
}


//***************************************************************
//***************************************************************
void		PrefsSection::CreateSection(const char* sectionName)
{
		BMessage			msg(PrefServ_CreateSectionMsg);
		
	if (server == NULL)
		return;

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	msg.AddString("section", sectionName);
	server->SendMessage(&msg);
}


//***************************************************************
//***************************************************************
void		PrefsSection::Reset(bool recursive)
{
		BMessage			msg(PrefServ_ResetMsg);
		
	if (server == NULL)
		return;

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	msg.AddBool("recursive", recursive);
	server->SendMessage(&msg);
}


//***************************************************************
//***************************************************************
void		PrefsSection::SetMessage(uint32 msgID)
{
		BMessage			msg(PrefServ_SetMessageMsg);
		thread_info			info;
		
	if (server == NULL)
		return;

	msg.AddInt32("sect_id", id);
	msg.AddInt32("sect_validator", validator);
	
	get_thread_info(find_thread(NULL), &info);
	msg.AddInt32("team_id", info.team);
	msg.AddInt32("msg_id", msgID);
	server->SendMessage(&msg);
}

