#ifndef __ANIMATE_CURSOR_H__
#define __ANIMATE_CURSOR_H__
/*
    Class: AnimateCursor
    
    Description: AnimateCursor animates a series of cursors
    	in their own thread.  The animation will continue
    	until this object is deleted, or the method Suspend()
    	or Quit() are called.
    
    Sample Usage:
    
    	#include "AnimateCursor.h"
    	#include "Spinwheel.h"
    	AnimateCursor *cursors = new AnimateCursor( 4 );
    	if( cursors )
    	{
    	  cursors->SetCursor( 0, Spinwheel0 );
    	  cursors->SetCursor( 1, Spinwheel45 );
    	  cursors->SetCursor( 2, Spinwheel90 );
    	  cursors->SetCursor( 3, Spinwheel135 );
    	  
    	  cursors->Run( 0.1 );
    	}
    
    Limitations:
    	This code is provided "as is". The author and/or
    	company is not responsible for any problems encountered
    	from using this class.  In using this class, the
    	developer and/or their company accepts the use of
    	this code in current form.
    
	Author: John Tegen
	Dated: 7 July 1997
	Company: OmicronSoft, (c) All Rights Reserved, 1997
	
	Revision History:
	- 7 July 1997 : Original code
*/

#define OS_SAME_RATE		0.0
#define CURSOR_THREAD_NAME	32

class AnimateCursor
{
  private:
    float			cycle;		// seconds
    int32			qty;		// number of cursors
    unsigned char 	**cursors;	// cursors
    thread_id		thrd;		// thread id
    char			name	[ CURSOR_THREAD_NAME + 1 ];
    
  protected:
    static int32 RunLoop( void *a_arg );
    int32	LoopCursors( void );
    
  public:
  
    //
    // Create the animation with a known quantity of
    // cursors
    //
    AnimateCursor	( const char *a_name,	// name of thread
    				  const int32& a_qty	// quantity of cursors
    				);
    ~AnimateCursor	( void );
    
    //
    // Set each cursor that was allocated in the
    // constructor
    //
    void	SetCursor	( const int32& a_index,
    					  const unsigned char *a_cursor );
    
    //
    // Start the animation at the given rate
    //
    void	Run		( float a_cycle = 0.1 );	// seconds
    
    //
    // returns TRUE if cursor is running or snoozing
    //
    bool	IsRunning( void );
    
    //
    // Suspend the animation
    //
    void	Suspend	( void );
    
    //
    // Resume the animation
    //
    void	Resume	( float a_cycle = OS_SAME_RATE );
    
    //
    // Quit the animation
    // The cursors have not been freed here, you may
    // Run(); the animation again.
    //
    void	Quit	( void );
};

#endif