Crunch.h                                                                                               644    2013      62         2141  6072610125   5513                                                                                                                                                                                                                                                                                                                                                                      /*
 File:	Crunch.h

 API for all the 'mathy' routines

 Copyright 1995, Be Incorporated, All Rights Reserved.
*/

#ifndef CRUNCH_H
#define CRUNCH_H

#include <math.h>

#ifndef _POINT_H
#include <Point.h>
#endif
#ifndef _RECT_H
#include <Rect.h>
#endif
#ifndef _VIEW_H
#include <View.h>
#endif

void	CalcGapAngles(BRect rect1, BRect rect2,
			float *start_angle, float *end_angle);
BPoint	CalcIntersection(float angle, BRect the_rect);
float	CalcAngle(BPoint pt1, BPoint pt2, long *quad);
void	DrawFrame(BView *v, BRect frame, float angleStart, float angleEnd);
BPoint	CalcIntersection(BPoint pt1, BPoint pt2, BRect the_rect,
			BPoint *new_speed);
float	CalcGapPosition(float angle, float start, float end);
BPoint	CalcInitialPosition(float rel_loc, float start, float end,
			BRect bounds);

inline float  segment_length(BPoint p1, BPoint p2)
{
	float dx = p2.x - p1.x;
	float dy = p2.y - p1.y;
	return (sqrt((dx * dx) + (dy * dy)));
}

inline BPoint rect_center(BRect rect)
{
	return BPoint((rect.right + rect.left) / 2.0,
					(rect.top + rect.bottom) / 2.0);
}

bool Between(BPoint p1, BPoint p2, BPoint p3);

#endif
                                                                                                                                                                                                                                                                                                                                                                                                                               main.cpp                                                                                               644    2013      62        43377  6072610126   5611                                                                                                                                                                                                                                                                                                                                                                      /*--------------------------------------------------------*/
/*
 File:	main.cpp

 main implementation file for the BeBox bouncing ball demo

 Copyright 1995, Be Incorporated, All Rights Reserved.
*/
/*--------------------------------------------------------*/

#include <Debug.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#ifndef MAIN_H
#include "main.h"
#endif
#ifndef CRUNCH_H
#include "Crunch.h"
#endif
#ifndef _LIST_H
#include <List.h>
#endif
#ifndef _MENU_BAR_H
#include <MenuBar.h>
#endif
#ifndef _MENU_ITEM_H
#include <MenuItem.h>
#endif

/*--------------------------------------------------------*/

main(int argc, char* argv[])
{
	/*
	 To randomize the movement of the bouncing ball set
	 the random seed to the system time
	*/
	srand((long) system_time());

	// make the new application object and start it running
	my_app = new TBounceApp();
	my_app->Run();

	// application is finished so cleanup and return
	delete my_app;
	return 0;
}


/*--------------------------------------------------------*/
// ??? should I use these CLASS_INFO macros or not ???
B_DEFINE_CLASS_INFO(TBounceApp, BApplication);

/*
 The Application object for the BeBounce application.
 Passing my signature to the system
*/

TBounceApp::TBounceApp()
	: BApplication(MY_SIG)
{
	BList list;
	BRect wpos(120, 40, 321, 241);

	fPartner = NULL;

	/*
	 This version of BeBounce only supports 2 instances of
	 the app running at the same time. So if there are
	 already 2 instances running force a QUIT.

	 In the BeOS the 'roster' object maintains system wide
	 information about all applications. We ask it
	 for a list of all the running applications with
	 the same signature as mine.
	*/
	be_roster->GetThreadList(MY_SIG, &list);
	long app_count = list.CountItems();
	if (app_count > 2) {
		// only support 2 apps at a time
		PostMessage(B_QUIT_REQUESTED);
		return;
	}
	
	if (app_count == 1) {
		// The first instance of BeBounce will have a ball.
		fWindow = new TWindow(wpos, "Bouncing 1", TRUE);
	} else {
		fWindow = new TWindow(wpos, "Bouncing 2", FALSE);

		/*
		 There are 2 instances of BeBounce. One is me(!)
		 and one is the other guy. Determine which is
		 which by comparing thread_id's in the list 
		 returned by GetThreadList() against my thread.
		*/
		thread_id tid = (thread_id) list.ItemAt(0);
		if (tid == Thread())
			tid = (thread_id) list.ItemAt(1);
		
		InitPartner(tid);		 // tid is our partner!

		/*
		 Send the introductory message along with my thread
		 id and location in screen corrdinates.
		*/
		BMessage *msg = new BMessage(BB_HELLO);
		msg->AddLong("thread", Thread());
		msg->AddRect("rect", wpos);
		SendToPartner(msg);
	}

	/*
	 A little bit of menu code. Let's add a 'Stop Ball'
	 menu item to the application menu (found in the
	 Dock window)
	*/
	BMenuItem	*item;
	BPopUpMenu	*menu = MainMenu();
	item = new BMenuItem("Stop Ball",
					new BMessage(BB_STOP), 'K');
	item->SetTarget(fWindow);
	menu->AddItem(item);

	fWindow->Show();
}

/*--------------------------------------------------------*/

TBounceApp::~TBounceApp()
{
	fWindow = NULL;
	RemovePartner();
}

/*--------------------------------------------------------*/

void TBounceApp::InitPartner(thread_id tid)
{
	// race condition is 2 apps both send BB_HELLO message
	if (fPartner)
		return;

	// Establish a 'messenger' as the link to our partner.
	fPartner = new BMessenger(MY_SIG, tid);
	if (fPartner->Error() != B_NO_ERROR) {
		delete fPartner;
		fPartner = NULL;
	} 
}

/*--------------------------------------------------------*/

void TBounceApp::RemovePartner()
{
	// Delete the messenger and tell the window.
	if (fPartner) {
		delete fPartner;
		fPartner = NULL;
	}
	if (fWindow && fWindow->Lock()) {
		fWindow->PartnerGone();
		fWindow->Unlock();
	}
}

/*--------------------------------------------------------*/

bool TBounceApp::SendToPartner(BMessage *msg)
{
	/*
	 Send the message to our partner (if we have one).
	 If the send fails then we must assume that the partner
	 is dead so it gets removed.
	*/
	if (!fPartner) {
		delete msg;
		return FALSE;
	}

	fPartner->SendMessage(msg);
	if (fPartner->Error() != B_NO_ERROR) {
		RemovePartner();
		return FALSE;
	}
	return TRUE;
}

/*--------------------------------------------------------*/

bool TBounceApp::SendPosition(BRect rect)
{
	/*
	 Give our partner that the current window position
	 (in screen coordinates).
	*/
	if (fPartner) {
		BMessage	*msg = new BMessage(BB_WINDOW_MOVED);
		msg->AddRect("rect", rect);
		return SendToPartner(msg);
	}

	return FALSE;
}

/*--------------------------------------------------------*/

void TBounceApp::MessageReceived(BMessage *msg)
{
	switch (msg->what) {
		case BB_HELLO:
			if (fWindow->Lock()) {
				/*
				 A new instance of BeBounce was just launched
				 and sent us the introductory message.
				*/
				InitPartner(msg->FindLong("thread"));

				// Tell our new partner our current location.
				BRect pos = fWindow->Frame();
				SendPosition(pos);

				// Initialize our partner's current location.
				pos = msg->FindRect("rect");
				fWindow->PartnerMoved(pos);
				fWindow->Unlock();
			}
			break;
		case BB_GOODBYE:
			if (fWindow->Lock()) {
				// Our partner is quitting.
				RemovePartner();
				if (msg->HasBool("ball")) {
					fWindow->AddBall();
				}
				fWindow->Unlock();
			}
			break;
		case BB_WINDOW_MOVED:
			// Our partner is informing us that it moved.
			if (fWindow->Lock()) {
				BRect pos = msg->FindRect("rect");
				fWindow->PartnerMoved(pos);
				fWindow->Unlock();
			}
			break;
		case BB_BALL:
			if (fWindow->Lock()) {
				// Our partner just passed us the ball.
				BPoint	speed = msg->FindPoint("speed");
				float	rel_loc = msg->FindFloat("rel_loc");
				fWindow->AddBall(rel_loc, speed);
				fWindow->Unlock();
			}
			break;
	}
}

/*--------------------------------------------------------*/
/*--------------------------------------------------------*/
/*--------------------------------------------------------*/

B_DEFINE_CLASS_INFO(TWindow, BWindow);

TWindow::TWindow(BRect frame, const char *title, bool ball)
	: BWindow(frame, title, B_TITLED_WINDOW, B_NOT_RESIZABLE)
{
	Lock();

	BRect b = frame;

	// shift be to window coordinate system
	b.OffsetTo(B_ORIGIN);

	fMyFrame = frame;
	fGapStart = -1.0;
	fGapEnd = -1.0;
	fBall = NULL;

	if (ball)
		AddBall();

	/*
	 The drawing will take place in the view fOffView that
	 will be added to the offscreen bitmap fBitmap. In 
	 this way we'll do the drawing offscreen and then just
	 blit the result to the screen.
	*/
	fBitmap = new BBitmap(b, B_COLOR_8_BIT, TRUE);
	fOffView = new BView(b, "", 0, B_WILL_DRAW);

	fBitmap->Lock();
	fBitmap->AddChild(fOffView);
	DrawOffScreen(fOffView);	// draw the initial setup
	fBitmap->Unlock();

	/*
	 This view will be added to the visible window. It's
	 only role is to blit the offscreen bitmap to the
	 visible window.
	*/
	fMainView = new TBitmapView(b, fBitmap);
	AddChild(fMainView);

	Unlock();
}

/*--------------------------------------------------------*/

TWindow::~TWindow()
{
	delete fBitmap;
}

/*--------------------------------------------------------*/

void TWindow::FrameMoved(BPoint new_pos)
{
	/*
	 Override of BWindow::FrameMoved() so that as the
	 window is moved around the screen we can inform our
	 partner of our new location.
	*/
	fMyFrame = Frame();
	if (my_app->SendPosition(fMyFrame))
		WindowsMoved(fMyFrame, fPartnerFrame);
}


/*--------------------------------------------------------*/

void TWindow::AddBall()
{
	/*
	 Simply add a ball in some default place with some
	 default speed.
	*/
	if (fBall)
		return;
	BRect boundary = fMyFrame;
	boundary.OffsetTo(B_ORIGIN);
	boundary.InsetBy(1,1);
	fBall = new TBall(this, boundary, BPoint(40,30),
		RADIUS, BPoint(8,19));
	fBall->SetGap(fGapStart, fGapEnd);
}

/*--------------------------------------------------------*/

void TWindow::AddBall(float rel_loc, BPoint speed)
{
	/*
	 We're adding a ball that just 'jumped' over from
	 our partner. We're given the info needed to
	 determine where to place the ball.
	*/
	if (fBall)
		return;

	float	radius = RADIUS;
	BRect	b = fMyFrame;
	b.OffsetTo(B_ORIGIN);
	b.InsetBy(radius, radius);
	BPoint	position = CalcInitialPosition(rel_loc, fGapStart, fGapEnd, b);

	b = fMyFrame;
	b.OffsetTo(B_ORIGIN);

	fBall = new TBall(this, b, position, radius, speed);
	fBall->SetGap(fGapStart, fGapEnd);
}

/*--------------------------------------------------------*/

void TWindow::PartnerMoved(BRect partner_frame)
{
	// our partner moved so we must update the 'gap'
	fPartnerFrame = partner_frame;
	WindowsMoved(fMyFrame, partner_frame);
}

/*--------------------------------------------------------*/

void TWindow::PartnerGone()
{
	// partner is gone so remove the gap!
	fGapStart = fGapEnd = -1.0;
	if (fBall)
		fBall->SetGap(fGapStart, fGapEnd);
	Update();
}

/*--------------------------------------------------------*/

void TWindow::MessageReceived(BMessage *msg)
{
	switch (msg->what) {
		case BB_DRAW:
			{
			// The ball is telling the window to redraw
			Update(msg->FindRect("update"));
			break;
			}
		case BB_HIT_HOLE:
			{
			/*
			 The ball is telling us that it just hit the
			 hole. So it should get sent to the partner.
			*/
			BMessage	*m = new BMessage(BB_BALL);
			fBall->Lock();
			m->AddPoint("speed", msg->FindPoint("speed"));
			m->AddFloat("rel_loc", msg->FindFloat("rel_loc"));
			fBall->Unlock();

			// send the 'ball' to our partner
			my_app->SendToPartner(m);

			// delete the ball from this window
			fBall->Quit();
			fBall = NULL;
			Update();
			break;
			}
		case BB_STOP:
			{
			// Stop the ball in mid-flight
			if (fBall) {
				fBall->SetEnabled(!fBall->IsEnabled());

				BMenuItem *item = cast_as(msg->FindObject("source"), BMenuItem);
				item->SetMarked(!item->IsMarked());
			}
			break;
			}
		default:
			inherited::MessageReceived(msg);
			break;
	}
}

/*--------------------------------------------------------*/

bool TWindow::QuitRequested()
{
	/*
	 The window was asked to quit/close. Send a message
	 to our partner, giving him the ball if we've
	 currently got it.
	*/
	BMessage	*m = new BMessage(BB_GOODBYE);
	if (fBall) {
		fBall->Quit();
		fBall = NULL;
		m->AddBool("ball", TRUE);
	}
	my_app->SendToPartner(m);

	// Tell the app to go ahead and quit.
	my_app->PostMessage(B_QUIT_REQUESTED);
	return TRUE;
}

/*--------------------------------------------------------*/

void TWindow::WindowsMoved(BRect window_frame, BRect partner_frame)
{
	// Either we moved or the partner. So recalculate the gap.
	CalcGapAngles(window_frame, partner_frame, &fGapStart, &fGapEnd);
	if (fBall)
		fBall->SetGap(fGapStart, fGapEnd);
	Update();
}

/*--------------------------------------------------------*/

void TWindow::Update(BRect rect)
{
	// redraw window, only blitting the given rect.
	DrawOffScreen(fOffView);
	fMainView->Draw(rect);
}

/*--------------------------------------------------------*/

void TWindow::Update()
{
	// redraw window, blitting the entire window
	DrawOffScreen(fOffView);
	fMainView->Draw(fMainView->Bounds());
}

/*--------------------------------------------------------*/

void TWindow::DrawOffScreen(BView *v)
{
	/*
	 Do the actual drawing of the 'room'. The walls will get
	 drawn (with a gap if there is one) and the ball will
	 be placed in the proper position.
	*/
	fBitmap->Lock();

	BRect b = v->Bounds();
	
	v->SetPenSize(2.0);
	v->SetHighColor(100,100,0);
	v->FillRect(b, &B_SOLID_LOW);

	b.bottom -= 1.0;
	b.right -= 1.0;

	// draw the appropriate border - with or without a 'gap'
	if (fGapStart == -1.0)
		v->StrokeRect(b);
	else
		DrawFrame(v, b, fGapStart, fGapEnd);

	if (fBall)
		fBall->Draw(v);

	v->Sync();			// flush all drawing to the bitmap

	fBitmap->Unlock();
}

/*--------------------------------------------------------*/
/*--------------------------------------------------------*/
/*--------------------------------------------------------*/

B_DEFINE_CLASS_INFO(TBitmapView, BView);

TBitmapView::TBitmapView(BRect frame, BBitmap *bitmap)
	: BView(frame, "", B_FOLLOW_NONE, B_WILL_DRAW)
{
	/*
	 The only job of this view is to blit the offscreen
	 bits into the onscreen view.
	*/
	fBitmap = bitmap;
}

/*--------------------------------------------------------*/

void TBitmapView::Draw(BRect update)
{
	DrawBitmap(fBitmap, update, update);
}

/*--------------------------------------------------------*/
/*--------------------------------------------------------*/
/*--------------------------------------------------------*/

B_DEFINE_CLASS_INFO(TBall, BLooper);

TBall::TBall(TWindow *window, BRect bounds, BPoint center,
	float radius, BPoint speed)
	: fLock()
{
	/*
	 The ball object actually runs in its own thread so
	 that it is independent of any window. It sends the
	 window messages as needed.
	*/
	fWindow = window;
	fCenter = center;
	fBoundary = bounds;
	fRadius = radius;
	fSpeed = speed;
	fLength = sqrt((fSpeed.x*fSpeed.x) + (fSpeed.y*fSpeed.y));
	fSleep = 40.0;
	fPercentRemaining = 1.0;
	fGapStart = fGapEnd = -1.0;
	fEnabled = TRUE;

	Run();
	// post initial message to get things rolling
	PostMessage(BB_TICK);
}

/*--------------------------------------------------------*/

void TBall::Draw(BView *view)
{
	// The balls draws itself in the given view
	Lock();
	rgb_color	c = view->HighColor();

	view->SetPenSize(1.0);
	view->SetHighColor(150, 30, 30);
	view->FillArc(fCenter, fRadius, fRadius, 0, 360);
	view->SetHighColor(c);
	Unlock();
}

/*--------------------------------------------------------*/

void TBall::Tick()
{
	/*
	 This method is repeatedly call to simulate the ball's
	 motion. The ball will wake up every fSnooze millisecs
	 and update its position.

	 So basically this is where the hit-testing and logic
	 lives for bouncing the ball around the window and
	 deciding when the ball hits the 'gap' and gets
	 teleported over to the partner application.
	*/
	float	angle;
	bool	hit_hole = FALSE;
	BRect	updateRect;

	// do nothing if the ball is disabled.
	if (!fEnabled)
		return;

	if ((rand() % 3000) == 0) {
		// add a little random bahavior to the ball.
		long r = (rand() % 3) - 1.0;
		fSpeed.x += r;
		fSpeed.y -= r;
		fLength = sqrt((fSpeed.x*fSpeed.x) +
						(fSpeed.y*fSpeed.y));
	}
/*--------------------------------------------------------*/

	/*
	 To minimize blitting we'll calculate the rect that
	 encloses the before & after locations of the ball
	 and only redraw that portion of the screen. That's
	 what 'updateRect' will do.
	*/
	updateRect = Bounds();

	// This will move the ball to its new location.
	NextPosition(&hit_hole, &angle);

	updateRect = updateRect | Bounds();
	updateRect.InsetBy(-1, -1);

	// inform the window to redraw the window.
	BMessage	*msg = new BMessage(BB_DRAW);
	msg->AddRect("update", updateRect);
	fWindow->PostMessage(msg);

	if (hit_hole) {
		/*
		 The 'gap' was hit. So we package up the info
		 like speed and relative location, which gives
		 our partner enough information to have the
		 ball appear in the correct place.
		*/
		float rel_loc = CalcGapPosition(angle, fGapStart,
			fGapEnd);

		BMessage *msg = new BMessage(BB_HIT_HOLE);
		msg->AddPoint("speed", fSpeed);
		msg->AddFloat("rel_loc", rel_loc);
		fWindow->PostMessage(msg);
		fEnabled = FALSE;
	}
}

/*--------------------------------------------------------*/

void TBall::MessageReceived(BMessage *msg)
{
	switch (msg->what) {
		case BB_TICK:
			{
			Lock();
			Tick();

			/*
			 Adjust the 'snooze' duration to keep the
			 animation flowing smoothly.
			*/
			float sleep_for = fPercentRemaining * fSleep;
			Unlock();

			snooze(sleep_for * 1000);
			PostMessage(BB_TICK);
			break;
			}
	}
}

/*--------------------------------------------------------*/

BRect TBall::Bounds()
{
	return BRect(fCenter.x - fRadius, fCenter.y - fRadius,
				fCenter.x + fRadius, fCenter.y + fRadius);
}

/*--------------------------------------------------------*/

void TBall::SetGap(float start, float end)
{
	Lock();
	fGapStart = start;
	fGapEnd = end;
	Unlock();
}

/*--------------------------------------------------------*/

void TBall::NextPosition(bool *hit_hole, float *angle)
{
	/*
	 Move the ball to it's next position.
	*/
	BPoint	move_by(fSpeed.x * fPercentRemaining,
					fSpeed.y * fPercentRemaining);
	BPoint	p1 = fCenter;
	BPoint	p2 = fCenter + move_by;
	BRect	b = fBoundary;

	b.InsetBy(BPoint(fRadius, fRadius));

	if (!b.Contains(p2)) {
		/*
		 The ball is now outside the boundary - so it either
		 hit a wall of it is going through the 'gap'.
		 Calculate the intersection point and the ball's
		 new speed (i.e. direction after the bounce).
		*/

//		PRINT(("\nCalcIntersection\n"));
//		PRINT_OBJ(b); PRINT_OBJ(p1); PRINT_OBJ(p2); PRINT_OBJ(fSpeed);

		BPoint new_speed = fSpeed;
		BPoint hit = CalcIntersection(p1, p2, b, &new_speed);

		// check to see if the ball is in the 'gap'
		if (fGapStart != -1.0) {
			BPoint	center = rect_center(fBoundary);
			long	unused;

			*angle = CalcAngle(center, hit, &unused);
			if (fGapEnd > fGapStart)
				*hit_hole = ((fGapStart < *angle) &&
							(*angle < fGapEnd));
			else
				*hit_hole = (*angle > fGapStart) ||
							(*angle < fGapEnd);
		}

		p2 = hit;

		if (!*hit_hole) {
			/*
			 Didn't hit the gap. The ball's new position
			 will be at the point of impact with the wall.
			 We do this so that the animation will show
			 the ball actually hitting the wall. Because
			 of this will need to adjust the 'snooze'
			 period until the next animation to keep to
			 speed of the ball relatively constant.
			*/
			fSpeed = new_speed;
			float partial = segment_length(p1, hit);
			fPercentRemaining -= (partial / fLength);
		}
	} else {
		// Didn't hit the wall.
		fPercentRemaining = 1.0;

		/*
		 However, if the ball stopped exactly on an edge,
		 reverse it's direction for next time.
		*/
		if ((p2.y == b.top) || (p2.y == b.bottom))
			fSpeed.y *= -1.0;
		if ((p2.x == b.left) || (p2.x == b.right))
			fSpeed.x *= -1.0;
	}

	fCenter = p2;	// finally update the ball's position

#if DEBUG
	if (!b.Contains(fCenter)) {
		PRINT(("fCenter is outside of boundary!\n"));
		PRINT_OBJ(b);
		PRINT_OBJ(fCenter);
		ASSERT(0);
	}
#endif
}
                                                                                                                                                                                                                                                                 main.h                                                                                                 644    2013      62         6372  6072610126   5230                                                                                                                                                                                                                                                                                                                                                                      /*
 File:	main.h

 Class declarations for the BeBox bouncing ball demo.

 Copyright 1995, Be Incorporated, All Rights Reserved.
*/

#ifndef MAIN_H
#define MAIN_H

#ifndef _APPLICATION_H
#include <Application.h>
#endif
#ifndef _LOCKER_H
#include <Locker.h>
#endif
#ifndef _MESSENGER_H
#include <Messenger.h>
#endif
#ifndef _WINDOW_H
#include <Window.h>
#endif
#ifndef _BITMAP_H
#include <Bitmap.h>
#endif

enum {
	/*
	 enum's for the various messages sent within and
	 between instances of the BeBounce application.
	*/

	BB_HELLO		= 'bhel',
	BB_GOODBYE		= 'bgby',
	BB_WINDOW_MOVED	= 'bwmv',
	BB_BALL			= 'ball',
	BB_DRAW			= 'draw',
	BB_HIT_HOLE		= 'hole',
	BB_TICK			= 'tick',
	BB_STOP			= 'stop'
};

const ulong MY_SIG = 'bbll';
const float	RADIUS = 10.0;

class TWindow;
class TBitmapView;
class TBall;

/*--------------------------------------------------------*/

class TBounceApp : public BApplication {
	B_DECLARE_CLASS_INFO(BApplication);

public:
					TBounceApp();
virtual				~TBounceApp();

virtual	void		MessageReceived(BMessage *msg);
		void		InitPartner(thread_id tid);
		void		RemovePartner();
		bool		SendToPartner(BMessage *msg);
		bool		SendPosition(BRect rect);

private:
		TWindow		*fWindow;
		BMessenger	*fPartner;
};

TBounceApp	*my_app;

/*--------------------------------------------------------*/

class TWindow : public BWindow {
	B_DECLARE_CLASS_INFO(BWindow);
public:
					TWindow(BRect frame, const char *title,
							bool with_ball);
virtual				~TWindow();

virtual	void		MessageReceived(BMessage *msg);
virtual bool		QuitRequested();
virtual void		FrameMoved(BPoint new_position);
		void		PartnerMoved(BRect pos);
		void		Update(BRect rect);
		void		Update();
		void		DrawOffScreen(BView *v);
		void		WindowsMoved(BRect window_frame,
								BRect partner_frame);
		void		PartnerGone();
		void		AddBall();
		void		AddBall(float rel_location,
							BPoint speed);

private:

		TBitmapView	*fMainView;
		BRect		fMyFrame;
		BRect		fPartnerFrame;
		float		fGapStart;
		float		fGapEnd;
		BBitmap		*fBitmap;
		BView		*fOffView;
		TBall		*fBall;
};

/*--------------------------------------------------------*/

class TBitmapView : public BView {
	B_DECLARE_CLASS_INFO(BView);

public:
					TBitmapView(BRect frame, BBitmap *bitmap);

virtual	void		Draw(BRect updateRect);

private:
		BBitmap		*fBitmap;
};

/*--------------------------------------------------------*/

class TBall : public BLooper {
	B_DECLARE_CLASS_INFO(BLooper);

public:
					TBall(TWindow *window, BRect bounds,
						BPoint center, float radius,
						BPoint speed);

		void		Draw(BView *view);
virtual	void		MessageReceived(BMessage *msg);

		void		Lock();
		void		Unlock();
		void		SetGap(float start, float end);
		void		SetEnabled(bool state);
		bool		IsEnabled();
		BRect		Bounds();

private:

		void		NextPosition(bool *hit, float *angle);
		void		Tick();

		BLocker		fLock;
		TWindow		*fWindow;
		BPoint		fCenter;
		float		fRadius;
		BPoint		fSpeed;
		float		fSleep;
		BRect		fBoundary;
		float		fPercentRemaining;
		float		fLength;
		float		fGapStart;
		float		fGapEnd;
		bool		fEnabled;
};

inline void TBall::Lock()
	{ fLock.Lock(); }

inline void TBall::Unlock()
	{ fLock.Unlock(); }

inline void TBall::SetEnabled(bool state)
	{ fEnabled = state; }

inline bool TBall::IsEnabled()
	{ return fEnabled; }

#endif
                                                                                                                                                                                                                                                                      make.ppc.deps                                                                                          644    2013      62         5734  6072610127   6510                                                                                                                                                                                                                                                                                                                                                                      ### This file generated by Make - do not modify ###

obj/Crunch.o: \
	$(BUILDHOME)/src/apps/bounce/Crunch.cpp \
	$(BUILDHOME)/headers/support/Debug.h \
	$(BUILDHOME)/headers/support/SupportDefs.h \
	$(BUILDHOME)/headers/support/Errors.h \
	$(BUILDHOME)/metro/inc/limits.h \
	$(BUILDHOME)/metro/inc/stdarg.h \
	$(BUILDHOME)/metro/inc/va_list.h \
	$(BUILDHOME)/metro/inc/stdio.h \
	$(BUILDHOME)/metro/inc/ansi_parms.h \
	$(BUILDHOME)/metro/inc/size_t.h \
	$(BUILDHOME)/metro/inc/null.h \
	$(BUILDHOME)/headers/os/OS.h \
	$(BUILDHOME)/src/apps/bounce/Crunch.h \
	$(BUILDHOME)/metro/inc/math.h \
	$(BUILDHOME)/headers/interface/Point.h \
	$(BUILDHOME)/headers/interface/Rect.h \
	$(BUILDHOME)/headers/interface/View.h \
	$(BUILDHOME)/headers/interface/InterfaceDefs.h \
	$(BUILDHOME)/headers/app/Receiver.h \
	$(BUILDHOME)/headers/support/Object.h \
	$(BUILDHOME)/headers/support/ClassInfo.h \
	$(BUILDHOME)/headers/app/Message.h \
	$(BUILDHOME)/metro/inc/stddef.h \
	$(BUILDHOME)/metro/inc/wchar_t.h \
	$(BUILDHOME)/headers/app/AppDefs.h \
	$(BUILDHOME)/headers/storage/StorageDefs.h \
	$(BUILDHOME)/headers/support/Locker.h \
	$(BUILDHOME)/headers/app/Looper.h \
	$(BUILDHOME)/headers/app/Clipboard.h \
	$(BUILDHOME)/headers/interface/Picture.h

obj/main.o: \
	$(BUILDHOME)/src/apps/bounce/main.cpp \
	$(BUILDHOME)/headers/support/Debug.h \
	$(BUILDHOME)/headers/support/SupportDefs.h \
	$(BUILDHOME)/headers/support/Errors.h \
	$(BUILDHOME)/metro/inc/limits.h \
	$(BUILDHOME)/metro/inc/stdarg.h \
	$(BUILDHOME)/metro/inc/va_list.h \
	$(BUILDHOME)/metro/inc/stdio.h \
	$(BUILDHOME)/metro/inc/ansi_parms.h \
	$(BUILDHOME)/metro/inc/size_t.h \
	$(BUILDHOME)/metro/inc/null.h \
	$(BUILDHOME)/headers/os/OS.h \
	$(BUILDHOME)/metro/inc/string.h \
	$(BUILDHOME)/metro/inc/stdlib.h \
	$(BUILDHOME)/metro/inc/div_t.h \
	$(BUILDHOME)/metro/inc/wchar_t.h \
	$(BUILDHOME)/metro/inc/math.h \
	$(BUILDHOME)/src/apps/bounce/main.h \
	$(BUILDHOME)/headers/app/Application.h \
	$(BUILDHOME)/headers/storage/StorageDefs.h \
	$(BUILDHOME)/headers/interface/InterfaceDefs.h \
	$(BUILDHOME)/headers/interface/Rect.h \
	$(BUILDHOME)/headers/interface/Point.h \
	$(BUILDHOME)/headers/app/Looper.h \
	$(BUILDHOME)/headers/app/Receiver.h \
	$(BUILDHOME)/headers/support/Object.h \
	$(BUILDHOME)/headers/support/ClassInfo.h \
	$(BUILDHOME)/headers/app/Message.h \
	$(BUILDHOME)/metro/inc/stddef.h \
	$(BUILDHOME)/headers/app/AppDefs.h \
	$(BUILDHOME)/headers/support/Locker.h \
	$(BUILDHOME)/headers/interface/Window.h \
	$(BUILDHOME)/headers/interface/View.h \
	$(BUILDHOME)/headers/app/Clipboard.h \
	$(BUILDHOME)/headers/interface/Picture.h \
	$(BUILDHOME)/headers/app/MessageQueue.h \
	$(BUILDHOME)/headers/support/List.h \
	$(BUILDHOME)/headers/app/Messenger.h \
	$(BUILDHOME)/headers/app/Roster.h \
	$(BUILDHOME)/headers/interface/PopUpMenu.h \
	$(BUILDHOME)/headers/interface/Menu.h \
	$(BUILDHOME)/headers/interface/Bitmap.h \
	$(BUILDHOME)/src/apps/bounce/Crunch.h \
	$(BUILDHOME)/headers/interface/MenuBar.h \
	$(BUILDHOME)/headers/interface/MenuItem.h
                                    makefile                                                                                               644    2013      62          660  6072610130   5600                                                                                                                                                                                                                                                                                                                                                                      #
#	HeapWatch Be Makefile
#

include $(BUILDHOME)/buildprefs/make.pre

APP_NAME=BeBounce
TARGET	:= $(OBJ_DIR)/$(APP_NAME)

CFLAGS := $(CFLAGS) -O4

SRCS = \
	Crunch.cpp \
	main.cpp

OBJS := $(SRCS_LIST_TO_OBJS)

default: $(TARGET)
		cp $(TARGET) $(APP_NAME)

$(TARGET):	$(OBJ_DIR) $(OBJS)
		$(LD) -o $@ $(OBJS) $(LDFLAGS)

delapp:
	-rm $(APPNAME)

tar: delapp	$(APPNAME)
	tar c $(APPNAME)

include $(BUILDHOME)/buildprefs/make.post
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                