/*
    ListWindow.cpp

    Copyright  1997 Levin-Delson Software.  All Rights Reserved.

    $Header: /Bedazzle/dvlp/DbEd/RCS/ListWindow.cpp,v 1.22 1997/01/03 20:17:36 adamld Exp $
*/

#include "ListWindow.h"

// Local protos.
BMenuBar *createMenuBar( BRect frame, BHandler *handler, ulong flags );

// This array is used to give the windows different colors.
rgb_color ColorArray[] =
{
	{ (uchar) 95, (uchar) 95, (uchar)255, 0 },  // Blue
	{ (uchar)255, (uchar) 95, (uchar) 95, 0 },  // Red
	{ (uchar) 63, (uchar)127, (uchar) 63, 0 },  // Green
	{ (uchar) 99, (uchar) 99, (uchar) 99, 0 },  // Grey
	{ (uchar)191, (uchar) 31, (uchar)159, 0 },  // Purple
	{ (uchar)  0, (uchar)127, (uchar)127, 0 },  // Aqua
	{ (uchar)127, (uchar)127, (uchar) 31, 0 },  // Army green
	{ (uchar)159, (uchar) 95, (uchar) 31, 0 }   // Brown
};

long numberofColors = (long)( sizeof( ColorArray ) / sizeof( rgb_color ) );


// The ListView can either draw in its native color, or in a rainbow of colors.
void MyListView :: DrawItem( BRect updateRect, long index )
{
	rgb_color saveColor = HighColor();
	
	if ( TRUE == doRainbow )
	{
		SetHighColor( ColorArray[ index % numberofColors ] );
	}
	
	BListView::DrawItem( updateRect, index );
	
	if ( TRUE == doRainbow )
	{
		SetHighColor( saveColor );
	}
}


ListWindow :: ListWindow( BRect frame, ulong menuFlags, BHandler *callersHandler, rgb_color *rgb_color, bool rainbow ) : BWindow( frame, NULL, B_TITLED_WINDOW, ( B_WILL_ACCEPT_FIRST_CLICK | B_WILL_FLOAT ) )
{
	status = B_ERROR;
	menuBar = NULL;
	pendingItems = NULL;
	supportsSelection = FALSE;
	supportsInvocation = FALSE;
	color = rgb_color;

	// Create the menubar if needed.	
	float menuHeight = 0.0;
	
	if ( LW_MENUFLAG_NONE != menuFlags )
	{
		frame.OffsetTo( B_ORIGIN );
		menuBar = createMenuBar( frame, this, menuFlags );
		if ( NULL != menuBar )
		{
			AddChild( menuBar );
			if ( TRUE == Lock() )
			{
				menuHeight = menuBar->Frame().Height();
				Unlock();
				
				if ( 0.0 < menuHeight )
				{
					menuHeight += 1.0;
				}
			}
		}
	}

	// This wierd offset is because of the ListView.
	frame.OffsetTo( -2.0, menuHeight );
	frame.SetRightBottom( BPoint( ( 2.0 + frame.right - B_V_SCROLL_BAR_WIDTH ), ( frame.bottom - menuHeight ) ) );
	
	listView = new MyListView( frame, LW_NAME_LISTVIEW, color, rainbow );
	if ( NULL != listView )
	{				
		BMessage *bMessage = new BMessage( LW_MESSAGEWHAT_ITEM_SELECTED );
		if ( NULL != bMessage )
		{
 			listView->SetSelectionMessage( bMessage );
			bMessage = new BMessage( LW_MESSAGEWHAT_ITEM_INVOKED );
			if ( NULL != bMessage )
			{
				// Target the window handler and set the message.
	 			listView->SetInvocationMessage( bMessage );
				if ( B_NO_ERROR <= listView->SetTarget( this ) )
				{
					BScrollView *bScrollView = new BScrollView( LW_NAME_SCROLLVIEW, cast_as( listView, BView ), B_FOLLOW_ALL, 0, FALSE, TRUE, FALSE ); 
					if ( NULL != bScrollView )
					{
						AddChild( bScrollView );
						parentHandler = callersHandler;
	
						pendingItems = new BList();
						if ( NULL != pendingItems )
						{
							InsureOnScreen();
							status = B_NO_ERROR;
						}
					}
				}
			}
		}
		
		// If there was an error, clean up.
		if ( B_NO_ERROR > status )
		{
			// Cause the ListView to free the message.
			listView->SetInvocationMessage( NULL );
			listView->SetSelectionMessage( NULL );

			delete listView;
		}
	}
}


ListWindow :: ~ListWindow()
{
	// We have to free the items on the pending list.  The items on the ListView list are freed automatically.
	if ( NULL != pendingItems )
	{
		delete pendingItems;
	}
}


/*
	QuitRequested()
	
	Ask our parent for permission to quit.
*/
bool ListWindow :: QuitRequested( void )
{
	bool result = TRUE;
	
	// Need to determine if *I* can quit.  (Not implemented yet.)
	
	if ( ( FALSE == CanChildrenQuit() ) || ( FALSE == CanPendingQuit() ) )
	{
		BAlert *bAlert = new BAlert( "CloseWarning", "If you close this window you will lose all changes made since you last saved.", "Close anyway", "Cancel", NULL, B_WIDTH_FROM_WIDEST, B_WARNING_ALERT );
		if ( NULL != bAlert )
		{
			bAlert->SetShortcut( 1L, B_ESCAPE );
			if ( 1L == bAlert->Go() )
			{
				result = FALSE;
			}
		}
	}
	
	if ( TRUE == result )
	{
		if ( NULL != parentHandler )
		{
			BMessenger *bMessenger = new BMessenger( parentHandler );
			if ( NULL != bMessenger ) 
			{
				BMessage *bMessage = new BMessage( LW_MESSAGEWHAT_CHILD_QUIT_REQUESTED );
				if ( NULL != bMessage )
				{
					if ( B_NO_ERROR <= bMessage->AddLong( LW_MESSAGEITEM_CHILD, (long)this ) )
					{
						if ( B_ERROR <= bMessenger->SendMessage( bMessage ) )
						{
							// If we were able to inform our parent that we want to quit, wait for them to tell us to do so.
							result = FALSE;
	
							bMessage = NULL;
						}
					}
					
					// Delete the message iff it never got sent.
					if ( NULL != bMessage )
					{
						delete bMessage;
					}
				}
				
				delete bMessenger;
			}
		}
	}
	
	return( result );
}


void ListWindow :: MessageReceived( BMessage *message )
{
	if ( NULL != message )
	{
		long index;
		BHandler *bHandler;
		
		switch ( message->what )
		{
			// A message was dragged onto us.
			case ( LW_MESSAGEWHAT_DND ):
				ItemDropped( message );
			break;
			
			case ( LW_MESSAGEWHAT_ITEM_SELECTED ):
				if ( TRUE == SupportsSelection() )
				{
					index = message->FindLong( BE_MESSAGEITEM_INDEX );
					if ( B_NO_ERROR <= message->Error() )
					{
						ItemSelected( index );
					}
				}
			break;
			
			case ( LW_MESSAGEWHAT_ITEM_INVOKED ):
				if ( TRUE == SupportsInvocation() )
				{
					index = message->FindLong( BE_MESSAGEITEM_INDEX );
					if ( B_NO_ERROR <= message->Error() )
					{
						ItemInvoked( index );
					}
				}
			break;
			
			// The Edit menu button was hit.				
			case ( LW_MESSAGEWHAT_MENUEDIT ):
				index = listView->CurrentSelection();
				if ( 0L <= index )
				{
					(void)EditItem( index );
				}
			break;

			// The New menu button was hit.				
			case ( LW_MESSAGEWHAT_MENUNEW ):
				(void)NewItem();
			break;

			// The Delete menu button was hit.				
			case ( LW_MESSAGEWHAT_MENUDELETE ):
				index = listView->CurrentSelection();
				if ( 0L <= index )
				{
					(void)DeleteItem( index );
				}
			break;

			/* An EditWindow is done.  Try to accept the data (there may be problems such as name clashes) and
			   then tell the EditWindow what to do.
			*/
			case ( LW_MESSAGEWHAT_ACCEPTNEW ):
				bHandler = (BHandler *)message->FindLong( LW_MESSAGEITEM_HANDLER );
				if ( ( NULL != bHandler ) && ( B_NO_ERROR <= message->Error() ) )
				{
					long command = LW_MESSAGEWHAT_QUIT;
					
					NamePlus *namePlus = FindPendingItem( bHandler );
					if ( NULL != namePlus )
					{
						switch ( AcceptNew( namePlus, message ) )
						{
							case ( LW_ACCEPTRESULT_ACCEPT ):
								command = LW_MESSAGEWHAT_QUIT;
								if ( TRUE == RemovePendingItem( namePlus ) )
								{
									AddItem( namePlus );
								}
							break;
							
							case ( LW_ACCEPTRESULT_RETRY ):
								command = LW_MESSAGEWHAT_ACTIVATE;
							break;

							case ( LW_ACCEPTRESULT_CANCEL ):
							default:
								command = LW_MESSAGEWHAT_QUIT;
								if ( TRUE == RemovePendingItem( namePlus ) )
								{
									disposeNamePlus( namePlus );
								}
							break;
						}
					}
					
					BMessenger *bMessenger = new BMessenger( bHandler );
					if ( NULL != bMessenger )
					{
						bMessenger->SendMessage( command );
						delete bMessenger;
					}
				}
			break;
			
			// An EditWindow was cancelled; forget the NamePlus and tell the EditWindow to quit.	
			case ( LW_MESSAGEWHAT_CANCELNEW ):
				bHandler = (BHandler *)message->FindLong( LW_MESSAGEITEM_HANDLER );
				if ( ( NULL != bHandler ) && ( B_NO_ERROR <= message->Error() ) )
				{
					NamePlus *namePlus = FindPendingItem( bHandler );
					if ( NULL != namePlus )
					{
						if ( TRUE == RemovePendingItem( namePlus ) )
						{
							disposeNamePlus( namePlus );
						}
					}
					
					BMessenger *bMessenger = new BMessenger( bHandler );
					if ( NULL != bMessenger )
					{
						bMessenger->SendMessage( LW_MESSAGEWHAT_QUIT );
						delete bMessenger;
					}
				}
			break;
			
			// A child window has requested to quit.  Tell them to do so, then mark our NamePlus entry as available.
			case ( LW_MESSAGEWHAT_CHILD_QUIT_REQUESTED ):
				BHandler *childHandler = (BHandler *)message->FindLong( LW_MESSAGEITEM_CHILD );
				if ( ( NULL != childHandler ) && ( B_NO_ERROR <= message->Error() ) )
				{
					if ( TRUE == ChildQuitRequested( childHandler ) )
					{
						BMessenger *bMessenger = new BMessenger( childHandler );
						if ( NULL != bMessenger )
						{
							if ( B_NO_ERROR <= bMessenger->SendMessage( LW_MESSAGEWHAT_QUIT ) )
							{
								// Mark this list item inactive.
								NamePlus *namePlus= FindItem( childHandler );
								if ( NULL != namePlus )
								{
									namePlus->np_Handler = NULL;
								}
							}
							
							delete bMessenger;
						}
					}
				}
			break;
			
			/* I've been asked if I can quit.  Ask all child and pending (edit) windows if they can quit.
			   if all of them can quit, I can quit.
			*/
			case ( LW_MESSAGEWHAT_CAN_YOU_QUIT ):
				BMessage *replyMessage = new BMessage( LW_MESSAGEWHAT_OK2QUIT );
				if ( NULL != replyMessage )
				{				
					if ( B_NO_ERROR <= replyMessage->AddBool( LW_MESSAGEITEM_OK2QUIT, CanYouQuit() ) )
					{
						if ( B_NO_ERROR <= message->SendReply( replyMessage ) )
						{
							replyMessage = NULL;
						}
					}
					
					// Only delete the message if it _wasn't_ sent. */
					if ( NULL != replyMessage )
					{
						delete( replyMessage );
					}
				}
			break;

			case ( LW_MESSAGEWHAT_ACTIVATE ):
				Activate( TRUE );
			break;

			/* Quit.  There is no backing out of this.  If the caller wanted to test it quitting was ok, they are expected
			   to have first sent "LW_MESSAGEWHAT_CAN_YOU_QUIT".
			*/
			case ( LW_MESSAGEWHAT_QUIT ):
				if ( FALSE == IsHidden() )
				{
					Hide();
				}
				QuitChildren();
				QuitPending();
				Quit();
			break;
			
			default:
                // If we don't recognize the message, pass it to the parent class.
                BWindow::MessageReceived( message );
			break;
		}
	}
}


// Stubs for functions expected to be replaced if needed.
void ListWindow :: ItemDropped( BMessage * )
{
}


void ListWindow :: ItemSelected( long )
{
}


void ListWindow :: ItemInvoked( long )
{
}


long ListWindow :: EditItem( long )
{
	long result = B_NO_ERROR;
	
	return( result );
}


long ListWindow :: NewItem( void )
{
	long result = B_NO_ERROR;
	
	return( result );
}


long ListWindow :: DeleteItem( long )
{
	long result = B_NO_ERROR;
	
	return( result );
}


long ListWindow :: AcceptNew( NamePlus *namePlus, BMessage *message )
{
	long result = LW_ACCEPTRESULT_CANCEL;
	
	if ( ( NULL != namePlus ) && ( NULL != message ) )
	{
		result = LW_ACCEPTRESULT_ACCEPT;
	}
	
	return( result );
}


bool ListWindow :: ChildQuitRequested( BHandler * )
{
	return( TRUE );
}


void ListWindow :: QuitChildren( void )
{
	NamePlus *namePlus;
	
	do
	{
		namePlus = RemoveItem( 0L );
		if ( ( NULL != namePlus ) && ( NULL != namePlus->np_Handler ) )
		{
			BMessenger *bMessenger = new BMessenger( namePlus->np_Handler );
			if ( NULL != bMessenger )
			{
				(void)bMessenger->SendMessage( LW_MESSAGEWHAT_QUIT );
				
				delete bMessenger;
			}
		}
	}
	while( NULL != namePlus );
}


void ListWindow :: QuitPending( void )
{
	NamePlus *namePlus;
		
	do
	{
		namePlus = RemovePendingItem( 0L );
		if ( ( NULL != namePlus ) && ( NULL != namePlus->np_Handler ) )
		{
			BMessenger *bMessenger = new BMessenger( namePlus->np_Handler );
			if ( NULL != bMessenger )
			{
				(void)bMessenger->SendMessage( LW_MESSAGEWHAT_QUIT );
				
				delete bMessenger;
			}
		}
	}
	while( NULL != namePlus );
}


// I can quit if all child and pending windows can quit.
bool ListWindow :: CanYouQuit( void )
{
	bool result;
	
	result = CanChildrenQuit();
	if ( TRUE == result )
	{
		result = CanPendingQuit();
	}
	
	return( result );
}


bool ListWindow :: CanChildrenQuit( void )
{
	bool childrenOK2Quit = TRUE;
	
	NamePlus *namePlus;

	long count = listView->CountItems();
	if ( 0L < count )
	{
		for ( long index = 0L; ( ( TRUE == childrenOK2Quit ) && ( index < count ) ); index++ )
		{
			namePlus = ItemAt( index );
			if ( ( NULL != namePlus ) && ( NULL != namePlus->np_Handler ) )
			{
				BMessenger *bMessenger = new BMessenger( namePlus->np_Handler );
				if ( NULL != bMessenger )
				{
					BMessage *replyMessage = NULL;
					if ( B_NO_ERROR <= bMessenger->SendMessage( LW_MESSAGEWHAT_CAN_YOU_QUIT, &replyMessage ) )
					{
						if ( NULL != replyMessage )
						{
							if ( B_NO_REPLY != replyMessage->what )
							{
								bool replyOK2Quit = replyMessage->FindBool( LW_MESSAGEITEM_OK2QUIT );
								if  ( B_NO_ERROR <= replyMessage->Error() )
								{
									if ( FALSE == replyOK2Quit )
									{
										childrenOK2Quit = FALSE;
									}
								}
							}
							delete replyMessage;
						}
					}
					
					delete bMessenger;
				}
			}
		}
	}
	
	return( childrenOK2Quit );
}


bool ListWindow :: CanPendingQuit( void )
{
	bool pendingOK2Quit = TRUE;
	
	NamePlus *namePlus;

	long count = pendingItems->CountItems();
	if ( 0L < count )
	{
		for ( long index = 0L; ( ( TRUE == pendingOK2Quit ) && ( index < count ) ); index++ )
		{
			namePlus = (NamePlus *)pendingItems->ItemAt( index );
			if ( ( NULL != namePlus ) && ( NULL != namePlus->np_Handler ) )
			{
				BMessenger *bMessenger = new BMessenger( namePlus->np_Handler );
				if ( NULL != bMessenger )
				{
					BMessage *replyMessage = NULL;
					if ( B_NO_ERROR <= bMessenger->SendMessage( LW_MESSAGEWHAT_CAN_YOU_QUIT, &replyMessage ) )
					{
						if ( NULL != replyMessage )
						{
							if ( B_NO_REPLY != replyMessage->what )
							{
								bool replyOK2Quit = replyMessage->FindBool( LW_MESSAGEITEM_OK2QUIT );
								if  ( B_NO_ERROR <= replyMessage->Error() )
								{
									if ( FALSE == replyOK2Quit )
									{
										pendingOK2Quit = FALSE;
									}
								}
							}
							delete replyMessage;
						}
					}
					
					delete bMessenger;
				}
			}
		}
	}
	
	return( pendingOK2Quit );
}


/* Here follow functions that mimic or add upon the BList counterparts, but apply to the ListView.
   The "...Pending..." versions operate on the Pending BList.
   Note that many of the functions handle the NamePlus structure.
*/
bool ListWindow :: AddItem( NamePlus *namePlus )
{
	bool result = FALSE;
	
	if ( NULL != namePlus )
	{
		if ( TRUE == Lock() )
		{
			result = listView->AddItem( (void *)namePlus );
			Unlock();
		}
	}
	
	return( result );
}


bool ListWindow :: AddPendingItem( NamePlus *namePlus )
{
	bool result = FALSE;
	
	if ( NULL != namePlus )
	{
		if ( TRUE == Lock() )
		{
			result = pendingItems->AddItem( (void *)namePlus );
			Unlock();
		}
	}
	
	return( result );
}


NamePlus *ListWindow :: RemoveItem( long index )
{
	NamePlus *namePlus = NULL;	
	
	if ( TRUE == Lock() )
	{
		namePlus = (NamePlus *)listView->RemoveItem( index );
		Unlock();
	}
	
	return( namePlus );
}


bool ListWindow :: RemoveItem( NamePlus *namePlus )
{
	bool result = FALSE;
		
	if ( NULL != namePlus )
	{
		if ( TRUE == Lock() )
		{
			result = listView->RemoveItem( namePlus );
			Unlock();
		}
	}
	
	return( result );
}


NamePlus *ListWindow :: RemovePendingItem( long index )
{
	NamePlus *namePlus = NULL;	
	
	if ( TRUE == Lock() )
	{
		namePlus = (NamePlus *)pendingItems->RemoveItem( index );
		Unlock();
	}
	
	return( namePlus );
}


bool ListWindow :: RemovePendingItem( NamePlus *namePlus )
{
	bool result = FALSE;
		
	if ( NULL != namePlus )
	{
		if ( TRUE == Lock() )
		{
			result = pendingItems->RemoveItem( namePlus );
			Unlock();
		}
	}
	
	return( result );
}


NamePlus *ListWindow :: ItemAt( long index )
{
	NamePlus *namePlus = NULL;	
	
	if ( TRUE == Lock() )
	{
		namePlus = (NamePlus *)listView->ItemAt( index );
		Unlock();
	}
	
	return( namePlus );
}


NamePlus *ListWindow :: FindItem( BHandler *handler )
{
	NamePlus *namePlus = NULL;
	
	long count = listView->CountItems();
	if ( 0L < count )
	{
		for ( long index = 0L; ( ( NULL == namePlus ) && ( index < count ) ); index++ )
		{
			namePlus = (NamePlus *)listView->ItemAt( index );
			if ( NULL != namePlus )
			{
				if ( handler != namePlus->np_Handler )
				{
					namePlus = NULL;
				}
			}
		}
	}
	
	return( namePlus );
}


NamePlus *ListWindow :: FindPendingItem( BHandler *handler )
{
	NamePlus *namePlus = NULL;
	
	long count = pendingItems->CountItems();
	if ( 0L < count )
	{
		for ( long index = 0L; ( ( NULL == namePlus ) && ( index < count ) ); index++ )
		{
			namePlus = (NamePlus *)pendingItems->ItemAt( index );
			if ( NULL != namePlus )
			{
				if ( handler != namePlus->np_Handler )
				{
					namePlus = NULL;
				}
			}
		}
	}
	
	return( namePlus );
}


NamePlus *ListWindow :: FindItem( char *name, void *id )
{
	NamePlus *namePlus = NULL;

	long count = listView->CountItems();
	if ( 0L < count )
	{
		for ( long index = 0L; ( ( NULL == namePlus ) && ( index < count ) ); index++ )
		{
			namePlus = (NamePlus *)listView->ItemAt( index );
			if ( NULL != namePlus )
			{
				if ( NULL != name )
				{
					if ( name != namePlus->np_Name )
					{
						namePlus = NULL;
					}
				}
				
				if ( id != namePlus->np_ID )
				{
					namePlus = NULL;
				}
			}
		}
	}
	
	return( namePlus );
}


// Insure that window appears on-scren.
void ListWindow :: InsureOnScreen( void )
{
	screen_info screenInfo;
	
	get_screen_info( &screenInfo );

	if ( TRUE == Lock() )
	{
		float minIndent = max( ( Frame().Width() / LW_PERCENTOFWINDOW_MUST_SHOW), ( 3 * B_V_SCROLL_BAR_WIDTH ) );
		float maxLeft = screenInfo.frame.right - minIndent;
		if ( Frame().left > maxLeft )
		{
			MoveTo( maxLeft, Frame().top );
		}
		
		Unlock();
	}
}


void ListWindow :: ResizeToFitText( long widthChars, long heightChars )
{
	float menuHeight = 0.0;
	
	if ( TRUE == Lock() )
	{
		if ( NULL != menuBar )
		{
			menuHeight = menuBar->Frame().Height();
		}
		ResizeTo( ( ( (float)( widthChars + 1 ) * listView->StringWidth( "M" ) ) + B_V_SCROLL_BAR_WIDTH ), ( ( (float)( heightChars + 1 ) * listView->GetItemHeight() ) + menuHeight ) );
		
		Unlock();
	}
}


BMenuBar *createMenuBar( BRect frame, BHandler *handler, ulong flags )
{
	BMenuBar *bMenuBar = NULL;

	if ( LW_MENUFLAG_NONE != flags )
	{
		bMenuBar = new BMenuBar( frame, LW_NAME_MENUBAR );
		if ( NULL != bMenuBar )
		{
			if ( B_NO_ERROR <= addMenuItem( bMenuBar, LW_MESSAGEWHAT_MENUEDIT, LW_LABEL_MENUEDIT, handler, ( LW_MENUFLAG_EDIT == ( flags & LW_MENUFLAG_EDIT ) ) ) )
			{
				flags &= ~LW_MENUFLAG_EDIT;
				if ( B_NO_ERROR <= addMenuItem( bMenuBar, LW_MESSAGEWHAT_MENUNEW, LW_LABEL_MENUNEW, handler, ( LW_MENUFLAG_NEW == ( flags & LW_MENUFLAG_NEW ) ) ) )
				{
					flags &= ~LW_MENUFLAG_NEW;
					if ( B_NO_ERROR <= addMenuItem( bMenuBar, LW_MESSAGEWHAT_MENUDELETE, LW_LABEL_MENUDELETE, handler, ( LW_MENUFLAG_DELETE == ( flags & LW_MENUFLAG_DELETE ) ) ) )
					{
						flags &= ~LW_MENUFLAG_DELETE;
					}
				}
			}
			
			if ( LW_MENUFLAG_NONE != flags )
			{
				delete bMenuBar;
				bMenuBar = NULL;
			}
		}
	}
	
	return ( bMenuBar );
}


NamePlus *createNamePlus( const char *name, void *id, BHandler *handler )
{
	NamePlus *namePlus = NULL;
	
	namePlus = (NamePlus *)malloc( sizeof( NamePlus ) );
	if ( NULL != namePlus )
	{
		(void) strncpy( namePlus->np_Name, ( ( NULL == name ) ? "" : name ), B_OS_NAME_LENGTH );
		namePlus->np_ID = id;
		namePlus->np_Handler = handler;
	}
											
	return( namePlus );
}


void disposeNamePlus( NamePlus *namePlus )
{
	if ( NULL != namePlus )
	{
		free( (void *)namePlus );
	}
}


long addMenuItem( BMenuBar *bMenuBar, long messageWhat, const char *menuItemLabel, BHandler *handler, bool isEnabled )
{
	long result = B_ERROR;
	
	if ( NULL != bMenuBar )
	{
		BMessage *bMessage = new BMessage( messageWhat );
		if ( NULL != bMessage )
		{
			BMenuItem *bMenuItem = new BMenuItem( menuItemLabel, bMessage );
			if ( NULL == bMenuItem )
			{
				delete bMessage;
			}
			else
			{
				result = B_NO_ERROR;
				
				if ( NULL != handler )
				{
					result = bMenuItem->SetTarget( handler );
				}
				
				if ( B_NO_ERROR <= result )
				{
					bMenuItem->SetEnabled( isEnabled );
					result = bMenuBar->AddItem( bMenuItem );
				}
			}
		}
	}
	
	return( result );
}

