/*
    DbEdWindow.cpp

    Copyright  1997 Levin-Delson Software.  All Rights Reserved.

    $Header: /Bedazzle/dvlp/DbEd/RCS/DbEdWindow.cpp,v 1.32 1997/01/03 20:17:36 adamld Exp $
*/

#include "DbEdWindow.h"

// The array of colors used in the ListViews are also used here.
extern rgb_color ColorArray[];
extern long      numberofColors;
	

/* The DbEd window lists the databases (volumes) on the BeBox.
   When created, the window obtains information on all currently available databases and adds them to the list.
   The application will inform this window if new volumes become available or existing ones are removed.
*/
DbEdWindow :: DbEdWindow( BRect frame, BHandler *callersHandler ) : ListWindow( frame, LW_MENUFLAG_NONE, callersHandler, NULL, TRUE )
{
	long    index = 0L;
	bool    done = FALSE;
	BVolume bVolume;
	ulong   volumeCount = 0UL;
	char    volumeName[ B_OS_NAME_LENGTH ];
	long    nameLength;
	long    longestLength = 0L;
	
	status = B_ERROR;
	
	SetTitle( DBED_NAME_APP " - Databases" );

	// One can double-click to open a database window, but not drag and drop (select).
	SetSupportsInvocation( TRUE );	
				
	do
	{
		bVolume = volume_at( index++ );
		if ( B_NO_ERROR > bVolume.Error() )
		{
			done = TRUE;
		}
		else
		{
			if ( B_NO_ERROR <= bVolume.GetName( volumeName ) )
			{
				// Get some info on the length of the strings.
				nameLength = strlen( volumeName );
				if ( longestLength < nameLength )
				{
					longestLength = nameLength;
				}
					
				NamePlus *namePlus = createNamePlus( volumeName, (void *)bVolume.ID(), (BHandler *)NULL );
				if ( NULL != namePlus )
				{
					if ( FALSE == AddItem( namePlus ) )
					{
						disposeNamePlus( namePlus );
					}
					else
					{
						volumeCount++;
					}
				}
				
			}
		}
	} while ( FALSE == done );
					
	status = B_NO_ERROR;

	ResizeToFitText( longestLength, volumeCount );
	
	if ( B_NO_ERROR <= status )
	{
		Show();
	}
}


void DbEdWindow :: MessageReceived( BMessage *message )
{
	if ( NULL != message )
	{
		switch ( message->what )
		{
			case ( DBED_MESSAGE_VOLUME_MOUNTED ):
			case ( DBED_MESSAGE_VOLUME_UNMOUNTED ):
			{
				bool          done = FALSE;
				long          index = 0L;
				long          volumeID;
				BVolume      *bVolume;
				
				do
				{
					volumeID = message->FindLong( DBED_MESSAGEITEMNAME_VOLUMEID, index++ );
					if ( B_NO_ERROR > message->Error() )
					{
						done = TRUE;
					}
					else
					{
						bVolume = new BVolume( volumeID );
						if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
						{
							if ( DBED_MESSAGE_VOLUME_MOUNTED == message->what )
							{
								char volumeName[ B_OS_NAME_LENGTH ];
								if ( B_NO_ERROR <= bVolume->GetName( volumeName ) )
								{
									NamePlus *namePlus = createNamePlus( volumeName, (void *)volumeID, (BHandler *)NULL );
									if ( NULL != namePlus )
									{
										if ( FALSE == AddItem( namePlus ) )
										{
											disposeNamePlus( namePlus );
										}
									}
								}
							}
							else
							{
								NamePlus *namePlus = FindItem( NULL, (void *)volumeID );
								if ( NULL != namePlus )
								{
									// If this item is active, tell it to quit.
									if ( NULL != namePlus->np_Handler )
									{
										BMessenger *bMessenger = new BMessenger( namePlus->np_Handler );
										if ( NULL != bMessenger )
										{
											if ( B_NO_ERROR <= bMessenger->SendMessage( LW_MESSAGEWHAT_QUIT ) )
											{
											}
											
											delete bMessenger;
										}
									}
										
									if ( TRUE == RemoveItem( namePlus ) )
									{
										disposeNamePlus( namePlus );
									}
								}
							}
							
							delete bVolume;
						}
					}
				} while ( FALSE == done );
			}
			break;
			
			default:
                // If we don't recognize the message, pass it to the parent class.
                ListWindow::MessageReceived( message );
			break;
		}
	}
}


// Open up a database window for the invoked item.
void DbEdWindow :: ItemInvoked( long index )
{ 
	// A database was selected.
	NamePlus *namePlus = ItemAt( index );
	if ( NULL != namePlus )
	{
		// If the data field is non-Null, this item's window already exists.
		if ( NULL != namePlus->np_Handler )
		{
			DatabaseWindow *databaseWindow = (DatabaseWindow *)namePlus->np_Handler;
			BMessenger *bMessenger = new BMessenger( cast_as( databaseWindow, BHandler ) );
			if ( NULL != bMessenger )
			{
				bMessenger->SendMessage( LW_MESSAGEWHAT_ACTIVATE );
			}
		}
		else
		{
			if ( TRUE == Lock() )
			{
				BRect myFrame = Frame();
				DatabaseWindow *databaseWindow = new DatabaseWindow( BRect( myFrame.right + B_V_SCROLL_BAR_WIDTH, myFrame.top, myFrame.right + 50.0, myFrame.top + 50.0 ), (long)namePlus->np_ID, this, &ColorArray[ index % numberofColors ] );
				if ( NULL != databaseWindow ) 
				{
					if ( B_NO_ERROR > databaseWindow->Status() )
					{
						delete databaseWindow;
					}
					else
					{
						namePlus->np_Handler = (BHandler *)databaseWindow;
					}
				}
				Unlock();
			}
		}
	}
}


// The Database window lists the tables in a database.
DatabaseWindow :: DatabaseWindow( BRect frame, long callersVolumeID, BHandler *callersHandler, rgb_color *rgb_color ) : ListWindow( frame, LW_MENUFLAG_NEW, callersHandler, rgb_color, FALSE )
{
	status = B_ERROR;

	SetSupportsSelection( TRUE );	
	SetSupportsInvocation( TRUE );	
	
	BVolume *bVolume = new BVolume( callersVolumeID );
	if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
	{
		volumeID = callersVolumeID;
		
		char volumeName[ B_OS_NAME_LENGTH ];
		if ( B_NO_ERROR <= bVolume->GetName( volumeName ) )
		{
			CreateTitle( volumeName );
		}

		BDatabase *bDatabase = bVolume->Database();
		if ( NULL != bDatabase )
		{
			status = B_NO_ERROR;
			
			bDatabase->Sync();
			
			long tables = bDatabase->CountTables();
			long nameLength;
			long longestLength = 0L;
			
			for ( long index = 0L; index < tables; index++ )
			{
				BTable *bTable = bDatabase->TableAt( index );
				if ( NULL != bTable )
				{
					// Get some info on the length of the strings.
					nameLength = strlen( bTable->Name() );
					if ( longestLength < nameLength )
					{
						longestLength = nameLength;
					}
					
					NamePlus *namePlus = createNamePlus( bTable->Name(), (void *)bTable, (BHandler *)NULL );
					if ( NULL != namePlus )
					{
						if ( FALSE == AddItem( namePlus ) )
						{
							disposeNamePlus( namePlus );
						}
					}
				}
			}
			
			if ( 0L < tables )
			{
				ResizeToFitText( longestLength, tables );
			}

			if ( B_NO_ERROR <= status )
			{
				InsureOnScreen();
				Show();
			}
		}	
		
		delete bVolume;
	}
}


void DatabaseWindow :: CreateTitle( const char *name )
{
	char title[ B_OS_NAME_LENGTH + 8 ];
	if ( NULL != title )
	{
		sprintf( title, "%s Tables", name );
		SetTitle( title );
	}
}


// The user has completed creating a new table.  Can we accept it?
long DatabaseWindow :: AcceptNew( NamePlus *namePlus, BMessage *message )
{
	long acceptResult = LW_ACCEPTRESULT_CANCEL;

	if ( ( NULL != namePlus ) && ( NULL != message ) )
	{
		char *name = (char *)message->FindString( EW_MESSAGEITEM_NAME );
		if ( ( NULL != name ) && ( B_NO_ERROR <= message->Error() ) )
		{
			if ( 0 == strlen( name ) )
			{
				BAlert *bAlert = new BAlert( "BadName", "Name was empty; table not added.", "OK" );
				if ( NULL != bAlert )
				{
					(void)bAlert->Go();
				}
			}
			else
			{
				BVolume *bVolume = new BVolume( volumeID );
				if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
				{
					char volumeName[ B_OS_NAME_LENGTH ];
					if ( B_ERROR > bVolume->GetName( volumeName ) )
					{
						strcpy( volumeName, "Unknown" );
					}
	
					BDatabase *bDatabase = bVolume->Database();
					if ( NULL != bDatabase )
					{
						acceptResult = LW_ACCEPTRESULT_ACCEPT;
									
						bDatabase->Sync();
		
						char alertString[ ( 2 * B_OS_NAME_LENGTH ) + 192 ];
						
						// Test for an existing (duplicate) entry.
						BTable *bTable = bDatabase->FindTable( name );
						if ( NULL != bTable )
						{
							(void)sprintf( alertString, "The \"%s\" database already has a table named \"%s\".  Duplicate names are allowed, but are strongly discouraged.\nPlease confirm the action.", volumeName, name );
							BAlert *bAlert = new BAlert( "ExistsWarning", alertString, "Add anyway", "Edit", "Cancel",  B_WIDTH_FROM_WIDEST, B_WARNING_ALERT );
							if ( NULL != bAlert )
							{
								bAlert->SetShortcut( 2L, B_ESCAPE );
								switch( bAlert->Go() )
								{
									case ( 0L ):
										acceptResult = LW_ACCEPTRESULT_ACCEPT;
									break;
									
									case ( 1L ):
										acceptResult = LW_ACCEPTRESULT_RETRY;
									break;
									
									case ( 2L ):
									default:
										acceptResult = LW_ACCEPTRESULT_CANCEL;
									break;
								}
							}
						}
						
						if ( LW_ACCEPTRESULT_ACCEPT == acceptResult )
						{
							BTable *bTable = NULL;
							
							char *inheritFrom = (char *)message->FindString( DBED_MESSAGEITEM_INHERIT );
							if ( ( NULL != inheritFrom ) && ( B_NO_ERROR <= message->Error() ) )
							{
								bTable = bDatabase->CreateTable( name, inheritFrom );
							}
							else
							{
								bTable = bDatabase->CreateTable( name );
							}
							
							if ( NULL != bTable )
							{
								(void) strncpy( namePlus->np_Name, name, B_OS_NAME_LENGTH );
								namePlus->np_ID = (void *)bTable;
								namePlus->np_Handler = NULL;
							}
							else
							{
								(void)sprintf( alertString, "Unable to create the \"%s\" table in the \"%s\" database.", name, volumeName );
								BAlert *bAlert = new BAlert( "CreationWarning", alertString, "Edit", "Cancel", NULL, B_WIDTH_FROM_WIDEST, B_WARNING_ALERT );
								if ( NULL != bAlert )
								{
									bAlert->SetShortcut( 1L, B_ESCAPE );
									if ( 0L == bAlert->Go() )
									{
										acceptResult = LW_ACCEPTRESULT_RETRY;
									}
									else
									{
										acceptResult = LW_ACCEPTRESULT_CANCEL;
									}
								}
							}
						}
					}
					
					delete bVolume;
				}
			}
		}
	}

	return( acceptResult );
}


// The user dragged and dropped another item onto us.  If it's a valid table, clone it (and all its fields) and add it to the database.
void DatabaseWindow :: ItemDropped( BMessage *bMessage )
{ 
	if ( NULL != bMessage )
	{
		long dropType = bMessage->FindLong( DBED_MESSAGEITEM_DROPTYPE );
		if ( B_NO_ERROR <= bMessage->Error() )
		{
			if ( DBED_DROPTYPE_TABLE != dropType )
			{
				BAlert *bAlert = new BAlert( "BadDND", "Only tables can be added to this database.", "Cancel" );
				if ( NULL != bAlert )
				{
					(void)bAlert->Go();
				}
			}
			else
			{
				bool success = FALSE;
				
				long droppedVolumeID = bMessage->FindLong( DBED_MESSAGEITEM_VOLUMEID );
				if ( B_NO_ERROR <= bMessage->Error() )
				{
					if ( droppedVolumeID == volumeID )
					{
						// Dropped on ourselves; ignore it.
						success = TRUE;
					}
					else
					{
						char *droppedTableName = (char *)bMessage->FindString( EW_MESSAGEITEM_NAME );
						if ( ( NULL != droppedTableName ) && ( B_NO_ERROR <= bMessage->Error() ) )
						{
							BVolume *bVolume = new BVolume( volumeID );
							if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
							{
								BDatabase *bDatabase = bVolume->Database();
								if ( NULL != bDatabase )
								{
									bDatabase->Sync();
								
									BTable *bTable = bDatabase->FindTable( droppedTableName );
									if ( NULL != bTable )
									{
										char volumeName[ B_OS_NAME_LENGTH ];
										(void) bVolume->GetName( volumeName );
										
										char alertString[ ( 2 * B_OS_NAME_LENGTH ) + 192 ];
										(void)sprintf( alertString, "The \"%s\" database already has a table named \"%s\".  Duplicate names are allowed, but are strongly discouraged.\nPlease confirm the action.", volumeName, droppedTableName );
										BAlert *bAlert = new BAlert( "ExistsWarning", alertString, "Add anyway", "Cancel", NULL, B_WIDTH_FROM_WIDEST, B_WARNING_ALERT );
										if ( NULL != bAlert )
										{
											bAlert->SetShortcut( 1L, B_ESCAPE );
											if ( 0L == bAlert->Go() )
											{
												bTable = NULL;
											}
											else
											{
												// Successful in cancelling; prevent the "Unable..." notice.
												success = TRUE;
											}
										}
									}
									
									if ( NULL == bTable )
									{
										bTable = bDatabase->CreateTable( droppedTableName );
										if ( NULL != bTable )
										{
											BVolume *bVolume = new BVolume( droppedVolumeID );
											if ( NULL != bVolume )
											{
												BDatabase *fromDatabase = bVolume->Database();
												delete bVolume;
												
												if ( NULL != fromDatabase )
												{
													fromDatabase->Sync();
													
													BTable *fromTable = fromDatabase->FindTable( droppedTableName );
													if ( NULL != fromTable )
													{
														long count = fromTable->CountFields();
														if ( 0L < count )
														{
															char fieldName[ B_OS_NAME_LENGTH ];
															field_key key = 0L;
															long type = 0L;
															long flags = 0L;

															success = TRUE;
																														
															for ( long index = 0L; ( index < count ); index++ )
															{
																if ( FALSE == fromTable->GetFieldInfo( index, fieldName, &key, &type, &flags ) )
																{
																	success = FALSE;
																}
																else
																{
																	switch( type )
																	{
																		case ( B_STRING_TYPE ):
																			key = bTable->AddStringField( fieldName, flags );
																		break;
																		
																		case ( B_LONG_TYPE ):
																			key = bTable->AddLongField( fieldName, flags );
																		break;
																		
																		case ( B_RAW_TYPE ):
																			key = bTable->AddRawField( fieldName, flags );
																		break;
																		
																		case ( B_TIME_TYPE ):
																			key = bTable->AddTimeField( fieldName, flags );
																		break;
																		
																		case ( B_RECORD_TYPE ):
																			key = bTable->AddRecordIDField(  fieldName, flags );
																		break;
																		
																		default:
																			success = FALSE;
																		break;
																	}
																}
															}
														}
													}
												}
											}
											
											NamePlus *namePlus = createNamePlus( droppedTableName, (void *)bTable, NULL );
											if ( NULL != namePlus )
											{
												AddItem( namePlus );
											}
										}
									}
								}
								
								delete bVolume;
							}
						}
					}
				}
				
				if ( FALSE == success )
				{
					BAlert *bAlert = new BAlert( "ErrorDND", "Unable to add table", "OK" );
					if ( NULL != bAlert )
					{
						(void)bAlert->Go();
					}
				}
			}
		}
	}
}


// The user wants to drag a table to copy it somewhere.
void DatabaseWindow :: ItemSelected( long index )
{ 
	// A table was selected.
	NamePlus *namePlus = ItemAt( index );
	if ( NULL != namePlus )
	{
		BMessage *bMessage = new BMessage( LW_MESSAGEWHAT_DND );
		if ( NULL != bMessage )
		{
			if ( B_NO_ERROR <= bMessage->AddLong( DBED_MESSAGEITEM_DROPTYPE, DBED_DROPTYPE_TABLE ) )
			{
				if ( B_NO_ERROR <= bMessage->AddLong( DBED_MESSAGEITEM_VOLUMEID, volumeID ) )
				{
					if ( B_NO_ERROR <= bMessage->AddString( EW_MESSAGEITEM_NAME, namePlus->np_Name ) )
					{
						StartDrag( bMessage, index );
						bMessage = NULL;
					}
				}
			}
			
			if ( NULL != bMessage )
			{
				delete bMessage;
			}
		}
	}
}

// Open up a table window for the invoked item.
void DatabaseWindow :: ItemInvoked( long index )
{ 
	// A table was invoked.
	NamePlus *namePlus = ItemAt( index );
	if ( NULL != namePlus )
	{
		// If the data field is non-Null, this item's window is already active.
		if ( NULL != namePlus->np_Handler )
		{
			TableWindow *tableWindow = (TableWindow *)namePlus->np_Handler;
			BMessenger *bMessenger = new BMessenger( cast_as( tableWindow, BHandler ) );
			if ( NULL != bMessenger )
			{
				bMessenger->SendMessage( LW_MESSAGEWHAT_ACTIVATE );
			}
		}
		else
		{
			if ( TRUE == Lock() )
			{
				BRect myFrame = Frame();
				TableWindow *tableWindow = new TableWindow( BRect( myFrame.right + B_V_SCROLL_BAR_WIDTH, myFrame.top, myFrame.right + 50.0, myFrame.top + 50.0 ), volumeID, (BTable *)namePlus->np_ID, this, color );
				if ( NULL != tableWindow ) 
				{
					if ( B_NO_ERROR > tableWindow->Status() )
					{
						delete tableWindow;
					}
					else
					{
						namePlus->np_Handler = (BHandler *)tableWindow;
					}
				}
				Unlock();
			}
		}
	}
}


// The user wants to create a new table.  Create an edit window.
long DatabaseWindow :: NewItem( void )
{
	long result = B_ERROR;
	
	NamePlus *namePlus = createNamePlus( NULL, (void *)NULL, (BHandler *)NULL );
	if ( NULL != namePlus )
	{
		if ( TRUE == Lock() )
		{
			BRect frame = Frame();
			DatabaseEditWindow *databaseEditWindow = new DatabaseEditWindow( BPoint( frame.right + B_V_SCROLL_BAR_WIDTH, frame.top ), volumeID, this, color );
			if ( NULL != databaseEditWindow )
			{
				namePlus->np_Handler = databaseEditWindow;
				if ( TRUE == AddPendingItem( namePlus ) )
				{
					result = B_NO_ERROR;
				}
			}
			Unlock();
		}
		
		if ( B_ERROR > result )
		{
			delete namePlus;
		}
	}
	
	return( result );
}


// The table window lists the fields in a table.
TableWindow :: TableWindow( BRect frame, long callersVolumeID, BTable *callersBTable, BHandler *callersHandler, rgb_color *rgb_color ) : ListWindow( frame, LW_MENUFLAG_NEW, callersHandler, rgb_color, FALSE )
{
	status = B_ERROR;

	SetSupportsSelection( TRUE );	
	SetSupportsInvocation( TRUE );	
		
	if ( NULL != callersBTable )
	{
		bTable = callersBTable;
		volumeID = callersVolumeID;
		
		status = B_NO_ERROR;
		
		CreateTitle( volumeID, bTable->Name() );

		char      fieldName[ B_OS_NAME_LENGTH ];
		field_key key;
		long      type;
		long      flags;
		
		long fields = bTable->CountFields();
		long nameLength;
		long longestLength = 0L;
		
		for ( long index = 0L; index < fields; index++ )
		{
			fieldName[ 0 ] = '\0';
			
			if ( TRUE == bTable->GetFieldInfo( index, fieldName, &key, &type, &flags ) )
			{
				// Get some info on the length of the strings.
				nameLength = strlen( fieldName );
				if ( longestLength < nameLength )
				{
					longestLength = nameLength;
				}
			
				NamePlus *namePlus = createNamePlus( fieldName, (void *)key, (BHandler *)NULL );
				if ( NULL != namePlus )
				{
					if ( FALSE == AddItem( namePlus ) )
					{
						disposeNamePlus( namePlus );
					}
				}
			}
		}
		
		if ( 0L < fields )
		{
			ResizeToFitText( longestLength, fields );
		}
		
		if ( B_NO_ERROR <= status )
		{
			InsureOnScreen();
			Show();
		}
	}
}


void TableWindow :: CreateTitle( long volumeID, const char *tableName )
{
	BVolume *bVolume = new BVolume( volumeID );
	if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
	{
		char volumeName[ B_OS_NAME_LENGTH ];
		if ( B_NO_ERROR <= bVolume->GetName( volumeName ) )
		{
			char title[ ( 2 * B_OS_NAME_LENGTH ) + 9 ];
			if ( NULL != title )
			{
				sprintf( title, "%s|%s Fields", volumeName, tableName );
				SetTitle( title );
			}
		}
		
		delete bVolume;
	}
}


// The user has completed creating a new field.  Can we accept it?
long TableWindow :: AcceptNew( NamePlus *namePlus, BMessage *message )
{
	long acceptResult = LW_ACCEPTRESULT_CANCEL;
	
	if ( ( NULL != namePlus ) && ( NULL != message ) )
	{
		char *name = (char *)message->FindString( EW_MESSAGEITEM_NAME );
		if ( ( NULL != name ) && ( B_NO_ERROR <= message->Error() ) )
		{								
			if ( 0 == strlen( name ) )
			{
				BAlert *bAlert = new BAlert( "BadName", "Name was empty; field not added.", "OK" );
				if ( NULL != bAlert )
				{
					(void)bAlert->Go();
				}
			}
			else
			{
				long type = message->FindLong( EW_MESSAGEITEM_TYPE );
				if ( B_NO_ERROR <= message->Error() )
				{
					long flags = message->FindLong( EW_MESSAGEITEM_FLAGS );
					if ( B_NO_ERROR <= message->Error() )
					{
						BDatabase *bDatabase = bTable->Database();
						if ( NULL != bDatabase )
						{
							acceptResult = LW_ACCEPTRESULT_ACCEPT;
	
							bDatabase->Sync();
								
							field_key tmpKey = 0L;
							long tmpType = 0L;
							long tmpFlags = 0L;
							
							// Test for an existing (duplicate) entry.
							if ( TRUE == bTable->GetFieldInfo( name, &tmpKey, &tmpType, &tmpFlags ) )
							{
								char volumeName[ B_OS_NAME_LENGTH ];
								BVolume *bVolume = new BVolume( volumeID );
								if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
								{
									(void)bVolume->GetName( volumeName );
									delete bVolume;
								}
								
								char alertString[ ( 3 * B_OS_NAME_LENGTH ) + 192 ];
								(void)sprintf( alertString, "The \"%s\" table in the \"%s\" database already has a field named \"%s\".  Duplicate names are allowed, but are strongly discouraged.\nPlease confirm the action.", bTable->Name(), volumeName, name );
								BAlert *bAlert = new BAlert( "ExistsWarning", alertString, "Add anyway", "Edit", "Cancel", B_WIDTH_FROM_WIDEST, B_WARNING_ALERT );
								if ( NULL != bAlert )
								{
									bAlert->SetShortcut( 2L, B_ESCAPE );
									switch( bAlert->Go() )
									{
										case ( 0L ):
											acceptResult = LW_ACCEPTRESULT_ACCEPT;
										break;
										
										case ( 1L ):
											acceptResult = LW_ACCEPTRESULT_RETRY;
										break;
										
										case ( 2L ):
										default:
											acceptResult = LW_ACCEPTRESULT_CANCEL;
										break;
									}
								}
							}
							
							
							if ( LW_ACCEPTRESULT_ACCEPT == acceptResult ) 
							{
								field_key key;
								
								switch ( type )
								{
									case ( B_STRING_TYPE ):
										key = bTable->AddStringField( name, flags );
									break;
									
									case ( B_LONG_TYPE ):
										key = bTable->AddLongField( name, flags );
									break;
									
									case ( B_RAW_TYPE ):
										key = bTable->AddRawField( name, flags );
									break;
									
									case ( B_TIME_TYPE ):
										key = bTable->AddTimeField( name, flags );
									break;
									
									case ( B_RECORD_TYPE ):
										key = bTable->AddRecordIDField( name, flags );
									break;
									
									default:
										acceptResult = LW_ACCEPTRESULT_CANCEL;
									break;
								}
							
								if ( LW_ACCEPTRESULT_ACCEPT == acceptResult )
								{
									(void) strncpy( namePlus->np_Name, name, B_OS_NAME_LENGTH );
									namePlus->np_ID = (void *)key;
									namePlus->np_Handler = NULL;
								}
							}
						}
					}
				}
			}
		}
	}
	
	return( acceptResult );
}


// The user dragged and dropped another item onto us.  If it's a valid field, clone it and add it to the table.
void TableWindow :: ItemDropped( BMessage *bMessage )
{ 
	if ( NULL != bMessage )
	{
		long dropType = bMessage->FindLong( DBED_MESSAGEITEM_DROPTYPE );
		if ( B_NO_ERROR <= bMessage->Error() )
		{
			if ( DBED_DROPTYPE_FIELD != dropType )
			{
				BAlert *bAlert = new BAlert( "BadDND", "Only fields can be added to this table.", "Cancel" );
				if ( NULL != bAlert )
				{
					(void)bAlert->Go();
				}
			}
			else
			{
				bool success = FALSE;
				
				long droppedVolumeID = bMessage->FindLong( DBED_MESSAGEITEM_VOLUMEID );
				if ( B_NO_ERROR <= bMessage->Error() )
				{
					char *droppedTableName = (char *)bMessage->FindString( DBED_MESSAGEITEM_TABLENAME );
					if ( ( NULL != droppedTableName ) && ( B_NO_ERROR <= bMessage->Error() ) )
					{
						if ( ( droppedVolumeID == volumeID ) && ( 0 == strcmp( droppedTableName, bTable->Name() ) ) )
						{
							// Dropped on ourselves; ignore it.
							success = TRUE;
						}
						else
						{
							char *droppedFieldName = (char *)bMessage->FindString( EW_MESSAGEITEM_NAME );
							if ( ( NULL != droppedFieldName ) && ( B_NO_ERROR <= bMessage->Error() ) )
							{
								field_key key = 0L;
								long type = 0L;
								long flags = 0L;
								
								BDatabase *bDatabase = bTable->Database();
								if ( NULL != bDatabase )
								{
									bDatabase->Sync();
								}
								
								bool exists = bTable->GetFieldInfo( droppedFieldName, &key,  &type, &flags );
								if ( TRUE == exists )
								{
									char volumeName[ B_OS_NAME_LENGTH ];
									BVolume *bVolume = new BVolume( volumeID );
									if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
									{
										(void)bVolume->GetName( volumeName );
										delete bVolume;
									}
									
									char alertString[ ( 3 * B_OS_NAME_LENGTH ) + 192 ];
									(void)sprintf( alertString, "The \"%s\" table in the \"%s\" database already has a field named \"%s\".  Duplicate names are allowed, but are strongly discouraged.\nPlease confirm the action.", bTable->Name(), volumeName, droppedFieldName );
									BAlert *bAlert = new BAlert( "ExistsWarning", alertString, "Add anyway", "Cancel", NULL, B_WIDTH_FROM_WIDEST, B_WARNING_ALERT );
									if ( NULL != bAlert )
									{
										bAlert->SetShortcut( 1L, B_ESCAPE );
										if ( 0L == bAlert->Go() )
										{
											exists = FALSE;
										}
										else
										{
											// Successful in cancelling; prevent the "Unable..." notice.
											success = TRUE;
										}
									}
								}
										
								if ( FALSE == exists )
								{
									BVolume *fromVolume = new BVolume( droppedVolumeID );
									if ( NULL != fromVolume )
									{
										BDatabase *fromDatabase = fromVolume->Database();
										if ( NULL != fromDatabase )
										{
											fromDatabase->Sync();
											
											BTable *fromTable = fromDatabase->FindTable( droppedTableName );
											if ( NULL != fromTable )
											{
												if ( TRUE == fromTable->GetFieldInfo( droppedFieldName, &key,  &type, &flags ) )
												{
													success = TRUE;
													
													switch( type )
													{
														case ( B_STRING_TYPE ):
															key = bTable->AddStringField( droppedFieldName, flags );
														break;
														
														case ( B_LONG_TYPE ):
															key = bTable->AddLongField( droppedFieldName, flags );
														break;
														
														case ( B_RAW_TYPE ):
															key = bTable->AddRawField( droppedFieldName, flags );
														break;
														
														case ( B_TIME_TYPE ):
															key = bTable->AddTimeField( droppedFieldName, flags );
														break;
														
														case ( B_RECORD_TYPE ):
															key = bTable->AddRecordIDField( droppedFieldName, flags );
														break;
														
														default:
															success = FALSE;
														break;
													}
													
													if ( TRUE == success )
													{
														NamePlus *namePlus = createNamePlus( droppedFieldName, (void *)key, NULL );
														if ( NULL != namePlus )
														{
															AddItem( namePlus );
														}
													}
												}
											}
										}
										
										delete fromVolume;
									}
								}
							}
						}
					}
				}
				
				if ( FALSE == success )
				{
					BAlert *bAlert = new BAlert( "ErrorDND", "Unable to add field", "OK" );
					if ( NULL != bAlert )
					{
						(void)bAlert->Go();
					}
				}
			}
		}
	}
}

// The user wants to drag a field to copy it somewhere.
void TableWindow :: ItemSelected( long index )
{ 
	// A field was selected.
	NamePlus *namePlus = ItemAt( index );
	if ( NULL != namePlus )
	{
		BMessage *bMessage = new BMessage( LW_MESSAGEWHAT_DND );
		if ( NULL != bMessage )
		{
			if ( B_NO_ERROR <= bMessage->AddLong( DBED_MESSAGEITEM_DROPTYPE, DBED_DROPTYPE_FIELD ) )
			{
				if ( B_NO_ERROR <= bMessage->AddLong( DBED_MESSAGEITEM_VOLUMEID, volumeID ) )
				{
					if ( B_NO_ERROR <= bMessage->AddString( DBED_MESSAGEITEM_TABLENAME, (char *)bTable->Name() ) )
					{
						if ( B_NO_ERROR <= bMessage->AddString( EW_MESSAGEITEM_NAME, namePlus->np_Name ) )
						{
							StartDrag( bMessage, index );
							bMessage = NULL;
						}
					}
				}
			}
			
			if ( NULL != bMessage )
			{
				delete bMessage;
			}
		}
	}
}


// Open up a field window for the invoked item.
void TableWindow :: ItemInvoked( long index )
{ 
	// A field was invoked.
	NamePlus *namePlus = ItemAt( index );
	if ( NULL != namePlus )
	{
		// If the data field is non-Null, this item's window is already active.
		if ( NULL != namePlus->np_Handler )
		{
			FieldWindow *fieldWindow = (FieldWindow *)namePlus->np_Handler;
			BMessenger *bMessenger = new BMessenger( cast_as( fieldWindow, BHandler ) );
			if ( NULL != bMessenger )
			{
				bMessenger->SendMessage( LW_MESSAGEWHAT_ACTIVATE );
			}
		}
		else
		{
			if ( TRUE == Lock() )
			{
				BRect myFrame = Frame();
				FieldWindow *fieldWindow = new FieldWindow( BRect( myFrame.right + B_V_SCROLL_BAR_WIDTH, myFrame.top, myFrame.right + 50.0, myFrame.top + 50.0 ), volumeID, bTable, (long)namePlus->np_ID, this, color );
				if ( NULL != fieldWindow ) 
				{
					if ( B_NO_ERROR > fieldWindow->Status() )
					{
						delete fieldWindow;
					}
					else
					{
						namePlus->np_Handler = (BHandler *)fieldWindow;
					}
				}
				Unlock();
			}
		}
	}
}


// The user wants to create a new field.  Create an edit window.
long TableWindow :: NewItem( void )
{
	long result = B_ERROR;
	
	NamePlus *namePlus = createNamePlus( NULL, (void *)NULL, (BHandler *)NULL );
	if ( NULL != namePlus )
	{
		if ( TRUE == Lock() )
		{
			BRect frame = Frame();
			TableEditWindow *tableEditWindow = new TableEditWindow( BPoint( frame.right + B_V_SCROLL_BAR_WIDTH, frame.top ), volumeID, bTable, this, color );
			if ( NULL != tableEditWindow )
			{
				namePlus->np_Handler = tableEditWindow;
				if ( TRUE == AddPendingItem( namePlus ) )
				{
					result = B_NO_ERROR;
				}
			}
			Unlock();
		}
		
		if ( B_ERROR > result )
		{
			delete namePlus;
		}
	}
	
	return( result );
}


// The field window displays the attributes of a field.
FieldWindow :: FieldWindow( BRect frame, long callersVolumeID, BTable *callersBTable, field_key myKey, BHandler *callersHandler, rgb_color *rgb_color ) : ListWindow( frame, LW_MENUFLAG_NONE, callersHandler, rgb_color, FALSE )
{
	char *typeString;
	char flagsString[ 80 ];
	
	status = B_ERROR;

	if ( NULL != callersBTable )
	{
		bTable = callersBTable;
		volumeID = callersVolumeID;
		key = myKey;
		
		SetSupportsInvocation( TRUE );	

		status = B_NO_ERROR;

		char      fieldName[ B_OS_NAME_LENGTH ];
		long      type = 0L;
		long      flags = 0L;

		fieldName[ 0 ] = '\0';
			
		if ( TRUE == bTable->GetFieldInfo( key, fieldName, &type, &flags ) )
		{
			CreateTitle( volumeID, bTable->Name(), fieldName );
		
			switch( type )
			{
				case ( B_RAW_TYPE ):
					typeString = BE_TYPENAME_RAW;
				break;
				
				case ( B_LONG_TYPE ):
					typeString = BE_TYPENAME_LONG;
				break;
				
				case ( B_STRING_TYPE ):
					typeString = BE_TYPENAME_STRING;
				break;
				
				case ( B_RECORD_TYPE ):
					typeString = BE_TYPENAME_RECORD;
				break;
				
				case ( B_TIME_TYPE ):
					typeString = BE_TYPENAME_TIME;
				break;
				
				default:
					typeString = BE_TYPENAME_UNKNOWN;
				break;
			}
			
			/* Create a string of flag values.
			   Use strcat to concatenate strings together.
			   Note importance of size of flagsString.
			*/
			
			strcpy( flagsString, "<none>" );
			
			if ( B_INDEXED_FIELD == ( flags & B_INDEXED_FIELD ) )
			{
				strcpy( flagsString, BE_FLAGNAME_INDEXED );
			}
			
			char attrString[ B_OS_NAME_LENGTH ];
			if ( NULL != attrString )
			{
				(void) sprintf( attrString, "0x%08lx  %s  %s", key, typeString, flagsString );

				NamePlus *namePlus = createNamePlus( attrString, (void *)NULL, (BHandler *)NULL );
				if ( NULL != namePlus )
				{
					if ( TRUE == AddItem( namePlus ) )
					{
						ResizeToFitText( (long)strlen( attrString ), 1L );
					
						if ( B_NO_ERROR <= status )
						{
							InsureOnScreen();
							Show();
						}
					}
				}
			}
		}
	}
}


void FieldWindow :: CreateTitle( long volumeID, const char *tableName, const char *fieldName )
{
	BVolume *bVolume = new BVolume( volumeID );
	if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
	{
		char volumeName[ B_OS_NAME_LENGTH ];
		if ( B_NO_ERROR <= bVolume->GetName( volumeName ) )
		{
			char title[ ( 3 * B_OS_NAME_LENGTH ) + 14 ];
			if ( NULL != title )
			{
				sprintf( title, "%s|%s|%s Attributes", volumeName, tableName, fieldName );
				SetTitle( title );
			}
		}
		
		delete bVolume;
	}
}


/*
************************************************************************************************
************************************************************************************************
************************************************************************************************
************************************************************************************************
************************************************************************************************
************************************************************************************************
*/


// The database edit window allows the creation of a table.
DatabaseEditWindow :: DatabaseEditWindow( BPoint origin, long volumeID, BHandler *callersHandler, rgb_color *rgb_color ) : EditWindow( origin, callersHandler, rgb_color )
{
	char titleP2[] = " Table Creation";
	char fullTitle[ B_OS_NAME_LENGTH + sizeof( titleP2 ) ]; // Includes NULL.
	status = B_ERROR;

	// Initialize the object's attributes.
	bTextView = NULL;

	BDatabase *bDatabase = NULL;
	
	// Create and set the window title.
	BVolume *bVolume = new BVolume( volumeID );
	if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
	{
		if ( B_ERROR > bVolume->GetName( fullTitle ) )
		{
			(void) strcpy( fullTitle, "" );
		}
		
		bDatabase = bVolume->Database();
		
		delete bVolume;
	}
	(void) strcat( fullTitle, titleP2 );
	SetTitle( fullTitle );
		
	float textWidth = fontCharWidth * B_OS_NAME_LENGTH;
	float textHeight = fontCharHeight;
	
	float textViewWidth = textWidth + fontCharWidth;
	float textViewHeight = textHeight + fontCharHeight;
	
	float textBoxWidth = textViewWidth + ( 2 * fontCharWidth ) + fontCharWidth;
	float textBoxHeight = fontCharHeight + textViewHeight + ( 2 * fontCharHeight );
	
	
	// Create a BBox-enrobed TextView to allow for naming of the new table.
	BBox *bBox = new BBox( BRect( nextH, nextV, textBoxWidth, textBoxHeight ), DBED_NAME_TEXTBOX ); 
	if ( NULL != bBox )
	{
		bBox->SetLabel( " Name " );
		bBox->SetViewColor( background->ViewColor() );
		if ( TRUE == Lock() )
		{
			bBox->SetLowColor( background->LowColor() );
			Unlock();
		}
		
		bTextView = new BTextView( BRect( fontCharWidth, fontCharHeight + 1, textViewWidth, fontCharHeight + textViewHeight ), DBED_NAME_TEXTVIEW, BRect( fontCharWidth / 2, fontCharHeight / 2, textWidth, textHeight ), B_FOLLOW_ALL, B_WILL_DRAW );
		if ( NULL != bTextView )
		{
			bBox->AddChild( bTextView );
			background->AddChild( bBox );

			bTextView->DisallowChar( B_ENTER );
			bTextView->SetMaxChars( B_OS_NAME_LENGTH );
			bTextView->SetText( "" );
			
			nextH += textBoxWidth;
			
			status = B_NO_ERROR;
		}
		else
		{
			delete bBox;
		}
	}

	// Create a BBox-enrobed PopUpMenu to allow selection of the "inherit from" table.
	if ( B_NO_ERROR <= status )
	{
		status = B_ERROR;
		if ( NULL != bDatabase )
		{
			bDatabase->Sync();
			
			float menuWidth = B_OS_NAME_LENGTH * fontCharWidth;

			menu_info menuInfo;
			if ( B_NO_ERROR <= get_menu_info( &menuInfo ) )
			{
				menuWidth = B_OS_NAME_LENGTH * menuInfo.font_size;
			}
			
			float menuHeight = menuBarHeight;
			
			float inheritBoxWidth = menuWidth + ( 2 * fontCharWidth );
			float inheritBoxHeight = menuHeight + ( 2 * fontCharHeight );

			long tableCount = bDatabase->CountTables();
			if ( 0L < tableCount )
			{
				BBox *inheritBox = new BBox( BRect( nextH, nextV, nextH + inheritBoxWidth, nextV + inheritBoxHeight ), DBED_NAME_INHERITBOX );
				if ( NULL != inheritBox )
				{
					inheritBox->SetLabel( " Inherit from " );
					inheritBox->SetViewColor( background->ViewColor() );
					if ( TRUE == Lock() )
					{
						inheritBox->SetLowColor( background->LowColor() );
						Unlock();
					}
		
					popUp = new BPopUpMenu( DBED_INHERIT_NONE );
					if ( NULL != popUp )
					{
						// add items.
						BMenuItem *menuItem;
						
						menuItem = new BMenuItem( DBED_INHERIT_NONE, NULL );
						if ( NULL != menuItem )
						{
							popUp->AddItem( menuItem );
							menuItem->SetMarked( TRUE );
						} 
					
						BSeparatorItem *separator = new BSeparatorItem();
						if ( NULL != separator )
						{
							popUp->AddItem( separator );
						}
						
						BTable *inheriTable;
						char *inheriTableName;
						
						status = B_NO_ERROR;
						
						for ( long index = 0L; ( ( B_NO_ERROR <= status ) && ( index < tableCount ) ); index++ )
						{
							inheriTable = bDatabase->TableAt( index );
							if ( NULL == inheriTable )
							{
								status = B_ERROR;
							}
							else
							{
								inheriTableName = inheriTable->Name();
								if ( NULL == inheriTableName )
								{
									status = B_ERROR;
								}
								else
								{
									menuItem = new BMenuItem( inheriTableName, NULL );
									if ( NULL == menuItem )
									{
										status = B_ERROR;
									}
									else
									{
										popUp->AddItem( menuItem );
									} 
								}
							}
						}
					
						if ( B_NO_ERROR <= status )
						{
							BMenuBar *bMenuBar = new BMenuBar( BRect( fontCharWidth, fontCharHeight + 3, menuWidth, menuHeight ), DBED_NAME_INHERITMENU, B_FOLLOW_NONE );
							if ( NULL == bMenuBar )
							{
								status = B_ERROR;
							}
							else
							{
								if ( Lock() )
								{
									bMenuBar->AddItem( popUp );
									Unlock();
								}
								inheritBox->AddChild( bMenuBar );
							
								background->AddChild( inheritBox );
					
								nextH += inheritBoxWidth + spacingH;
			
								nextV += max( textBoxHeight, inheritBoxHeight );
							}
						}
						
						if ( B_ERROR > status )
						{
							delete popUp;
						}
					}
					
					if ( B_ERROR > status )
					{
						delete inheritBox;
					}
				}
			}
		}
	}

	if ( B_NO_ERROR <= status )
	{
		AutoResize();
		InsureOnScreen();

		Show();

		if ( TRUE == Lock() )
		{
			bTextView->MakeFocus( TRUE );
			Unlock();
		}
	}
}


// The user wants to accept the data.  Bundle it up and pass it back to my parent for approval.
long DatabaseEditWindow :: PrepareAccept( BMessage *bMessage )
{
	long result = B_ERROR;
	
	if ( NULL != bMessage )
	{
		result = bMessage->AddString( EW_MESSAGEITEM_NAME, bTextView->Text() );
		if ( B_NO_ERROR <= result )
		{
			BMenuItem *bMenuItem = popUp->FindMarked();
			if ( NULL != bMenuItem )
			{
				const char *inheritFrom = bMenuItem->Label();
				if ( NULL != inheritFrom )
				{
					if ( 0 != strcmp( "None", inheritFrom ) )
					{
						result = bMessage->AddString( DBED_MESSAGEITEM_INHERIT, inheritFrom );
					}
				}
			}
		}
	}
	
	return( result );
}
		

// The table edit window allows the creation of a field.
TableEditWindow :: TableEditWindow( BPoint origin, long volumeID, BTable *bTable, BHandler *callersHandler, rgb_color *rgb_color ) : EditWindow( origin, callersHandler, rgb_color )
{
	char titleP2[] = " Field Creation";
	char fullTitle[ B_OS_NAME_LENGTH + sizeof( titleP2 ) ]; // Includes NULL.
	status = B_ERROR;

	// Initialize the object's attributes.
	bTextView = NULL;

	// Create and set the window title.
	BVolume *bVolume = new BVolume( volumeID );
	if ( ( NULL != bVolume ) && ( B_NO_ERROR <= bVolume->Error() ) )
	{
		if ( B_ERROR > bVolume->GetName( fullTitle ) )
		{
			(void) strcpy( fullTitle, "" );
		}
		
		delete bVolume;
	}
	(void) strcat( fullTitle, "|" );
	(void) strcat( fullTitle, bTable->Name() );
	(void) strcat( fullTitle, titleP2 );
	SetTitle( fullTitle );
	
	float textWidth  = fontCharWidth * B_OS_NAME_LENGTH;
	float textHeight = fontCharHeight;
	
	float textViewWidth  = textWidth + fontCharWidth;
	float textViewHeight = textHeight + fontCharHeight;
	
	float textBoxWidth  = textViewWidth + ( 2 * fontCharWidth ) + fontCharWidth;
	float textBoxHeight = fontCharHeight + textViewHeight + ( 2 * fontCharHeight );
	
	
	BBox *textBox = new BBox( BRect( nextH, nextV, textBoxWidth, textBoxHeight ), DBED_NAME_TEXTBOX ); 
	if ( NULL != textBox )
	{
		textBox->SetLabel( " Name " );
		textBox->SetViewColor( background->ViewColor() );
		if ( TRUE == Lock() )
		{
			textBox->SetLowColor( background->LowColor() );
			Unlock();
		}
		
		bTextView = new BTextView( BRect( fontCharWidth, fontCharHeight + 1, textViewWidth, fontCharHeight + textViewHeight ), DBED_NAME_TEXTVIEW, BRect( fontCharWidth / 2, fontCharHeight / 2, textWidth, textHeight ), B_FOLLOW_ALL, B_WILL_DRAW );
		if ( NULL != bTextView )
		{
			textBox->AddChild( bTextView );
			background->AddChild( textBox );

			bTextView->DisallowChar( B_ENTER );
			bTextView->SetMaxChars( B_OS_NAME_LENGTH );
			bTextView->SetText( "" );
			
			nextH += textBoxWidth;
			
			status = B_NO_ERROR;
		}
		else
		{
			delete textBox;
		}
	}

	
	float menuWidth = 6 * fontCharWidth;

	menu_info menuInfo;
	if ( B_NO_ERROR <= get_menu_info( &menuInfo ) )
	{
		menuWidth = 6 * menuInfo.font_size;
	}

	float menuHeight = menuBarHeight;

	float typeBoxWidth = menuWidth + ( 2 * fontCharWidth );
	float typeBoxHeight = menuHeight + ( 2 * fontCharHeight );

		
	if ( B_NO_ERROR <= status )
	{
		status = B_ERROR;
		
		BBox *typeBox = new BBox( BRect( nextH, nextV, nextH + typeBoxWidth, nextV + typeBoxHeight ), DBED_NAME_TYPEBOX );
		if ( NULL != typeBox )
		{
			typeBox->SetLabel( " Type " );
			typeBox->SetViewColor( background->ViewColor() );
			if ( TRUE == Lock() )
			{
				typeBox->SetLowColor( background->LowColor() );
				Unlock();
			}

			popUp = new BPopUpMenu( BE_TYPENAME_LONG );
			if ( NULL != popUp )
			{
				// add items.
				BMenuItem *menuItem;
				
				menuItem = new BMenuItem( BE_TYPENAME_STRING, NULL );
				if ( NULL != menuItem )
				{
					popUp->AddItem( menuItem );
				} 
			
				menuItem = new BMenuItem( BE_TYPENAME_LONG, NULL );
				if ( NULL != menuItem )
				{
					popUp->AddItem( menuItem );
					menuItem->SetMarked( TRUE );
				} 
			
				menuItem = new BMenuItem( BE_TYPENAME_RAW, NULL );
				if ( NULL != menuItem )
				{
					popUp->AddItem( menuItem );
				} 
			
				menuItem = new BMenuItem( BE_TYPENAME_TIME, NULL );
				if ( NULL != menuItem )
				{
					popUp->AddItem( menuItem );
				} 
			
				menuItem = new BMenuItem( BE_TYPENAME_RECORD, NULL );
				if ( NULL != menuItem )
				{
					popUp->AddItem( menuItem );
				}
				 
				BMenuBar *bMenuBar = new BMenuBar( BRect( fontCharWidth, fontCharHeight + 3, menuWidth, menuHeight ), DBED_NAME_TYPEMENU, B_FOLLOW_NONE );
				if ( NULL != bMenuBar )
				{
					if ( Lock() )
					{
						bMenuBar->AddItem( popUp );
						Unlock();
					}
					typeBox->AddChild( bMenuBar );
				
					background->AddChild( typeBox );
		
					nextH += typeBoxWidth + spacingH;
	
					status = B_NO_ERROR;
				}
			}
			else
			{
				delete typeBox;
			}
		}
	}
	

	float checkboxWidth  = ( 9 * fontCharWidth ) + B_V_SCROLL_BAR_WIDTH;
	float checkboxHeight = fontCharHeight;
	
	float flagBoxWidth  = checkboxWidth + ( 2 * fontCharWidth );
	float flagBoxHeight = fontCharHeight + checkboxHeight + fontCharHeight;
	

	if ( B_NO_ERROR <= status )
	{
		status = B_ERROR;
		
		BBox *flagBox = new BBox( BRect( nextH, nextV, nextH + flagBoxWidth, nextV + flagBoxHeight ), DBED_NAME_FLAGBOX );
		if ( NULL != flagBox )
		{
			flagBox->SetLabel( " Flags " );
			flagBox->SetViewColor( background->ViewColor() );
			if ( TRUE == Lock() )
			{
				flagBox->SetLowColor( background->LowColor() );
				Unlock();
			}
			bCheckBox = new BCheckBox( BRect( fontCharWidth, fontCharHeight + 1, checkboxWidth + fontCharWidth, checkboxHeight + fontCharHeight ), DBED_NAME_CHECKBOX, BE_FLAGNAME_INDEXED, B_FOLLOW_NONE ); 
			if ( NULL != bCheckBox )
			{
				flagBox->AddChild( bCheckBox );
				background->AddChild( flagBox );
				
				nextH += flagBoxWidth + spacingH;
				nextV += max( max( textBoxHeight, typeBoxHeight ), flagBoxHeight );
	
				status = B_NO_ERROR;
			}
			else
			{
				delete flagBox;
			}
		}
	}
	
	if ( B_NO_ERROR <= status )
	{
		AutoResize();
		InsureOnScreen();
		
		Show();
		
		if ( TRUE == Lock() )
		{
			bTextView->MakeFocus( TRUE );
			Unlock();
		}
	}
}


// The user wants to accept the data.  Bundle it up and pass it back to my parent for approval.
long TableEditWindow :: PrepareAccept( BMessage *bMessage )
{
	long result = B_ERROR;
	
	if ( NULL != bMessage )
	{
		if ( B_NO_ERROR <= bMessage->AddString( EW_MESSAGEITEM_NAME, bTextView->Text() ) )
		{
			long type = 0L;
			BMenuItem *bMenuItem = popUp->FindMarked();
			if ( NULL != bMenuItem )
			{
				const char *label = bMenuItem->Label();
				if ( NULL != label )
				{
					if ( 0 == strcmp( BE_TYPENAME_STRING, label ) )
					{
						type = B_STRING_TYPE;
					}
					else if ( 0 == strcmp( BE_TYPENAME_LONG, label ) )
					{
						type = B_LONG_TYPE;
					}
					else if ( 0 == strcmp( BE_TYPENAME_RAW, label ) )
					{
						type = B_RAW_TYPE;
					}
					else if ( 0 == strcmp( BE_TYPENAME_TIME, label ) )
					{
						type = B_TIME_TYPE;
					}
					else if ( 0 == strcmp( BE_TYPENAME_RECORD, label ) )
					{
						type = B_RECORD_TYPE;
					}
					else 
					{
						type = 0L;
					}
				}
			}
			
			if ( B_NO_ERROR <= bMessage->AddLong( EW_MESSAGEITEM_TYPE, type ) )
			{
				long flags = 0L;
				if ( B_CONTROL_ON == bCheckBox->Value() )
				{
					flags |= B_INDEXED_FIELD;
				}
				
				result = bMessage->AddLong( EW_MESSAGEITEM_FLAGS, flags );
			}
		}
	}
	
	return( result );
}
