/* ++++++++++
	FILE:	DigitalPort.cpp
	REVS:	$Revision$
	NAME:	herold
	DATE:	Fri April 19, 1996
	Copyright (c) 1996 by Be Incorporated.  All Rights Reserved.
+++++ */

#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include "DigitalPort.h"
#include "digital_driver.h"


/* ----------
	constructor
----- */

BDigitalPort::BDigitalPort()
{

	ffd = -1;
}

/* ----------
	destructor
----- */

BDigitalPort::~BDigitalPort()
{
	SetInput();
	close(ffd);
}

/* ----------
	open - actually open the device.
----- */

long
BDigitalPort::Open(const char *portName)
{
	static struct {
		char	*object_name;	/* name for users of this object */
		char	*dev_name;		/* name for the driver */
	} names[2] = {
		{ "DigitalA", "/dev/digital_port_a" },
		{ "DigitalB", "/dev/digital_port_b" }
	};
	char	*dev = NULL;
	int		i;
	
	// look for passed name, map to device name
	for (i = 0; !dev && i < sizeof (names) / sizeof (names[0]); i++)
		if (!strcmp (names[i].object_name, portName))
			dev = names[i].dev_name;
	
	if (!dev)									// name not found?
		return B_ERROR;
		
	if (ffd >= 0)								// it's already open
		close(ffd);								// close to reopen

	ffd = open(dev, O_RDWR);					// open the port
	if (ffd >= 0)								// if it worked
		SetInput();								// make it inputs

	return(ffd);
}

/* ----------
	Close - close the underlying device
----- */

void
BDigitalPort::Close()
{
	close(ffd);
	ffd = -1;
}

/* ----------
	Read - get current digital port value
----- */

long
BDigitalPort::Read(char *buf)
{
	return (ffd >= 0) ? read(ffd, buf, 1) : B_ERROR;
}

/* ----------
	Write - set digital port value
----- */

long
BDigitalPort::Write(char value)
{
	char	buf = value;
	
	return (ffd >= 0) ? write(ffd, &buf, 1) : B_ERROR;
}


/* ----------
	SetOutput - set the digital port to be outputs
----- */
long
BDigitalPort::SetOutput(void)
{
	if (ffd < 0)
		return B_ERROR;
	return ioctl (ffd, SET_DIGITAL_PORT_DIRECTION, 
		(void *) DIGITAL_PORT_OUTPUT);
}


/* ----------
	SetInput - set the digital port to be inputs
----- */
long
BDigitalPort::SetInput(void)
{
	if (ffd < 0)
		return B_ERROR;
	return ioctl (ffd, SET_DIGITAL_PORT_DIRECTION, 
		(void *) DIGITAL_PORT_INPUT);
}


