/*****************************************************************************
//
//	File:			3dWorld.h
//
//	Description:	Description of a 3d space sharinga consistent background.
//
//	Copyright 1997, Be Incorporated
//
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
******************************************************************************/

#ifndef _3D_WORLD_H
#define _3D_WORLD_H

#include <OS.h>

#ifndef _3D_DEFS_H 
#include "3dDefs.h"
#endif
#ifndef _3D_MATRIX_H 
#include "3dMatrix.h"
#endif
#ifndef _3D_UNIVERSE_H 
#include "3dUniverse.h"
#endif

class B3dCamera;
class B3dLight;
class B3dThing;
class B3dLens;
class B3dLighter;
class B3dRenderer;
class B3dTouchDesc;

/**************************************
// LOW-LEVEL API
/*************************************/

/* values used for the status variable of the B3dThingLink. */
enum {
	B_ROOT_THING =     0x0001,
	B_LIGHT_THING =    0x0002,
    B_VISIBLE_THING =  0x0004		
};

typedef struct B3dThingLink {
	uint16       status;
	uint16       update_count;
	union {
		struct B3dThingLink *next;
		B3dThing *thing;
	};
	struct B3dThingLink  *link_next;
	struct B3dThingLink  *link_prev;
} B3dThingLink;

typedef struct B3dThingSort {
	B3dThing     *thing;
	float        z;
} B3dThingSort;

/**************************************
// B3dWorldSort.
/*************************************/

class B3dWorldSort {

/*************************************
// LOW-LEVEL API                    */

 public:
	B3dWorldSort();
	virtual			~B3dWorldSort();

	int32			Count();
	void			SetCount(int32 count);
	bool			AssertCount(int32 count);

	B3dThingSort	*ThingList();

/*************************************
// PRIVATE STUFF                    */

 private:
    int32			count;
	int32			max_count;
	B3dThingSort	*list_things;
};

/**************************************
// B3dWorld.
/*************************************/

class B3dWorld {

/*************************************
// LOW-LEVEL API                    */

 public:
	B3dWorld(char *name, B3dUniverse *universe);
	virtual				~B3dWorld();

	inline const char	*Name();
	void				SetName(char *name);
	inline B3dUniverse	*Universe();

	virtual void		LinkThing(B3dThing *thing, bool root);
	virtual void		UnlinkThing(B3dThing *thing);
	virtual void		SetUpdate(	B3dThingLink	*link,
									uint32			update_count,
									bool			root);

	virtual void		SortThings(B3dCamera *camera);

	virtual B3dThingLink *GetFirstLight();
	
	virtual void		CalcTouch(	B3dVector		*origin,
									B3dVector		*axis,
									B3dVector		*origin0,
									B3dVector		*axis0,
									B3dTouchDesc	*touch,
									B3dMatrix		*mat);
	virtual bool 		GetTouch(	B3dVector		*origin,
									B3dVector		*axis,
									B3dTouchDesc	*touch);
	
/*************************************
// PRIVATE STUFF                    */

 protected:
	void        KiltSort(B3dThingSort *min, B3dThingSort *max);

 private:
	char					*name;
	/* a name, as every other object with meaning for the user.
	   */
	int32					array_count, array_max;
	/* The references to things are stored by blocks of LIST_BLOC_COUNT. Those
	   blocks are static after allocation (to allow storage of static pointer
	   in the B3dThing themself). All the blocs are index in order in a dynamic
	   buffer. array_max is the size of that dynamic buffer, array_count is the
	   count of entries really used (pointing to allocated blocks).
	   */
	int32					list_count;
	/* Total number of entries available in all the allocated blocks.
	   */
	B3dThingLink			*list_free;
	/* first free entry in the list of blocks.
	   */
	B3dUniverse				*uni;
	/* the universe owner of this world.
	   */
	B3dThingLink			**list_array;
	/* the pointer to the buffer list of blocks (described by array_count and
	   array_max).
	   */
	B3dThingLink			*first_root;
	/* first root of the current list of thing.
	   */
	int32					root_count;
	/* Total number of things in this world having the root status.
	   */
	B3dThingLink			*first_light;
	/* first light of the current list of thing.
	   */
	int32					light_count;
	/* Total number of light in this world.
	   */
	int32                   crc_alea;
	
	void		ExtendThingList();
	void		RootLink(B3dThingLink *link);
	void		RootUnlink(B3dThingLink *link);
	void		LightLink(B3dThingLink *link);
	void		LightUnlink(B3dThingLink *link);
};

/**************************************
// INLINE FUNCTION DEFINITIONS
/*************************************/

const char *B3dWorld::Name() {
	return name;
}

B3dUniverse *B3dWorld::Universe() {
	return uni;
}

#endif





