/*****************************************************************************
//
//	File:			3dView.h
//
//	Description:	Allow to map a 3dCamera view inside a BView..
//
//	Copyright 1997, Be Incorporated
//
// ******** 3dKit ARCHITECTURE NOTE :
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
******************************************************************************/

#ifndef _3D_VIEW_H
#define _3D_VIEW_H

#ifndef _VIEW_H
#include <View.h>
#endif
#ifndef _BITMAP_H
#include <Bitmap.h>
#endif
#ifndef _3D_CHANNEL_H
#include <3dChannel.h>
#endif
#ifndef _3D_CAMERA_H
#include <3dCamera.h>
#endif
#ifndef _3D_UNIVERSE_H
#include <3dUniverse.h>
#endif

/**************************************
// PRIVATE STUFF
/*************************************/

typedef struct {
	char		Haut;
	char		Bas;
	char		Gauche;
	char		Droit;		
	char		Bouton[4];
	int32 		Tempo[4];
} B3dKeyConfig;

typedef struct {
	BRect		bounds;
	void		*buffer;
	int32		bufferSize;
	int32		bytesPerLine;
} ZBuffer;

/**************************************
// B3dView.
/*************************************/

class B3dView : public BView {
	friend void B3dView_erase_buffer(void*, camera_buffer*);
	friend void B3dView_switch_buffer(void*, camera_buffer*);

/*************************************
// HIGH-LEVEL API                   */

 public:
						B3dView(	char		*name,
									BRect		frame, 
									B3dUniverse	*universe = NULL,
									uint32		renderFlags = B_3D_BLACK_BACKGROUND);
	/* Create a new 3dView, using the standard View parameters and an optional
	   Universe. If the universe is not specified, the view will allocate its own.
	   In any case, the view will create its own world. This is the standard way
	   to create an universe and a world. If you want to have multiple views
	   'looking' at different worlds but sharing the same universe, then you just
	   need to pass the universe of the first view to the other view. The renderFlags
	   options are described in 3dDefs.h
	   */
						B3dView(	char		*name,
									BRect		frame, 
									B3dWorld	*world,
									uint32		renderFlags = B_3D_BLACK_BACKGROUND); 
	/* This is the same thing in the case you want to have multiple views 'looking'
	   at the same world. You just have to pass the world of the first view to the
	   other views.
	   */
	
	virtual				~B3dView();
	/* Will free the universe if it was allocated by the view at construction. At
	   that time, the universe will destroy any object still linked to it. That
	   includes things and lights created in this universe, and all links between
	   them.
	   */

	inline B3dUniverse	*Universe();
	/* The universe that contains the camera of this view.
	   */

	inline B3dWorld		*World();
	/* The world that contains the camera of this view.
	   */

	B3dCamera				*Camera();
	void					SetCamera(B3dCamera *camera);	
	/* The current camera of this view. A default camera is allocated by the view
	   constructor. You can replace it by a different camera if you want. The view
	   will free its current camera at destruction time in any case. */	

	uint32				RenderFlags();
	void					SetRenderFlags(uint32 flags);
	/* Get and set render flags. B_3D_ZBUFFER will enable z-buffering, 
	   B_3D_ANTIALIAS_TEXTURES does 4-point linear interpolation of texture
	   maps (note that both of these are quite expensive in software) */

	float               ZoomFactor();
	void                SetZoomFactor(float zoom);
	/* set the zoom factor of the camera. Default is 1.0. A bigger value make
	   the view closer from the object.
	   */
	

/*************************************
// LOW-LEVEL API                    */

	virtual	void			Draw(BRect updateRect);
	virtual	void			MouseDown(BPoint point);
	/*	Implements default click behavior (notifies object with TouchDown and
		TouchMoved messages) */	
	virtual	void			KeyDown(const char *bytes, int32 numBytes);
	virtual	void			MessageReceived(BMessage *msg);
	virtual	void			FrameResized(float width, float height);

	virtual	void			AttachedToWindow(void);
	/* Don't forget to call the inherited version if you want to override this
	   function. This is a good place to create the objects (things, links, lights)
	   of your scene.  Don't forget to lock the universe first!
	   */

	virtual	void			DetachedFromWindow(void);
	/* Don't forget to call the inherited method if you want to extend the abilities
	   of any of these functions.
	   */
	
/*************************************
// PRIVATE STUFF                    */

 private:
	/* Universe and whether we own it */
	bool					own_uni;
	B3dUniverse			*uni;
	/* World and whether we own it */
	bool					own_world;
	B3dWorld			*world;

	/* Benaphore + variables used to store infos about multiple locks
	   by the same thread (and check discipline) */
	int32				view_lock;
	sem_id				view_sem;
	uint32				locker_stack_page;
	thread_id				locker_pid;
	int32				locker_count;
	int32				cur_index;

	/* Offscreen buffer, described in both InterfaceKit and 3dKit format */
	BBitmap				*bitmap[2];
	camera_buffer			buffer[2];
	ZBuffer				m_zBuffer;

	/* New specification of window properties, if modified */
	float				new_width;
	float				new_height;
	int32				m_newOptionFlags;
	color_space			new_mode;

	/* The camera used by the view to create frames */
	B3dCamera				*camera;

	/* Keyboard user interaction configuration */
	B3dKeyConfig			cfg;
	
	thread_id				m_mouseTracker;
	float                   zoom_factor;
	
	static int32			LaunchMouseTracker(B3dView *view);
	void					MouseTracker();
	void					Lock();
	void					Unlock();
	void                    InitData(uint32 flags);
};

/**************************************
// INLINE FUNCTION DEFINITIONS
/*************************************/

B3dUniverse *B3dView::Universe() {
	return uni;
}


B3dWorld *B3dView::World() {
	return world;
}

#endif











