/*****************************************************************************
//
//	File:			3dUniverse.h
//
//	Description:	Main container object.
//
//	Copyright 1997, Be Incorporated
//
// ******** 3dKit ARCHITECTURE NOTE :
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
******************************************************************************/

#ifndef _3D_UNIVERSE_H
#define _3D_UNIVERSE_H

#include <OS.h>
#include <Debug.h>

#ifndef _3D_DEFS_H
#include "3dDefs.h"
#endif
#ifndef _3D_THING_H
#include "3dThing.h"
#endif


/**************************************
// HIGH-LEVEL API
/*************************************/

/* values used for the mode parameter of SetTimeMode() */
enum {
	B_REAL_TIME,    /* The universe's time is passing at real time speed */ 
	B_FROZEN_TIME,  /* The universe's time is frozen */
	B_EXPANDED_TIME /* The universe's time is the real time multiplied by a constant */
};

/**************************************
// LOW-LEVEL API
/*************************************/

/* used for the flag of AddFrameHook */
enum {
	B_HEAVY_DUTY =     0x0001,
	/* for a task long enough to require its own thread.
	   */
	B_EVERY_FRAME =    0x0002,
	/* need to be called each time the universe time change
	   */
	B_UNIVERSE_LOCK =  0x0004
	/* need exclusive access to the universe
	   */
};

/* struct use to talked wit active elements linked with the Universe.
   */
typedef struct B3dTimeBase {
	bigtime_t    universe_old;
	bigtime_t    universe_new;
	bigtime_t    system_old;
	bigtime_t    system_new;
} B3dTimeBase;

typedef void b_frame_hook(void *, B3dTimeBase *); 


/**************************************
// PRIVATE STUFF
/*************************************/

#define DEBUG_MODE  0

/* Bit masks used for the status field.
   Be careful, bits 4 to 6 are reserved for Links properties bits.
   */
enum {
	B_GROUP_MASTER = 0x0080,
	B_SCAN_FLAG = 0x0100
};


/* propagation mode for thing destruction */
enum {
	B_PROPAGATION_MASK = 0x0f,

	B_GROUP_PROPAGATION = 1,
	B_LINK_PROPAGATION = 2,
	B_NO_PROPAGATION = 3,

	B_FORCE_THING_REMOVAL = 0x10
};


/* different locking mode for the universe */
enum {
	B_LOCK_READ_AND_WRITE,
	B_LOCK_READ_ONLY
};


enum {
	B_READER_LOCK_COUNT = 8,
	/* maximum number of simultaneous readers
	   */
	B_WORLD_COUNT = 8
	/* maximum number of parallel worlds
	   */
};


/* Values define for the group member of BUniNode,
   and also the level member.
   */
enum {
	B_ROOT_THING_LEVEL = 0,
	/* graph dependencies level of root node (node without masters).
	   */
	B_ALL_GROUP_LEVEL = 255,
	/* maximal possible hierarchical group level. It's implicitly checked as the
	   hierarchical level is checked into the constructor, and the group level
	   should always be >= to the hierarchical level.
	   */
	B_LEVEL_COUNT = 256
	/* How many different hierarchical level can be encoded in the BUniNode
	   struct.
	   */
};


/* id reserved to indicate that a thing is not attached to the universe any longer.
   */
enum {
	B_INVALID_THING_ID = 0x7fffffff
};


class B3dCamera;


/* private struct, used for the node of the graph dependencies.
   */
typedef struct BUniNode {	
	B3dThing   *thing;
	/* those node are use to describe the graph of thing/link. Each node represent a
	   thing, pointed to by this pointer.
	   */
	uint16     status;
	/* bit 0-3 : UniNode size parameter n (the n of (2^n)-3 entries). 1<n<16.
	   bit 4-6 : TimeLink properties
	   bit 7   : GroupMaster flag
	   bit 8   : internal flag used during graph exploration (for cycle detection).
	   */
	uint8      level;
	/* maximal depth path linking that node to a root node.
	   */
	uint8      group;
	/* hierarchy level of this thing in the group hierarchy (no group = 0).
	   the group master is +0, the group members are +1.
	   */
	uint16     slave;
	/* count of entries representing slaves of that node.
	   */
	uint16     count;
	/* total count of entries in the link table ("list"), slave and master.
	   */
	union {
		int32        time_index;
	/* index of the pointer to this node in the time link dependent table,
	   or -1 if not time dependent.
	   */
		struct BUniNode *next;
	/* or pointer to the next unused node of that size.
	   */
	};
	B3dThing    *next_update;
	/* pointer to the next thing needing update, from the same hierarchical level.
	   */
	struct BUniNode *list[1];
	/* first element of a link table (counting 2^n-3 entries). The first entries,
	   [0, slave-1] represent the nodes depending of this node. The last ones
	   [slave, count-1] represent the nodes from which this node depends.
	   Those node are referenced by pointers.
	   */
} BUniNode;


/* private struct used for the control of the locking mechanism.
   */
typedef struct {
	uint32       stack_page;
	/* last page used by the stack of this thread.
	   */
	int32        status;
	/* status of the lock done by this thread :
	   0 : unused,
	   1 : read only,
	   2 : read and write.
	   */
	thread_id   pid;
	/* thread_id of this thread.
	   */
} BUniLock;


/* private struct used for the call-back of FrameHook functions.
   */
typedef struct {
	b_frame_hook    *hook;
	/* the hook function to be called.
	   */
	int32           flag;
	/* the flag properties of the hook (see enum about AddFrameHook).
	   */
	thread_id       caller;
	/* caller thread spawned to deal with heavy-duty hook not asking for full lock.
	   */
	void            *data;
	/* private data to be send to the hook function.
	   */
	B3dUniverse     *uni;
	/* a local copy of the universe pointer.
	   */
} BHookNode;


/* Macro defining the size of the BUniNode header in terme of BUniNode pointer.
   */
#define	B_NODE_HEADER_SIZE ((sizeof(BUniNode)/sizeof(BUniNode*))-1)


/* Macro used to calculate the maximum number of item that a node x can store
   (size of the list table of the node).
   */
#define BUniNodeMax(x) ((1<<(((x)->status)&15))-B_NODE_HEADER_SIZE)


class B3dWorld;


/**************************************
// B3dUniverse.
/*************************************/

/* The 3dUniverse class is the main container for every other objects of a
   commun space.
   */
class B3dUniverse {
    friend class B3dThing;
	friend int32 refresh_thread(void*);
	friend int32 update_master(void*);
	friend int32 caller_thread(void*);

/*************************************
// HIGH-LEVEL API                   */

 public:
    inline const char *Name();
	void              SetName(char *name);
	/* universes have names...
	   */	
	void       Lock();
	void       Unlock();
	/* before modifying anything in the state of any object (thing, light, links...)
	   of the universe, or accessing/using/setting any camera, lens, lighter and so on,
	   you have to lock the universe first, and unlock it at the end.
	   */
	void       SetDiscipline(bool flag);
	/* This enable/disable debug tests to verify the universe is properly locked before
	   being used (doesn't check every possible case, but most of them. It's a debug,
	   not a correctness guarantee). The default state is true (debug on). In case of
	   error, it will print on both the standard output and the serial output :
	   "LOCK_ERROR : stack:0xaaaaaaaa/0xbbbbbbbbb, pid:cccc/dddd, status:ee"
	   where 'aaaaaaaaa' is the adress of the stack page of the thread currently locking
	                     the universe (if any).
			 'bbbbbbbbb' is the adress of the stack page of the thread responsable for
			             the locking error.
			 'cccc' is the thread id of the thread currently locking the universe (if any)
			 'dddd' is the thread id of the thread responsable for the locking error.
			 'ee' == 0 if the universe is not locked, > 0 if locked by the wromg thread.
	   */	
	bigtime_t  Time();
	void       SetTime(bigtime_t time);
	void       SetTimeMode(int32 mode, float timer);
	/* Return the current time, set the time and set the time mode (see previous enum).
	   */
	inline uint32   UpdateCount();
	/* Return the count of updates since the creation of the universe. Can be used as
	   an unique ID to check if two events happened in the same update.
	   */	
	float      FrameRate();
	void       SetFrameRate(float rate);
	/* Return and set the current frame rate in frames/s (real time not universe's time)
	   control the refresh of all the cameras of the universe.
	   */
	uint32      ThingCount();
	B3dThing   *GetNthThing(uint32 index, int32 group_level = B_ALL_GROUP_LEVEL);
	/* Get the count of things in the universes, and return the pointer to each of them
	   by index.
	   */
	B3dThing   *FindThing(char *name, int32 group_level = 0, int32 index = 0);
	/* Return the pointer to the 'index' things of the universe called 'name'.
	   index == 0 returns the first 1. group_level is reserved for LOW-LEVEL calls.
	   */
	
/*************************************
// LOW-LEVEL API                    */

					B3dUniverse(char *name);
	virtual			~B3dUniverse();
	/* Usualy, universe are created and destroyed by B3dView classes. You should
	   not have to call this constructor.
	   */

	int32           AddFrameHook(b_frame_hook *hook, void *data, int32 flag = 0);
	status_t        RemoveFrameHook(int32 token);

	uint32			WorldCount();
	B3dWorld		*GetNthWorld(uint32 index);
	B3dWorld		*FindWorld(char *name, int32 index = 0);
	
	status_t		AddWorld(B3dWorld *world);
	status_t		RemoveWorld(B3dWorld *world);

	uint32			AddThing(B3dThing *thing, B3dWorld *world);
	status_t		RemoveThing(B3dThing *thing, int32 mode = B_NO_PROPAGATION);
	status_t		SwapThings(B3dThing *t1, B3dThing *t2);
	
	void			AddCamera(B3dCamera *camera);
	status_t		RemoveCamera(B3dCamera *camera);
	
	status_t		AddLink(	B3dThing *thing, B3dLink *link,
					 			bool check_cycle = true, int32 group_max = 0);
	status_t		RemoveLink(	B3dThing *thing, bool force_removal = false);			

	status_t		CheckGroup(int32 *count, B3dThing **list, uint32 mode, B3dThing ***new_list);
	status_t		SetGroupMaster(B3dThing *thing, bool status, bool check_cycle = true);
	B3dThing		*GroupMaster(B3dThing *thing);
	
	void			Update();

/* debug calls.	*/
	bool				CheckUniverse();
	void				DumpUniverse();
	struct B3dThingInfo	*GetThingInfo(B3dThing *thing);
	
	bool     		debug;
	bool     		spy;
	
/*************************************
// PRIVATE STUFF                    */

 private:
	char		*name;
	/* name of the universe. Only for archive purpose as universe are not subject
	   of any other class.
	   */
	bigtime_t	Time0, Time1;
	float		Timer;
	/* time system information. Need to be changed
	   */
	bigtime_t	LastUpdateTime;
	/* time of the last update, using the Update mechanism.
	   */
	int32		uni_lock;
	/* lock of the benaphore protecting the access to the universe.
	   */
	sem_id		uni_sem;
	/* semaphore of the benaphore protecting the access to the universe.
	   */
	BUniNode	**id2node;
	/* Table used to convert a thing id into a node pointer. Also used to store
	   the index of the next free index of this table.
	   */
	int32		free_id;
	/* head index of the free list in id2node.
	   */
	uint32		max_id;
	/* count of the current id2node table.
	   */
	BUniNode	*n2free_node[16];
	/* table of pointers (or NULL) to the first free node supporting 2^n-3
	   link entries.
	   */
	struct n_nw	*buffer_list;
	/* struct used to memorize all the buffers really allocated to the main
	   memory manager.
	   */
	int32		h_mask;
	/* mask use  to truncate index in the name hash table (= max_id*2-1).
	   */
	struct h_nw	*h_table;
	/* hash table of things hashed by name. Size max_id * 2.
	   */
	uint32		thing_count;
	/* count of things in the universe (= number of entries in the h_table
	   and = number of node in the graph dependencies).
	   */
	bool		discipline;
	/* lock discipline mode.
	   */
	BUniLock	locker;
	/* variable used to store the id and the status of the current locker, for
	   discipline purpose.
	   */
	uint32		last_thing_index;
	uint32		last_node_id;
	bool		id2node_modified;
	/* Cache used to allow fast access to indexed thing in direct order.
	   Memorize the last index == node, used it again if list not modified,
	   and the new index asked for is >= to the last one.
	   */
	B3dThing	**TimeList;
	/* Table to store the list of thing linked to the flow of time (without
	   dependencies solving (capacity of "max_id" items.)
	   */
	uint32		TimeListCount;
	/* Count of item currently used in the TimeList table.
	   */
	B3dThing	*UpdateTable[B_LEVEL_COUNT];
	/* table giving the first BUniNode needing update for each specific level.
	   */
	int32		current_max_level;
	/* the highest hierarchical level ever encoutered in the current database.
	   */
	B3dWorld	*world_list[B_WORLD_COUNT];
	/* current active world.
	   */
	bigtime_t	frame_delay;
	/* current frame_rate to display animation in the universe
	   */
	int32		camera_waiting_count;
	/* Count of cameras waiting for the next frame.
	   */
	sem_id		camera_waiting_sem;
	/* Semaphore control to release semaphore after update.
	   */
	sem_id		update_waiting_sem;
	/* Semaphore control to check end of update by cameras.
	   */
	struct camera_link *first_camera;
	/* first camera of the list of cameras attached to this universe.
	   */
	thread_id	update_master_id;
	/* keep track of the update_master thread id.
	   */
	uint32		update_count;
	/* id increments for each different update, turn around approximately one
	   time every 7 years.
	   */
	BHookNode   *hook_node_list;
	/* The list of hook functions needing update on a time base.
	   */
	int32       hook_node_count;
	/* The total count of element (used or not) in the hook_node_list.
	   */
	sem_id      hook_frame_in_sem;
	sem_id      hook_time_in_sem;
	sem_id      hook_out_sem;
	/* semaphores used to synchronise the caller of the heavy-duty non full lock
	   hook frame functions, needing update each frame or each time step.
	   */
	int32       caller_frame_count;
	int32       caller_time_count;
	/* count of caller threads currently spawned for frame update or time step
	   only update.
	   */
	B3dTimeBase time_base;
	/* the struct used to communicate time information to hook frame functions.
	   */
	
	void		CheckLock();
	void		CheckUnlock();

	uint32		HashName(const char *name);
	void		HashNode(BUniNode *node);
	void		UnhashNode(BUniNode *node);
	int32		FindHashIndex(char *name, int32 level, int32 index);
	void		ExpandId2Node();
	BUniNode	*GetUniNode(int32 n);
	BUniNode	*AllocUniNode(int32 count);
	void		FreeUniNode(BUniNode *node);

	status_t	AddRefLinks(	BUniNode	*node,
								B3dThing	**ref,
								int32		ref_count,
								int32		group_max,
								bool		check_cycle);
	void		AddSlaveLink(B3dThing *ref, BUniNode *node);
	void		RemoveMasterLinks(BUniNode *node);
	void		RemoveSlaveLink(BUniNode *master, BUniNode *node);

	BUniNode	*GroupMaster(BUniNode *node);
	
	bool		AssertGraphCycle(BUniNode *slave, BUniNode *master);
	void		UntagGraph(BUniNode *node);

	bool		AdjustLevel(BUniNode *node, int32 level);
	void		DownGradeLevel(BUniNode *node);
	void		UpdateGroupLevel(B3dThing *group, int32 offset);
	
	void		ExtendSelection(BUniNode *node, uint8 *set);
	bool		CheckRemoveGroupSlaves(BUniNode *node);
	bool		CheckRemoveThingOnly(BUniNode *node);
	void		RemoveSlaves(BUniNode *node);
	void		RemoveGroupSlaves(BUniNode *node);
	void		RemoveAllLinks(BUniNode *node);
	void		RemoveThingOnly(BUniNode *node);
	void		RemoveUnlinkNode(BUniNode *node);
	void		ExtractFromGroup(BUniNode *node);
	void		UpdateUniNode(BUniNode *node, BUniNode *from, BUniNode *to);
	void		UpdateHashTableNode(BUniNode *from, BUniNode *to);
	void		UpdateThingRefInSlaveLink(BUniNode *node, B3dThing *new_thing);
	void		UpdateThingRefInGroupMaster(BUniNode *node, B3dThing *new_thing);

	inline void	SetForUpdate(uint32 id);
	void		CancelUpdate(uint32 id);
	
	void		DrawFrame(B3dCamera *camera);
	
/* debug calls.*/
	void		Spy(BUniNode *n, char *label);
};

#ifndef DEBUG_MODE
 #define SPY(n, s) {}
 #define debug true
#else
 #define SPY(n, s) if (spy) Spy(n, s)
#endif

/**************************************
// INLINE FUNCTION DEFINITIONS.
/*************************************/

const char *B3dUniverse::Name() {
	return name;
}


uint32 B3dUniverse::UpdateCount() {
	return update_count;
}


void B3dUniverse::SetForUpdate(uint32 id) {
	int32		level;
	BUniNode	*n;

	n = id2node[id];
	level = n->level;
	n->next_update = UpdateTable[level];
	UpdateTable[level] = n->thing;
}

#endif

























