/*****************************************************************************
//
//	File:			3dThing.h
//
//	Description:	Generic description of a 3d object.
//
//	Copyright 1997, Be Incorporated
//
// ******** 3dKit ARCHITECTURE NOTE :
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
******************************************************************************/

#ifndef _3D_THING_H
#define _3D_THING_H

#include <unistd.h>

#ifndef _3D_DEFS_H 
#include "3dDefs.h"
#endif
#ifndef _RGBA_COLOR_H 
#include "RGBAColor.h"
#endif
#ifndef _3D_LINK_H 
#include "3dLink.h"
#endif
#ifndef _MESSAGE_H 
#include "Message.h"
#endif

class B3dUniverse;
class B3dThingLink;
class B3dWorld;
class B3dLighter;
class B3dLens;
class B3dRenderer;
class B3dCamera;
class B3dTouchDesc;
class B3dSharedState;

/**************************************
// HIGH-LEVEL API
/*************************************/

/* possible values to define the way you want to look at a target with LookAt */
enum {
	B_HORIZONTAL_VIEW   /* try to preserve the horizontal plan as the "ground".*/
};

/* possible values for the assert parameter of SetRotation */
enum {
	B_NO_ASSERT,    /* set the rotation matrix without checking the 'rotation property'.*/
	B_ASSERT_ON_X,  /* set and check the 'rotation property' by preserving the X axis.*/
	B_ASSERT_ON_Y,  /* set and check the 'rotation property' by preserving the Y axis.*/
	B_ASSERT_ON_Z   /* set and check the 'rotation property' by preserving the Z axis.*/
};

/* flags for group creation mode, used with group_things for the mode parameter */
enum {
	B_GROUP_MODE_MASK = 0x10,

	B_EXACT_GROUP =     0x00,   /* Check that no object ouside the group is using an object*/
                                /* of the group as reference (a group can only have internal*/
								/* links).*/
	B_EXTEND_GROUP =    0x10    /* Add to the group any object linked to one of its member*/
		                        /* (insures that there will not be any external links). You*/
								/* can also create a group by selecting the root of a graph*/
								/* of links and extend it to all members of the graph just*/
								/* by using this mode.*/
};

/**************************************
// LOW-LEVEL API
/*************************************/

/* properties of the status field. */
enum {
	B_THING_TYPE_MASK =   0x007f,  /* mask used to extract the type from the status*/
	B_IS_THING =          0x0000,  /* default value for a basic thing (used for group master)*/
	B_IS_BODY =           0x0001,  /* types of sub-classes*/
	B_IS_LIGHT =          0x0002,

	B_NEED_RENDERING  =   0x0080,  /* this thing is affecting rendering.*/
		
	B_ZOOM_ONLY =         0x0100,  /* this thing has a twist used for zoom only*/

	B_RADIUS_UPDATE =     0x0200,  /* the steric radius has been modified*/
	B_MOVE_UPDATE =       0x0400,  /* the origin of the thing has been modified*/
	B_ROTATE_UPDATE =     0x0800,  /* the rotation of the thing has been modified*/
	B_TWIST_UPDATE =      0x1000,  /* the twist of the thing has been modified*/
	B_GEOMETRIC_UPDATES = 0x1e00,  /* mask to test we need any geometric update*/
	B_STRUCT_UPDATE =     0x2000,  /* the master list of world should be update*/
	B_ALL_UPDATES =       0x3e00,  /* mask to test we need any update.*/

	B_VIEW_POINT =        0x4000,  /* can be used as a ViewPoint   */
	B_BACKGROUND_THING =  0x8000,  /* the object is always behind any normal object*/
	B_ENVIRONMENT_THING = 0x10000, /* the thing is part of the environement, is behind*/
		                           /* even the background, and do not translate.*/

	B_PROPERTIES_MASK =   0xffc000, /* global masks for all properties.*/
		
	B_LINK_2_THING =         6,  /* count of left shift to go from link coding to*/
		                         /* thing coding. Should be convert by a big macro,*/
		                         /* to go from one mask_bit to another (63 cases,*/
							     /* all possible shifts and equal). */
	B_THING_UPDATE_SHIFT =   9,  /* number of right shifts to get the Updates bits*/
		                         /* in position 3 to 0.*/
	B_LINK_TYPE_SHIFT =      4   /* number of left shifts to get the link type bits */
		                         /* in position 7 to 4.*/
};

/* struct used to return infos about the links of a thing in the universe */
typedef struct B3dThingInfo {
	long       link_level;
	long       group_level;
	long       slave_max_index;
	long       master_max_index;
	B3dThing   *list[1];
} B3dThingInfo;


/**************************************
// PRIVATE STUFF
/*************************************/

typedef struct {
	float     zoom;
	B3dMatrix rotation;
	B3dMatrix twister;
} B3dTwist;

typedef struct {
 public:
	long      count;
	ulong     mode;
	B3dThing  *list[1];
} B3dGroup;

B3dGroup *NewGroup(long count);


/**************************************
// B3dThing.
/*************************************/

class B3dThing {	
	friend long group_things(long,B3dThing**,B3dThing**,char*,B3dVector*,float,ulong);
	friend long group_things(long,B3dThing**,B3dThing*,B3dVector*,float,ulong);

/*************************************
// HIGH-LEVEL API                   */

 public:
	virtual B3dThing		*Clone(char *name = 0L, ulong clone_flag = 0);
	/* Create another instance of a 3dThing, but shared its description. That the
	   most efficient way of getting multiple copies of the same object. You can
	   change the name of the clone. The flag is unused for now. */

	inline const char		*Name();
	void					SetName(char *name);
	/* Return or change the name of the thing */

	inline B3dUniverse		*Universe();
	/* Return the universe in which the thing exists */	

	inline B3dWorld		    *World();
	/* Return the world in which the thing exists */	

	int32				LinkTo(B3dLink *link, ulong mode = B_CHECK_CYCLE);
	/* Apply a link to this object. The link must be created first, describing
	   how the state of the slave should depend of the state of some masters
	   and the flow of time. This method set this object as being the slave of
	   that link. If this thing was previously linkto another link, that
	   previous link is free. The flag B_CHECK_CYCLE will check that creating the
	   link will not generate unsolvable dependencies (cycle). This is much safer,
	   but can be extremely time consuming on very complex groups. */

	int32				Unlink(ulong mode = 0L);
	/* Release the link on this thing (if any), and free the link. mode is used
	   by advanced low-level calls only. */	

	int32				LookAt(B3dVector *target, ulong mode = B_HORIZONTAL_VIEW);
	inline int32			LookAt(B3dThing *target, ulong mode = B_HORIZONTAL_VIEW);
	/* Turn the thing to have its main axis (X) looking straight to the current
	   position of the target. mode define which way the others axis ahould be
	   oriented. */

	inline bool			IsGroup();
	/* Return true if the thing is a group (not just a basic thing). */

	B3dThing            *GroupMaster();
	/* return the master of your group if you're member of a group, 0L in the
	   other case. */

	int32				GroupCount();
	/* Return the count of group members */
	B3dThing				*GetNthThing(uint32 n);
	/* Return the Nth group member */

	int32				TieThing(B3dThing *thing, float threshold_size);
	/* Create a LOD thing by using the current object as the simplified version,
	   and "thing" as the complex version. The renderer will use the simple one
	   when the apparent size of the object on screen (in pixel) is smaller than
	   the specified threshold. Both simple and complex version should have the
	   same global size to get a good result. */

	int32				TieGroup(B3dThing *group, float threshold_size);
	/* Do the same than TieThing, but this time the complex version is a group
	   of many things. */

	B3dThing				*UntieGroup(char *name);
	/* Break a LOD-thing, keeping the simplified version in the current thing and
	   returning a new thing that contains the complex version. If the LOD thing
	   was created using TieThing, then the complex thing is return in a group of
	   only one thing. */

	inline const B3dVector	*Origin();
	void					SetOrigin(float new_x, float new_y, float new_z);
	void					SetOrigin(B3dVector *new_origin);
	/* Return or set the current position of the origin (usually the center) of the
	   thing. */

	inline const B3dMatrix *Rotation();
	void                   SetRotation(B3dMatrix *new_rotation, long assert = B_NO_ASSERT);
	/* Return or set the current rotation matrix of the thing, describing how the
	   thing is oriented in space. */

	B3dVector				Object2World(B3dVector vectorInObjectSpace);
	B3dVector				World2Object(B3dVector vectorInWorldSpace);
	/* Handy routines to transform world coordinates into object coordinates and back */

	virtual bool			GetTouchInfo(	B3dVector *		origin,
											B3dVector *		axis,
											B3dVector *		touch,
											void **			personalInfo);
	/* Asks the object if the specified ray intersects the object.  If it does,
	   this should return true and the coordinates of the intersection point (in
	   object space) should be in "touch".  "personalInfo" can be used to
	   store any state information the object needs about this touch */
	virtual void			FreeTouchInfo(		void *			personalInfo);
	/* Frees any personalInfo allocated by GetTouchInfo, is called after the touch
	   is released */
	   
	virtual void			TouchDown(			B3dTouchDesc *	touch,
											B3dVector *		touchOnScreen,
											uint32			buttons);
	/* Called to inform the object that it has been touched */
	
	virtual void			TouchMoved(			B3dTouchDesc *	touch,
											B3dVector *		touchOnScreen,
											uint32			buttons);
	/* Called during the duration of a touch whenever the state of the touch
	   changes */

	virtual void			MessageReceived(BMessage *message, B3dTouchDesc *touch);
	/* By default does nothing */
	
/*************************************
// LOW-LEVEL API                    */

	B3dThing(char *name, ulong status, B3dWorld *world);
	B3dThing(B3dThing *thing, char *name, ulong clone_flag);
	virtual				~B3dThing();
	inline long			Type();
	inline ulong			Properties();
	void					SetRadius(float r);
	void					SetProperties(ulong mode);
	long					UnGroup(B3dGroup **group);
	long					GroupAgain(B3dGroup *group, ulong mode = B_CHECK_CYCLE); 
	virtual void			Debug();
	void					GetInfo(B3dThingInfo **info);
	inline float			Zoom();
	void					SetZoom(float zoom = 1.0);
	inline const B3dMatrix	*Twist();
	long					SetTwist(B3dMatrix *mat = 0L);
	void					Update(bigtime_t curTime, bigtime_t prevTime);
	virtual void			Draw(B3dLighter  *lighter,
								B3dLens     *projector,
								B3dRenderer *renderer,
								B3dVector   *offset0,
								B3dMatrix   *rot,
								B3dSharedState *buf);

	ulong       status;
	/* type of the thing and others properties (see before).
	   */
	ulong       id;
	/* id of the node associated to the thing in the universe
	   */
	float       threshold;
	/* threshold size under which the group is represented by its
	   simple thing substitution.
	   */
	float       radius;
	/* current radius of the sphere including the thing 3d model
	   */
	float       radius0;
	/* same value before twist or resize (for Zoom = 1.0)
	   */
	B3dLink     *link;
	/* pointer to the optional link with other things
	   */
	B3dTwist    *twist;
	/* twist descriptor (optional)
	   */
	B3dGroup    *group;
	/* group descriptor (optional), when the thing is a group.
	   The thing is a group == (group != 0L).
	   */
	B3dVector   origin;
	/* current origin point
	   */
	B3dMatrix   rotation;
	/* current linear transformation matrix (can be more than a
	   simple rotation matrix when the twist is enabled
	   */
	RGBAColor   highlightColor;
	/* highlight substitution color for selection
	   */
	B3dUniverse *uni;
	/* universe owner pointer
	   */
	B3dWorld    *world;
	/* pointer to the world to which the thing of this node is linked to.
	   */
	B3dThingLink *world_link;
	/* link to the entry representing the thing of this node in its world.
	   */

/*************************************
// PRIVATE STUFF                    */

 private:
	char           *name;
	/* name of the thing (dynamic size).
	   */

    inline void    AssertTwist();
	inline void    AssertUpdate(ulong update_mode);
};

/**************************************
// HIGH-LEVEL API
/*************************************/

long group_things(long      count,
				  B3dThing  **list,
				  B3dThing  **group,
				  char      *name,
				  B3dVector *origin = 0L,
				  float     threshold = -1.0,
				  ulong     mode = B_EXTEND_GROUP | B_RIGID_LINK | B_CHECK_CYCLE);
/* Make a new group named 'name' by grouping the 'count' things of 'list'.
   You can say where you want the origin of the group to be using 'origin'. If not
   (origin = 0L), the origin will be put at the center of the smallest box including
   all objects of the group (the box being X, Y and Z oriented only). The threshold
   is the apparent size on screen (in pixel) below which you don't want the group
   to be displayed (-1.0 means always visible). The mode flag contains 3 types of
   informations :
   - Group extension : see comments about B_EXACT_GROUP and B_EXTEND_GROUP.
   - Group link type : see comments about link type in B3dLink.h
   - Link cycle detection : see comments about LinkTo.
   */

/**************************************
// LOW-LEVEL API
/*************************************/

long group_things(long      count,
				  B3dThing  **list,
				  B3dThing  *group,
				  B3dVector *origin = 0L,
				  float     threshold = -1.0,
				  ulong     mode = B_EXTEND_GROUP | B_RIGID_LINK | B_CHECK_CYCLE);
/* The same than the previous function, but this time the group is return into a
   already existing thing.
   */

/**************************************
// INLINE FUNCTION DEFINITIONS.
/*************************************/

const char *B3dThing::Name() {
	return name;
}

long B3dThing::Type() {
	return (status & B_THING_TYPE_MASK);
}

ulong B3dThing::Properties() {
	return status & B_PROPERTIES_MASK;
}

long B3dThing::LookAt(B3dThing *target, ulong mode) {
	return LookAt(&target->origin, mode);
}

bool B3dThing::IsGroup() {
	return (group != 0L);
}

const B3dVector *B3dThing::Origin() {
	return &origin;
}

const B3dMatrix *B3dThing::Rotation() {
	if (twist == 0L)
		return &rotation;
	else
		return &(twist->rotation);
}

float B3dThing::Zoom() {
	if ((twist != 0L) && (status & B_ZOOM_ONLY))
		return twist->zoom;
	else
		return 1.0;
}

const B3dMatrix *B3dThing::Twist() {
	if ((twist == 0L) || (status & B_ZOOM_ONLY)) return 0L;
	return &twist->twister;
}

B3dUniverse *B3dThing::Universe() {
	return uni;
}

B3dWorld *B3dThing::World() {
	return world;
}

#endif









