/*****************************************************************************
//
//	File:			3dLink.h
//
//	Description:	Standard 3dLink for commun cases.
//
//	Copyright 1997, Be Incorporated
//
// ******** 3dKit ARCHITECTURE NOTE :
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
*****************************************************************************/

#ifndef _3D_STD_LINKS_H_
#define _3D_STD_LINKS_H_

#ifndef _3D_LINK_H_
#include <3dLink.h>
#endif

/**************************************
// B3dAxisLink.
/*************************************/

class B3dAxisLink: public B3dLink {

/*************************************
// HIGH-LEVEL API                   */

 public:
	B3dAxisLink(	float		speed,
					B3dVector	*axis,
					B3dVector	*origin = 0L);
	/* Create a permanent rotation around a fixed axis of space, defined
	   by the point 'origin' (the origin of space by default) and the vector
	   'axis'. The rotation speed 'speed' is given in radian/s (6.28 radians
	   per rotation).
	   */
	B3dAxisLink(	float		speed,
					B3dThing	*center,
					B3dVector	*axis = 0L,
					B3dVector	*origin = 0L);
	/* The same than the previous one, but this time the reference point is
	   defined relative to an object 'thing'. By default ('origin = 0'), it's
	   the origin of the thing itself. In the other case, it's a point in the
	   local referential of the thing.
	   */

/*************************************
// LOW-LEVEL API                    */

	virtual void DoLink(B3dThing *Object, bigtime_t time0, bigtime_t time1);
	virtual int32 GetRefs(B3dThing ***refs);

/*************************************
// PRIVATE STUFF                    */

 private:
	int32		mode;
	float		speed;
	float		radius,	dz;
	B3dThing	*center;
	B3dVector	axis;
	B3dVector	origin;
};

/**************************************
// B3dJigLink.
/*************************************/

class B3dJigLink: public B3dLink {

/*************************************
// HIGH-LEVEL API                   */

 public:
	B3dJigLink(	float		rot_alpha,
				float		rot_theta,
				float		rot_phi,
				B3dVector	*origin = 0L);
	/* Create a spherical rotation around a fixed point of space, defined
	   by the point 'origin' (the origin of space by default). rot_alpha,
	   rot_theta and rot_phi give the rotation speed around the 3 spherical
	   axis in radian/s.
	   */
	B3dJigLink(	float		rot_alpha,
				float		rot_theta,
				float		rot_phi,
				B3dThing	*center);
	/* The same, but tuning around the origin of the thing 'center'.
	   */

/*************************************
// LOW-LEVEL API                    */

	virtual void SetSlave(B3dThing *slave);
	virtual void DoLink(B3dThing *Object, bigtime_t time0, bigtime_t time1);
	virtual int32 GetRefs(B3dThing ***refs);

/*************************************
// PRIVATE STUFF                    */

 private:
	float		Pos[3];
	float		Speed[3];
	B3dVector	Port;
	B3dVector	*Origin;
};

/**************************************
// B3dOrbitLink.
/*************************************/

class B3dOrbitLink: public B3dLink {

/*************************************
// HIGH-LEVEL API                   */

 public:
	B3dOrbitLink(	float		speed,
					B3dThing	*center,
					B3dVector	*axis,
					B3dVector	*origin = 0L);
	/* Create a permanent rotation around an axis of space, defined by a point
	   of the thing 'center' (at the internal position 'origin' in the thing
	   own referential) and the direction 'axis'.  The rotation speed 'speed'
	   is given in radian/s (6.28 radians per rotation).
	   */

/*************************************
// LOW-LEVEL API                    */

	virtual void SetSlave(B3dThing *slave);
	virtual void DoLink(B3dThing *Object, bigtime_t time0, bigtime_t time1);
	virtual int32 GetRefs(B3dThing ***refs);

/*************************************
// PRIVATE STUFF                    */

 private:
	int32		mode;
	float		speed, h_radius;
	float		angle;
	B3dThing	*master;
	B3dMatrix	ref;
	B3dVector	origin;
};

#endif












