/*****************************************************************************
//
//	File:		    3dLighter.h
//
//	Description:	Engine used to control light processing.
//
//	Copyright 1997, Be Incorporated
//
*****************************************************************************/

#ifndef _3D_LIGHTER_H
#define _3D_LIGHTER_H

#include <InterfaceDefs.h>

#ifndef _3D_DEFS_H 
#include "3dDefs.h"
#endif
#ifndef _3D_AMBIENT_LIGHT_H 
#include "3dAmbientLight.h"
#endif
#ifndef _3D_PARALLEL_LIGHT_H 
#include "3dParallelLight.h"
#endif
#ifndef _3D_RADIAL_LIGHT_H 
#include "3dRadialLight.h"
#endif
#ifndef _3D_LIGHT_PROC_H 
#include "3dLightProc.h"
#endif


/* maximum number of each type of light that can be process for a specific
   point at any time.
   */
enum {
	B_MAX_LIGHT_COUNT = 32
};


/* options of the default 3dLighter class.
   */
typedef struct B3dLighterOptions {
	long      max_parallel_count;
	/* maximun number of parallel light that can be process for
	   one object. Should  be <= B_MAX_LIGHT_COUNT.
	   */
	long      max_radial_count;
	/* maximun number of radial light that can be process for
	   one object. Should  be <= B_MAX_LIGHT_COUNT.
	   */
	float     linear_radial_ratio;
	/* ratio distance/radius from which the radial light can be
	   processed as simple parallel light of equivalent parameters.
	   */
	float     cut_radial_threshold;
	/* fraction of nominal power under which radial light are
	   cut off for the lighting of a specific object.
	   */
} B3dLighterOptions;


/*
	General descriptor of a light source, use for real time light
	computation (common for both radial and parallel, and reusable)
*/
class B3dLightProcDesc {
 public:
	RGBAColor		color; 				/* color of the light source, modified by the current material*/
	RGBAColor		color_alone;			/* color of the light source unmodified by the material		*/
	RGBAColor		specularColor; 		/* color of the light source highlight					*/
	RGBAColor		specularColor_alone;	/* color of the light source highlight unmodified by the material*/
	RGBAColor		color_ref;			/* backup of the standard color of the light source			*/
	B3dVector		vector;				/* light source direction for parallel light				*/
	B3dVector		vectorInObj;			/* light source direction in object space					*/
	B3dVector		origin;				/* light source origin for radial light 
										(we need both for on fly optimization)				*/
	B3dVector		originInObj;			/* light source origin in object space					*/
	float		power;				/* the power of the radial light source					*/
	float		threshold;			/* threshold parameter (in a specific format) 
									for radial light									*/
};

/* sub-class handling lighting with only ambient light */
class B3dLightProc_1A : public B3dLightProc {

	public:

		B3dLightInfo		ambient_color;
		RGBAColor			ambient_color_alone;
		int32 			materialCount;

virtual	void				Light(			B3dLensImage *projDesc, 
										uint16 pt, uint16 norm,
										PNLD *pnld);
virtual	void				SetMaterial(		B3dMaterial *material);
};

/* sub-class handling lighting with only parallel and ambient lights */
class B3dLightProc_1P : public B3dLightProc {

	public:

		int				parallel_count;
		B3dLightInfo *	lighting_buffer;
		RGBAColor			ambient_color;
		RGBAColor			ambient_color_alone;
		bool				needSeparateSpecular;
		bool				useSpecular;
		float			specularCoeff;
		int32 			materialCount;
		B3dLightProcDesc *	parallelProcs[B_MAX_LIGHT_COUNT];

virtual	void				SetLightContext(	B3dMatrix *rotation,
										B3dVector *translation);
virtual	void				Light(			B3dLensImage *projDesc, 
										uint16 pt, uint16 norm,
										PNLD *pnld);
virtual	void				SetMaterial(		B3dMaterial *material);
};

/* sub-class handling lighting with all radial, parallel and ambient lights */
class B3dLightProc_1R : public B3dLightProc {

	public:

		int				parallel_count;
		int				radial_count;
		B3dLightInfo *	lighting_buffer;
		RGBAColor			ambient_color;
		RGBAColor			ambient_color_alone;
		bool				needSeparateSpecular;
		bool				useSpecular;
		float			specularCoeff;
		int32 			materialCount;
		B3dLightProcDesc *	parallelProcs[B_MAX_LIGHT_COUNT];
		B3dLightProcDesc *	radialProcs[B_MAX_LIGHT_COUNT];

virtual	void				SetLightContext(	B3dMatrix *rotation,
										B3dVector *translation);
virtual	void				Light(			B3dLensImage *projDesc, 
										uint16 pt, uint16 norm,
										PNLD *pnld);
virtual	void				SetMaterial(		B3dMaterial *material);
};


class B3dWorld;


/* Lighter class.
   */
class B3dLighter {
 public:
	B3dLighter();
	virtual              ~B3dLighter();

	virtual void         CollectLight(B3dWorld *world);
	virtual B3dLightProc *SelectLight(B3dThing *thing, int pointCount, int vectCount);

	virtual void         *GetOptions();
	virtual void         SetOptions();

 private:
	int					parallel_count, radial_count;
	int					parallel_max, radial_max;
	int					buffer_count;
	char				*linear_flags;
	float				*radial_coeffs;
	B3dLightInfo		*lighting_buffer;
	RGBAColor			ambient_color;
	B3dLightProc_1A		lp_A;
	B3dLightProc_1P		lp_P;
	B3dLightProc_1R		lp_R;
	B3dLightProcDesc	*parallel_list;
	B3dLightProcDesc	*radial_list;
	B3dLighterOptions	options;
	
	void    PresetLightProc();
};

#endif
