/*****************************************************************************
//
//	File:			3dDefs.h
//
//	Description:	Shared definitions.
//
//	Copyright 1997, Be Incorporated
//
// ******** 3dKit ARCHITECTURE NOTE :
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
*****************************************************************************/

#ifndef _3D_DEFS_H
#define _3D_DEFS_H

#include <SupportDefs.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/**************************************
// HIGH-LEVEL API
/*************************************/

enum {
	B_EXTERNAL_LINK_GROUP_ERROR = -1,
	/* You tried to create a group in B_EXACT_GROUP mode that includes a link to a thing
	   external to the group. Only the group itself can have links external to the group. */

	B_EMPTY_GROUP_ERROR         = -2,
	/* You tried to create a group without any member. */

	B_MULTIPLE_UNIVERSE_ERROR = -3,
	/* The members of teh group you tried to create are not all in the same universe.
	   Only Zeus is allowed to use multiple universes in the same group. */

	B_ALREADY_GROUP_MEMBER_ERROR = -4,
	/* You tried to add a thing to the same group twice. */

	B_MULTIPLE_THING_ERROR = -5,
	/* You tried to create a group with two copies of the same thing. */

	B_FROZEN_LINK_ERROR = -6,
	/* You tried to break an internal link of a group. You must UnGroup the group first. */

	B_LOOP_LINK_ERROR = -7,
	/* You tried to create a link (or a group) creating an illegal dependency (cycle). */

	B_NO_LINK_ERROR = -8,
	/* You tried to UnLink a thing that is not linked to anything. */

	B_ALREADY_GROUP_ERROR = -9,
	/* You tried to use a group in place of a basic thing. */

	B_TOO_MANY_LEVELS_ERROR = -10,
	/* You tried to create a link that would create too many levels of dependencies
	   between recursive masters and slaves. */

	B_INDIRECT_TWIST_ERROR = -11,
	/* You tried to apply an indirect twist transformation. Only Direct twist are
	   allowed (the referential of the twist matrix must be direct). */

	B_NO_INTERSECTION = -12
	/* Returned by B3dFaceBody::SurfaceIntersection if the ray does not intersect the
	   object, or if it does not intersect the object N or more times when nthIntersection
	   is > 0 */
};

/* Constants and typedefs for the B3dFace API */

/* selection_ids are used for all B3dFace manipulation */
typedef uint32 selection_id;

#define B_BAD_SELECTION_ID		0
#define B_NO_SELECTION_ID		0
#define B_OBJECT_SELECTION_ID	1
#define B_BAD_ELEMENT_NUMBER	-1

/*	Levels of optimization.  Should be passed to B3dFaceBody::Optimize to
	specify how much effort we want to expend to make sure the object
	is packed tightly. */
enum {
	B_OPTIMIZE_LIGHT=0,
	/*	Garbage collection.  Removes unused points,
		normals, point looks, and materials.  Guarentees
		the sharing state of the object will not change.
		(i.e. will not merge points, normals, etc.)  */
	B_OPTIMIZE_MEDIUM,
	/*	This level takes advantage of surface continuity
		to reduce memory use and (in some cases) increase
		performance.  Merges identical points.  Merges identical
		normals which share the same point look.  Merges
		identical point looks which share the same normal.
		Will convert down to lower grade model or look if
		doing so after these merges loses no information. */
	B_OPTIMIZE_HEAVY,
	/*	This level basically says, "I'm going to be spending
		the next hour drawing this thing, so make it fast."
		Packs the object very tightly.  Editing after doing
		this will require some internal gymnastics which are
		probably real-time but probably shouldn't be done on
		a regular basis. */
	B_OPTIMIZE_TEAPOT
	/*	Applies the famous "teapot optimization" to the object.
		Usually makes the object draw up to 10,000 times faster.
		Can be applied twice for speed increases of 100,000,000:1. */
};

/*	Special identifiers that can be used in place of
	specific element numbers, in most cases.  (Used
	for the "which*" arguments to B3dFace methods) */
enum {
	B_ALL_ELEMENTS			= -2,

	B_ALL_POINTS 				= -2,
	B_ALL_NORMALS 			= -2,
	B_ALL_POINTLOOKS 			= -2,
	B_ALL_FACES 				= -2,
	B_ALL_TEAPOTS 			= -2,

	B_ONE_TO_ALL_POINTS 		= -3,
	B_ONE_TO_ALL_NORMALS 		= -3,
	B_ONE_TO_ALL_POINTLOOKS 	= -3,
	B_ONE_TO_ALL_FACES 		= -3,
	B_ONE_TO_ALL_TEAPOTS 		= -3
};

/* Selection operations */
enum {
	B_SELECTION_UNION=0,
	B_SELECTION_INTERSECTION,
	B_SELECTION_XOR,
	B_SELECTION_SUBTRACTION
};

/* Types of elements that can be selected */
enum {
	B_3DSELECT_NONE		= 0x0000,
	B_3DSELECT_POINTS		= 0x0001,
	B_3DSELECT_NORMALS		= 0x0002,
	B_3DSELECT_POINTLOOKS	= 0x0004,
	B_3DSELECT_FACES		= 0x0008,
	B_3DSELECT_ALL		= 0x000F
};

/* Ways to merge elements of a selection */
enum {
	B_MERGE_AVERAGED	= 1,
	/*	The unified element should assume the value of the average of all the elements */
	B_MERGE_ONE		= 2
	/*	The unified element should assume the value of one of the individuals */
};

enum {
	B_NO_NORMAL 		= -1,
	B_NO_POINTLOOK	= -1
};

/* Rendering options */
enum {
	B_3D_ZBUFFER 				= 0x00000001,
	B_3D_ANTIALIAS_TEXTURES		= 0x00000002,
	B_3D_BLACK_BACKGROUND       = 0x00000004
};

/*	Creation flags, taken by some constructors */
enum {
/*	These are ownership flags.  Passing B_OWN_X means that I am giving
	over ownership of the list or structure specified (meaning the object
	must free it).  The intention is to minimize memory copying.  B_OWN_DESC
	means I am giving over ownership of the description structure that I
	am passing in.  (i.e. B3dFaceModelDesc, B3dSpinDesc, etc.) */
	B_OWN_RENDER_IDS	= 0x0001,
	B_OWN_MATERIALS	= 0x0002,
	B_OWN_POINTLOOKS	= 0x0004,
	B_OWN_POINTS		= 0x0008,
	B_OWN_NORMALS		= 0x0010,
	B_OWN_FACES		= 0x0020,
	B_OWN_DESC		= 0x0040
};

#endif
