/*****************************************************************************
//
//	File:			RGBAColor.h
//
//	Description:	4 components. floating-point RGBA color object.
//
//	Copyright 1997, Be Incorporated
//
// ******** 3dKit ARCHITECTURE NOTE :
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
******************************************************************************/

#ifndef _RGBA_COLOR_H
#define _RGBA_COLOR_H

/**************************************
// B3dView.
/*************************************/

class RGBAColor {

/*************************************
// HIGH-LEVEL API                   */

 public:
	float  red;
	float  green;
	float  blue;
	float  alpha;
	/* 0.0 refers to black, 1.0 to white. Negative values are
	   meaningless, values higher than 1.0 will oversaturated
	   (supported for lights in some rendering modes) */
	inline RGBAColor& Set(float r, float g, float b, float a = 0.0);
	/* Set the color */
	
/*************************************
// LOW-LEVEL API                    */

	inline RGBAColor operator* (const RGBAColor& c) const;
	inline void operator*= (const RGBAColor& c);
	/* multiply 2 colors component per component, for lighting */
	inline RGBAColor operator* (const float f) const;
	inline void operator*= (const float f);
	/* Change intensity. Do not check saturation */
	inline RGBAColor operator+ (const RGBAColor& c) const;
	inline void operator+= (const RGBAColor& c);
	/* Add colors. Do not check saturation */
	inline RGBAColor operator- (const RGBAColor& c) const;
	inline void operator-= (const RGBAColor& c);
	/* Subtract colors. Do not check saturation */
	inline bool operator== (const RGBAColor& c) const;
	inline bool operator!= (const RGBAColor& c) const;
/*************************************
// PRIVATE STUFF                    */

	inline RGBAColor operator^ (const RGBAColor& c) const;
	inline void operator^= (const RGBAColor& c);
};

/**************************************
// INLINE FUNCTION DEFINITIONS
/*************************************/

bool RGBAColor::operator== (const RGBAColor& c) const
{
	return ((red==c.red) && (green==c.green) && (blue==c.blue) && (alpha==c.alpha));
};

bool RGBAColor::operator!= (const RGBAColor& c) const
{
	return ((red!=c.red) || (green!=c.green) || (blue!=c.blue) || (alpha!=c.alpha));
};

RGBAColor& RGBAColor::Set(float r, float g, float b, float a) {
	red = r;
	green = g;
	blue = b;
	alpha = a;
	return *this;
}

RGBAColor RGBAColor::operator* (const RGBAColor& c) const {
	RGBAColor    c2;

	c2.red = red*c.red;
	c2.green = green*c.green;
	c2.blue = blue*c.blue;
	c2.alpha = alpha*c.alpha;
	return c2;
}

void RGBAColor::operator*= (const RGBAColor& c) {
	red *= c.red;
	green *= c.green;
	blue *= c.blue;
	alpha *= c.alpha;
}

RGBAColor RGBAColor::operator* (const float f) const {
	RGBAColor    c2;

	c2.red = red*f;
	c2.green = green*f;
	c2.blue = blue*f;
	c2.alpha = alpha*f;
	return c2;
}

void RGBAColor::operator*= (const float f) {
	red *= f;
	green *= f;
	blue *= f;
	alpha *= f;
}

RGBAColor RGBAColor::operator+ (const RGBAColor& c) const {
	RGBAColor    c2;

	c2.red = red+c.red;
	c2.green = green+c.green;
	c2.blue = blue+c.blue;
	c2.alpha = alpha+c.alpha;
	return c2;
}

void RGBAColor::operator+= (const RGBAColor& c) {
	red += c.red;
	green += c.green;
	blue += c.blue;
	alpha += c.alpha;
}

RGBAColor RGBAColor::operator- (const RGBAColor& c) const {
	RGBAColor    c2;

	c2.red = red-c.red;
	c2.green = green-c.green;
	c2.blue = blue-c.blue;
	c2.alpha = alpha-c.alpha;
	return c2;
}

void RGBAColor::operator-= (const RGBAColor& c) {
	red -= c.red;
	green -= c.green;
	blue -= c.blue;
	alpha -= c.alpha;
}

RGBAColor RGBAColor::operator^ (const RGBAColor& c) const {
	RGBAColor    c2;

	c2.red = red*c.alpha+c.red;
	c2.green = green*c.alpha+c.green;
	c2.blue = blue*c.alpha+c.blue;
	c2.alpha = alpha;
	return c2;
}

#endif

 






