/*****************************************************************************
//
//	File:			3dLink.h
//
//	Description:	Class used to describe the link between a slave and its masters.
//
//	Copyright 1997, Be Incorporated
//
// ******** 3dKit ARCHITECTURE NOTE :
// The 3dKit is a fully object-oriented library, going from low-level 3d
// engine to high-level 3d API. The whole system is decomposed in functional
// blocks, that can include both high-level and low-level API. To allow all
// those classes to communicate smoothly and efficiently, most of their datas
// and methods are public. That can introduce some confusion in the sense that
// reading the headers will not allow developer to differenciate API levels
// by their protection only. That's why some comments were added in the key
// classes of the 3dKit to help you recognize which API you should use
// depending of the level of usage you want.
*****************************************************************************/

#ifndef _3D_LINK_H
#define _3D_LINK_H

#ifndef _3D_DEFS_H 
#include "3dDefs.h"
#endif
#ifndef _3D_MATRIX_H 
#include "3dMatrix.h"
#endif

/**************************************
// HIGH-LEVEL API
/*************************************/

/* link attributes definition */
enum {
	B_TRANSLATION_MASK       = 0x03,
	
	B_ABSOLUTE_TRANSLATION   = 0x00,
	B_PARALLEL_TRANSLATION   = 0x01,
	B_ROTATED_TRANSLATION    = 0x02,
	B_TWISTED_TRANSLATION    = 0x03,
	/* Those 4 flags specify the type of dependency between the position of the slave
	   of the link and some properties of its masters. This is used to optimize the
	   performance when updating the state of all the objects of the universe, after
	   an user interaction or a time change.
	   - B_ABSOLUTE_TRANSLATION : the slave position doesn't depend on its masters.
	   - B_PARALLEL_TRANSLATION : the slave moves when one of its masters moves.
	   - B_ROTATED_TRANSLATION : the slave moves when one of its masters moves or turns.
	   - B_TWISTED_TRANSLATION : the slave moves when one of its masters changes in any
	                             way. This is the default setting (the less efficient,
								 but the safest).
	   */
	B_ROTATION_MASK          = 0x0c,
	
	B_ABSOLUTE_ROTATION      = 0x00,
	B_PARALLEL_ROTATION      = 0x04,
	B_TWISTED_ROTATION       = 0x08,
	/* Those 3 flags specify the type of dependency between the orientation/zoom/twist
	   (essentially the spatial state of the object, except its position) of the slave
	   of the link and some properties of its masters. This is used to optimize the
	   performance when updating the state of all the objects of the universe, after an
	   user interaction or a time change.
	   - B_ABSOLUTE_ROTATION : the slave state (except its position) doesn't depend on
	                           its masters.
	   - B_PARALLEL_ROTATION : the slave state (except its position) changes when one of
	                           its masters moves or turns.
	   - B_TWISTED_ROTATION : the slave state (except its position) can change when any
	                          of its masters state changes in any way. This is the default
							  setting (the less efficient, but the safest).
	   */
	B_LINK_TYPE_MASK         = 0x0f,

	B_RIGID_LINK             = 0x03 | 0x08,
	/* the one default setting : the slave state depends on its masters states in any way
	   */
	
	B_TIME_TRANSLATION       = 0x10, /* those 3 should not be moved as they're used      */ 
	B_TIME_ROTATION          = 0x20, /* with the same index in Universe.h, for BUniNode. */
	B_TIME_TWIST             = 0x40, /* Check also Thing.h B_LINK_2_THING.               */
	/* Those 3 flags specify how is the state of the slave depending of the flow of time.
	   - B_TIME_TRANSLATION : the slave's position depends on the flow of time.
	   - B_TIME_ROTATION : the slave's orientation depends on the flow of time.
	   - B_TIME_TWIST : the slave's zoom or twist depends on the flow of time.
	   */
	B_TIME_ACTION            = 0x70, /* ***  WARNING  ***                                */
	
	B_USER_TRANSLATION       = 0x80, /* reserved for future user-interaction extension */
	B_USER_ROTATION          = 0x100,
	B_USER_TWIST             = 0x200,
	B_USER_ACTION            = 0x380,
	
	B_SLAVE_TRANSLATION      = 0x400,
	B_SLAVE_ROTATION         = 0x800,
	B_SLAVE_TWIST            = 0x1000,
	B_SLAVE_ACTION           = 0x1c00,

/* possible attribute for link control */
	B_FORCE_LINK             = 0x10000000,
	B_CHECK_CYCLE            = 0x20000000
};

class B3dThing;

/**************************************
// B3dLink.
/*************************************/

class B3dLink {
 public:

/*************************************
// HIGH-LEVEL API                   */

	B3dLink(uint32 my_status);
	/* When creating your own subclass of 3dLink, the only thing you need to do when
	   calling the inherited constructor is to specify the dependencies between the
	   slave and its masters (see previous enums).
	   */
	virtual ~B3dLink();
	virtual void SetSlave(B3dThing *slave);
	/* Masters and other parameters of a link are specified in the subclass constructor.
	   But the real slave of the link is not defined before the link is attached to an
	   thing using B3dThing->LinkTo(). At that time, the link will be call back using
	   this method to complete its initialisations. The link SHOULD NOT keep a copy of
	   the slave object. A valid pointer will be provide by the universe when necessary.
	   */
	virtual void DoLink(B3dThing *Object, bigtime_t time0, bigtime_t time1);
	/* This is called each time the universe thinks that the state of the slave should
	   be updated. 'Object' is a pointer to the slave, time0 is the time of the previous
	   update, time1 is the current time (both times expressed in the timebase of the
	   universe, not the real-time timebase).
	   */
	virtual int32 GetRefs(B3dThing ***refs);
	/* This functions has to return the count of masters, and the pointer to the internal
	   list of masters (*refs is a B3dThing**, a array of B3dThing*). This call is used
	   by the universe to update the master's pointers when necessary.
	   */
	
/*************************************
// PRIVATE STUFF                    */

	uint32     status;
	/* 0-1   : Translation link class
	   2-3   : Rotation/Twist link class
	   4     : Interactive user translation
	   5     : Interactive user rotation
	   6     : Interactive user twist
	   7     : Dynamic link (time control)
	   8     : Translation control
	   9     : Rotation control
	   10    : Twist control
	 */	
};

/**************************************
// B3dBasicLink.
/*************************************/

class B3dBasicLink : public B3dLink {
 public:

/*************************************
// HIGH-LEVEL API                   */

	B3dBasicLink(uint32 my_status = B_RIGID_LINK, B3dThing *master = 0L);
	/* Static link (time independant) to zero or one master. The type of the link is
	   defined by the status flag (see previous enum comments).
	   */
	
/*************************************
// PRIVATE STUFF                    */

	virtual ~B3dBasicLink();
	virtual void SetSlave(B3dThing *slave);
	virtual void DoLink(B3dThing *Object, bigtime_t time0, bigtime_t time1);
	virtual int32 GetRefs(B3dThing ***refs);

 private:
	B3dThing  *master;
	B3dVector trans;
	B3dMatrix rotate;
};

#endif















