/*****************************************************************************
//
//	File:			3dRenderer.h
//
//	Description:	Low level rasterization engine.
//
//	Copyright 1997, Be Incorporated
//
//***************************************************************************/

#ifndef _3D_RENDERER_H
#define _3D_RENDERER_H

#include <InterfaceDefs.h>

#include "3dDefs.h"
#include "RGBAColor.h"
#include "3dLighter.h"
#include "3dCamera.h"

typedef void (*RENDER_FILLED_1)
	(
		void *pixels, int32 pixelBytesPerRow,
		int32 h1, int32 h2, int32 h3,
		int32 v1, int32 v2, int32 v3,
		uint32 color,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_FILLED_3)
	(
		void *pixels, int32 pixelBytesPerRow,
		int32 h1, int32 h2, int32 h3,
		int32 v1, int32 v2, int32 v3,
		uint32 c1, uint32 c2, uint32 c3,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_TEXTURED_0)
	(
		void *pixels, int32 pixelBytesPerRow,
		int32 x1, int32 x2, int32 x3,
		int32 y1, int32 y2, int32 y3,
		float s1, float s2, float s3,
		float t1, float t2, float t3,
		float w1, float w2, float w3,
		void *textureMap, int32 textureX, int32 textureY,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_TEXTURED_1)
	(
		void *pixels, int32 pixelBytesPerRow,
		int32 x1, int32 x2, int32 x3,
		int32 y1, int32 y2, int32 y3,
		uint32 color,
		float s1, float s2, float s3,
		float t1, float t2, float t3,
		float w1, float w2, float w3,
		void *textureMap, int32 textureX, int32 textureY,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_TEXTURED_3)
	(
		void *pixels, int32 pixelBytesPerRow,
		int32 x1, int32 x2, int32 x3,
		int32 y1, int32 y2, int32 y3,
		uint32 c1, uint32 c2, uint32 c3,
		float s1, float s2, float s3,
		float t1, float t2, float t3,
		float w1, float w2, float w3,
		void *textureMap, int32 textureX, int32 textureY,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_TEXTURED_3_ADD_COLOR)
	(
		void *pixels, int32 pixelBytesPerRow,
		int32 x1, int32 x2, int32 x3,
		int32 y1, int32 y2, int32 y3,
		uint32 c1, uint32 c2, uint32 c3,
		uint32 ca1, uint32 ca2, uint32 ca3,
		float s1, float s2, float s3,
		float t1, float t2, float t3,
		float w1, float w2, float w3,
		void *textureMap, int32 textureX, int32 textureY,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_FILLEDZ_1)
	(
		void *pixels, int32 pixelBytesPerRow,
		void *zbuffer, int32 zBytesPerRow,
		int32 h1, int32 h2, int32 h3,
		int32 v1, int32 v2, int32 v3,
		float z1, float z2, float z3,
		uint32 color,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_FILLEDZ_3)
	(
		void *pixels, int32 pixelBytesPerRow,
		void *zbuffer, int32 zBytesPerRow,
		int32 h1, int32 h2, int32 h3,
		int32 v1, int32 v2, int32 v3,
		float z1, float z2, float z3,
		uint32 c1, uint32 c2, uint32 c3,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_TEXTUREDZ_0)
	(
		void *pixels, int32 pixelBytesPerRow,
		void *zbuffer, int32 zBytesPerRow,
		int32 x1, int32 x2, int32 x3,
		int32 y1, int32 y2, int32 y3,
		float z1, float z2, float z3,
		float s1, float s2, float s3,
		float t1, float t2, float t3,
		float w1, float w2, float w3,
		void *textureMap, int32 textureX, int32 textureY,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_TEXTUREDZ_1)
	(
		void *pixels, int32 pixelBytesPerRow,
		void *zbuffer, int32 zBytesPerRow,
		int32 x1, int32 x2, int32 x3,
		int32 y1, int32 y2, int32 y3,
		float z1, float z2, float z3,
		uint32 color,
		float s1, float s2, float s3,
		float t1, float t2, float t3,
		float w1, float w2, float w3,
		void *textureMap, int32 textureX, int32 textureY,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_TEXTUREDZ_3)
	(
		void *pixels, int32 pixelBytesPerRow,
		void *zbuffer, int32 zBytesPerRow,
		int32 x1, int32 x2, int32 x3,
		int32 y1, int32 y2, int32 y3,
		float z1, float z2, float z3,
		uint32 c1, uint32 c2, uint32 c3,
		float s1, float s2, float s3,
		float t1, float t2, float t3,
		float w1, float w2, float w3,
		void *textureMap, int32 textureX, int32 textureY,
		uint8 *indexMap, rgb_color *colorList
	);

typedef void (*RENDER_TEXTUREDZ_3_ADD_COLOR)
	(
		void *pixels, int32 pixelBytesPerRow,
		void *zbuffer, int32 zBytesPerRow,
		int32 x1, int32 x2, int32 x3,
		int32 y1, int32 y2, int32 y3,
		float z1, float z2, float z3,
		uint32 c1, uint32 c2, uint32 c3,
		uint32 ca1, uint32 ca2, uint32 ca3,
		float s1, float s2, float s3,
		float t1, float t2, float t3,
		float w1, float w2, float w3,
		void *textureMap, int32 textureX, int32 textureY,
		uint8 *indexMap, rgb_color *colorList
	);

typedef uint32 render_id;
/* Here is the composition of a render_id */
enum {
	NO_LOOK				= 0,
	
	/*	These describe both what "kind" of face this is _and_
		how it should be rendered.  The renderer can mask this
		with a "render using" bitmask in order to render faces
		with less detail.  The mapping of "what kind" to 
		"how to render" is not perfect, but it's pretty good. */
	FACE_LOOK_VISIBLE		= 1L << 0, // Face is visible
	FACE_LOOK_FILLED		= 1L << 1, // Face should be filled in (not wireframe)
	FACE_LOOK_COLOR		= 1L << 2, // Face material has a color associated with it
	FACE_LOOK_SPECULAR		= 1L << 3, // Face material has specular light reflection
	FACE_LOOK_LIT			= 1L << 4, // Face is affected by scene lighting
	FACE_LOOK_SMOOTH		= 1L << 5, // Face should be shaded smoothly (not flat)
	FACE_LOOK_TEXTURED		= 1L << 6, // Face has a texture associated with it
	FACE_LOOK_MIRRORED		= 1L << 7, // Face shows an environment map
	FACE_LOOK_ALPHAMAP		= 1L << 8, // Face should be alpha blended according to texture data
	FACE_LOOK_ALPHACOLOR	= 1L << 9, // Face should be alpha blended according to face color data
	FACE_LOOK_ALPHAPOINT	= 1L << 10, // Face should be alpha blended according to point alpha data
	FACE_LOOK_MASK		= 0x000007FF,

	/* These describe what elements the face look structure contains */
	FACE_HAS_SPECULAR		= 1L << 11, // The specular reflection coefficient for the face
	FACE_HAS_TEXTURE		= 1L << 12, // A pointer to texture data
	FACE_HAS_BLENDFACTOR	= 1L << 13, // Ratio of texture to color 
	FACE_HAS_ZOOMFACTOR	= 1L << 14, // "Zoom factor" for environment mapping
	FACE_HAS_COLOR		= 1L << 15, // The color of the face
	FACE_HAS_SHIFT		= 11,
	FACE_HAS_MASK			= 	FACE_HAS_COLOR |
							FACE_HAS_SPECULAR |
							FACE_HAS_TEXTURE |
							FACE_HAS_BLENDFACTOR |
							FACE_HAS_ZOOMFACTOR ,

	/* These describe what elements the point look structure contains */
	POINT_HAS_TCOORDS		= 1L << 16, // Point has texture coords associated with it
	POINT_HAS_ALPHA		= 1L << 17, // Point has an alpha value associated with it
	POINT_HAS_SHIFT		= 16,
	POINT_HAS_MASK		= 	POINT_HAS_TCOORDS |
							POINT_HAS_ALPHA
};

/*

	Note that FACE_LOOK_COLOR with FACE_LOOK_TEXTURED or FACE_LOOK_MIRRORED
	implies that the texture or mirror should be blended with the material
	color, and this requires a "texture saturation" or "reflectivity" constant,
	respectively, both of which specify a ratio of color to texture.  The blending
	is done at low-level rendering time.

*/

/* Hook to call high level rendering package */
typedef void (*render_hook)(PNLD *p1, PNLD *p2, PNLD *p3, PNLD *face,
							struct render_desc *desc);

/* Hook used for interpolation of pnld parameters during cliping */
typedef void (*inter_hook)(PNLD *p1, float ref1, PNLD *p2, float ref2, PNLD *new_p);

/* Struct describing the rendering context and the rendering package pipeline */ 
struct render_desc {
	B3dRenderer		*theRenderer;

	uint16			needFlags;	

	void				*bits;
	uint32			bytes_per_row;

	void				*zbuffer;
	uint32			zbytes_per_row;

	uint8			*index_map;
	rgb_color			*color_list;

	float			overScreenX;
	float			overScreenY;
	int32			screenX;
	int32			screenY;

	/* hook functions */
	inter_hook		inter;
	render_hook		draw;
	union {
		void *				lowRenderFunc;

		RENDER_FILLED_1				fillTriangle1;
		RENDER_FILLED_3				fillTriangle3;
		RENDER_TEXTURED_0				mapTriangle0;
		RENDER_TEXTURED_1				mapTriangle1;
		RENDER_TEXTURED_3				mapTriangle3;
		RENDER_TEXTURED_3_ADD_COLOR		mapTriangle3_AddColor;

		RENDER_FILLEDZ_1				fillTriangleZ1;
		RENDER_FILLEDZ_3				fillTriangleZ3;
		RENDER_TEXTUREDZ_0				mapTriangleZ0;
		RENDER_TEXTUREDZ_1				mapTriangleZ1;
		RENDER_TEXTUREDZ_3				mapTriangleZ3;
		RENDER_TEXTUREDZ_3_ADD_COLOR	mapTriangleZ3_AddColor;
	};
};

/* settings for renderer code */
enum {
	ROTR  = 8,
	ROTG  = 16,
	ROTB  = 24,
	EXPO  = 20,
	
	PROJECT_STEP_FACT = 8,
	PROJECT_STEP_MASK = 7,
	PROJECT_STEP_EXP = 3,
	EXPOFAC	= 10,
	RANDOM_TABLE_MASK = 0x000003FF
};

#define FLOAT_EXPO 1048576.0
#define FACTOR     1024.0

extern 	ulong    invert[1024];
extern	int32	randomValues[1024];

class B3dRenderer {

 public:	
	/*	Flags which allow the renderer to tell a Look which 
		information it needs.  This information is given in
		the render_desc which is passed back from a 
		FillRenderDesc call. */
	enum {
		NEED_POINT_NORMALS		= 0x0001,
		NEED_POINT_LIGHTING	= 0x0002,
		NEED_FACE_NORMAL		= 0x0004,
		NEED_FACE_LIGHTING		= 0x0008
	};

	B3dRenderer();
	virtual				~B3dRenderer();
	
	long					SetBuffer(struct camera_buffer *buffer);
	uint32				RenderFlags();
	void					InitRenderDesc(render_desc *desc, B3dLensImage *projDesc);
	void					FillRenderDesc(render_id rid, render_desc *desc);

private:

	void					PickRenderers();

	camera_buffer			m_buffer;
	render_hook			m_renderDispatch[4][4];
	inter_hook			m_interpDispatch[4][4];
	void *				m_lowLevelDispatch[4][4];
	uint32				m_needs[4][4];
	uint32				m_renderOptions;
	uint32				m_renderMode;
	
	union {
	RENDER_FILLED_1		m_fillTriangle1;
	RENDER_FILLEDZ_1	m_fillTriangleZ1;
	};
	union {
	RENDER_FILLED_3		m_fillTriangle3;
	RENDER_FILLEDZ_3	m_fillTriangleZ3;
	};
	union {
	RENDER_TEXTURED_0	m_mapTriangle0;
	RENDER_TEXTUREDZ_0	m_mapTriangleZ0;
	};
	union {
	RENDER_TEXTURED_1	m_mapTriangle1;
	RENDER_TEXTUREDZ_1	m_mapTriangleZ1;
	};
	union {
	RENDER_TEXTURED_3	m_mapTriangle3;
	RENDER_TEXTUREDZ_3	m_mapTriangleZ3;
	};
	union {
	RENDER_TEXTURED_3_ADD_COLOR		m_mapTriangle3_AddColor;
	RENDER_TEXTUREDZ_3_ADD_COLOR	m_mapTriangleZ3_AddColor;
	};
};

#endif
