/***************************************************
**
**	File: wView.cpp
**	Description: 3d view constructor.
**	Copyright 1997, Be Incorporated
**
***************************************************/

#include <3dImport.h>
#include "3dRadialLens.h"
#include "wView.h"
#include "wGlobe.h"
#include "3dCinepakChannel.h"

extern B3dFileChannelServer *channelServer;

wView::wView(BRect r, char *n) : B3dView(n, r)
{
}

/*

	Demos:

	1		Getting something on the screen
	2		Using level-of-detail management 
			(try setting USE_LOD to 0 to see the difference)
	3		Importing VRML 1.0.  You have to start the
			app and then drag a VRML object onto it.
	4		Advanced rendering: using specular reflection,
			antialiased texture-mapping
	5		Integrating media sources
			Drag and drop images and movies onto both objects

*/

#define USE_LOD 1
#define DEMO 1

#if (DEMO == 1)

void wView::AttachedToWindow()
{
	B3dVector  v;
	RGBAColor  color;
	RGBAColor  shineColor;
	B3dFaceBody *globe;
	B3dMaterial material;
	
	// First call the inherited method
	inherited::AttachedToWindow();

	// Lock the universe before modifying it
	Universe()->Lock();

	// Create a sphere
	v.Set(1,1,1);
	myBody = new B3dSphere("Sphere",World(),&v,300);
	myBody->SetOrigin(-0.62,0,3.0);

	// Set it to be shiny red
	color.Set(1,0,0,1);
	shineColor.Set(1,1,1,1);
	material.SetColor(&color);
	material.SetShine(&shineColor,10);
	myBody->WriteFaceMaterials(B_OBJECT_SELECTION_ID,
		&material,B_ONE_TO_ALL_FACES);

	// Create a cube next to it
	v.Set(0.85,0.85,0.85);
	myBody = new B3dCube("Cube",World(),&v);
	myBody->SetOrigin(0.62,0,3.0);

	// Set it to be a dull blue
	color.Set(0.2,0.2,1,1);
	material.SetColor(&color);
	material.SetProperties(material.Properties() & 
		~(B_MATERIAL_IS_SMOOTH|B_MATERIAL_IS_SHINY));
	myBody->WriteFaceMaterials(B_OBJECT_SELECTION_ID,
		&material,B_ONE_TO_ALL_FACES);

	// Point the camera to look in between them
	v.Set(0,0,3.0);
	Camera()->ViewPoint()->LookAt(&v);
	
	// Create some light
	color.Set(0.9, 0.9, 1.0, 0.0);
	shineColor.Set(1.0, 1.0, 1.0, 0.0);
	v.Set(-2.0, 4.0, 3.0);
	new B3dParallelLight("Test Light1", World(),
	                     0.9, &color, &shineColor, &v);
 	new B3dAmbientLight("Test Light2", World(),
	                    0.4, &color);

	// end of universe modification
	Universe()->Unlock();	
}

wView::~wView()
{
}

#elif (DEMO == 2)

//##################################################
// Using level of detail.
//##################################################

void wView::AttachedToWindow() {
	int        i;
	B3dThing   *list[3];
	B3dVector  v;
	RGBAColor  color;

	// Call the inherited method
	inherited::AttachedToWindow();

	// Lock the universe before modifying it
	Universe()->Lock();

	// Create a cube
	v.Set(0.7, 0.7, 0.7);
 	myBody = new B3dCube("cube", World(), &v);
	myBody->SetOrigin(5.0, 1.2, -1.3);

	// Create a simple sphere
	v.Set(2.0, 2.0, 2.0);
 	sndBody = new B3dSphere("sphere500", World(), &v, 500);
	
#if USE_LOD
	// Add two lesser levels of detail
	trdBody = new B3dSphere("sphere100", World(), &v, 100);
	trdBody->TieThing(sndBody, 60);
	sndBody = new B3dSphere("sphere20", World(), &v, 20);
	sndBody->TieThing(trdBody, 15);
#endif
	
	sndBody->SetOrigin(8.0, 0.0, -3.0);
	// Clone another sphere
	trdBody = (B3dFaceBody*)sndBody->Clone();
	trdBody->SetOrigin(14.0, 0.0, -3.0);

	// clone the 248 other spheres
	v.Set(0.0, 0.0, 1.0);
	list[0] = sndBody;
	list[1] = trdBody;
 	for (i=2; i<250; i++) {
		list[2] = (B3dFaceBody*)sndBody->Clone();
		list[2]->SetOrigin(8.0+6.0*i, 0.0, -3.0);
		list[2]->LinkTo(new wLink(list[1],
								  list[0],
								  myBody, &v));
		list[0] = list[1];
		list[1] = list[2];
	}

	// Create light
	color.Set(1.0, 1.0, 1.0, 0.0);
	v.Set(2.0, 4.0, 3.0);
	new B3dParallelLight("Test Light1", World(),
	                     0.8, &color, &color, &v);
 	new B3dAmbientLight("Test Light2", World(),
	                    0.3, &color);

	// Point the camera at the spheres
	v.Set(14.0, 0.0, -3.0);
	Camera()->ViewPoint()->LookAt(trdBody);

	// End of universe modification
	Universe()->Unlock();	
};


wView::~wView()
{
}

#elif (DEMO == 3)

void wView::MessageReceived(BMessage *msg)
{
	B3dImporter *importer;
	B3dFaceBody *vrmlBody;
	B3dVector worldCoords,screenCoords;
	entry_ref tref;
	BPath path;

	// If a file was dropped on us...
	if (msg->WasDropped() && msg->HasRef("refs")) {

		// Get the entry_ref for the dropped file
		msg->FindRef("refs", &tref);

		// Find the path of the file
		BEntry entry(&tref,true);
		entry.GetPath(&path);
		
		// Create a VRML importer object to import it's contents
		B3dVRMLImporter *importer = new B3dVRMLImporter(&path,channelServer);
		importer->SetTargetScale(1.0);

		// Now lock the universe, so we can import the object
		Universe()->Lock();

		// Do the actual object import
		vrmlBody = new B3dFaceBody("VRML object",World(),importer);

		// Delete the importer; we're done with it
		delete importer;

		// Place the imported object
		screenCoords.Set(0,0,vrmlBody->radius*3.0);
		worldCoords = Camera()->Camera2World(screenCoords);
		vrmlBody->SetOrigin(worldCoords.x,worldCoords.y,worldCoords.z);

		// Done importing
		Universe()->Unlock();
	} else {
		inherited::MessageReceived(msg);
	};
};

void wView::AttachedToWindow()
{
	RGBAColor color,specColor;
	B3dVector v;

	// Call the inherited method
	inherited::AttachedToWindow();

	// Lock the universe before modifying it
	Universe()->Lock();

	// Let there be light
	color.Set(0.9, 0.9, 1.0, 0.0);
	specColor.Set(1.0, 1.0, 1.0, 0.0);
	v.Set(-2.0, 4.0, 3.0);
	new B3dParallelLight("Demo Light 1", World(), 0.9, &color, 
		&specColor, &v);
 	new B3dAmbientLight("Demo Light 2", World(), 0.4, &color);

	/*	Uncomment this if the object is very complex,
		but it's quite a bit slower */
//	SetRenderFlags(RenderFlags()|B_3D_ZBUFFER);

	// Done modifications
	Universe()->Unlock();
};

wView::~wView()
{
}

//##################################################
// Advanced rendering
//##################################################

#elif (DEMO == 4)

void wView::AttachedToWindow() {
	B3dVector  v;
	RGBAColor  color,shineColor;
	B3dMaterial material;

	inherited::AttachedToWindow();

	// Lock before modifying
	Universe()->Lock();

	// Load a map of earth
	B3dChannel *earthPict = new B3dBitmap("earth",
		LoadTiff("/boot/BeDC3d/3dw/earthmap_bright.tif"));

	// Create a globe object using that texture
	myBody = new wGlobe("Earth",World(),20,40,earthPict);

	// We want the atmosphere to be slightly reflective
	myBody->ReadFaceMaterials(B_OBJECT_SELECTION_ID,
		&material,0);
	shineColor.Set(0.5,0.5,0.5,1);
	material.SetShine(&shineColor,10.0);
	myBody->WriteFaceMaterials(B_OBJECT_SELECTION_ID,&material,
		B_ONE_TO_ALL_FACES);

	// Tilt it on it's axis
	B3dMatrix mat;
	v.Set(0,0,1.0);
	mat.Set(0,0,(-22.0/360.0)*2*PI);
	v = mat*v;
	mat = mat * *myBody->Rotation();
	myBody->SetRotation(&mat);

	// Give it a rotation
	myBody->LinkTo(new B3dAxisLink(0.05, &v));

	// Place it in the world and point the camera at it
	myBody->SetOrigin(10.0, 0.0, 0.0);
	Camera()->ViewPoint()->LookAt(myBody);

	// Create a large fusion reaction
	color.Set(1.0, 1.0, 1.0, 0.0);
	shineColor.Set(1.0, 1.0, 1.0, 0.0);
	v.Set(-2.0, 4.0, 3.0);
	Light = new B3dParallelLight("Test Light1", World(),
	                     0.9, &color, &shineColor, &v);

	// We want the earth to look good close up
	// Try commenting this to see the difference
	SetRenderFlags(RenderFlags()|B_3D_ANTIALIAS_TEXTURES);

	// Done messing with the universe, or at least
	// the solar system
	Universe()->Unlock();	
}

wView::~wView()
{
}

#elif (DEMO == 5)

void wView::MessageReceived(BMessage *msg)
{
	BNode *tnode;
	BNodeInfo *ninfo;
	entry_ref tref;
	char mtype[256];
	B3dChannel *newpict;
	BPath path;
	int32 whichChannel=-1;
	B3dVector screenCoords;
	BPoint screen,point;
	B3dTouchDesc touch;

	// If a file was dropped on us...
	if (msg->WasDropped() && msg->HasRef("refs")) {
		screen = msg->DropPoint();
		point = (BPoint)ConvertFromScreen(screen);
		screenCoords = Camera()->Pixel2Screen(point,0);

		// Figure out which object it was dropped on
		Universe()->Lock();
		if (Camera()->GetTouchDesc(&screenCoords, &touch)) {
			if (touch.object == teapot)
				whichChannel = 0;
			else
				whichChannel = 1;
			touch.object->FreeTouchInfo(touch.personalInfo);
		};

		// The drop didn't land on any object, so throw it away
		if (whichChannel == -1) {
			Universe()->Unlock();
			return;
		};

		// Get the entry_ref for the dropped file
		msg->FindRef("refs", &tref);

		// Find the MIME type of the dropped file
		tnode = new BNode(&tref);
		ninfo = new BNodeInfo(tnode);
		ninfo->GetType(mtype);
		delete ninfo;
		
		// Depending on the MIME type of the dropped file,
		// load it and place it in a channel
		if (strcmp(mtype, "video/quicktime") == 0) {
			Window()->Unlock();
			Universe()->Unlock();

			newpict = new B3dCinepakChannel("", Universe(), &tref);

			Window()->Lock();
			Universe()->Lock();

			if (newpict == NULL)
				return;
		} else if (strcmp(mtype, "image/tiff") == 0) {
			Window()->Unlock();
			Universe()->Unlock();

			// Find the path of the file
			BEntry entry(&tref,true);
			entry.GetPath(&path);
			
			BBitmap *bm = LoadTiff((char*)path.Path());
			if (!bm)
				return;
				
			newpict = new B3dBitmap("mirror",bm);

			Window()->Lock();
			Universe()->Lock();

			if (newpict == NULL)
				return;
		} else {
			Window()->Unlock();
			Universe()->Unlock();

			// Find the path of the file
			BEntry entry(&tref,true);
			entry.GetPath(&path);
			
			newpict = LoadTexture(path.Path());

			Window()->Lock();
			Universe()->Lock();

			if (newpict == NULL)
				return;
		}
		
		if (whichChannel == 0) {
			// Set the background of the offscreen view 
			// in the teapot
			offView->SetBackground(newpict);
			if (channel[0] && (channel[0] != channel[1]))
				delete channel[0];
			channel[0] = newpict;
		} else {
			// Set the channel which the cow is reflecting
			B3dMaterial material;
			material.SetMirror(newpict,0.6);
			material.SetProperties(material.Properties() & 
				~B_MATERIAL_IS_LIT);
			vrmlBody->WriteFaceMaterials(B_OBJECT_SELECTION_ID,
				&material, B_ONE_TO_ALL_FACES);
			if (channel[0] != channel[1])
				delete channel[1];
			channel[1] = newpict;
		};
		
		// Optimize the objects for fast rendering
		vrmlBody->Optimize(B_OPTIMIZE_LIGHT);
		teapot->Optimize(B_OPTIMIZE_LIGHT);

		// Finished changes
		Universe()->Unlock();
	};
};

//##################################################
// Media integration
//##################################################
void wView::AttachedToWindow() {
	B3dVector        v;
	B3dAxisLink      *link;
	B3dRadialOptions *opt_ptr;
	RGBAColor 		color,specColor;
	B3dMaterial		material;
	B3dMatrix			matrix;
	
	// Call the inherited method.
	inherited::AttachedToWindow();

	// Lock the universe first
	Universe()->Lock();

	// Load in a picture as a graphic channel
	channel[1] = NULL;
	channel[0] = pict = new B3dBitmap("mirror",
		LoadTiff("/boot/BeDC3d/images/soleil.tiff"));

	// Create an importer
	B3dVRMLImporter *importer = 
		new B3dVRMLImporter("/boot/BeDC3d/objects/cow.wrl");
	importer->SetTargetScale(2.0);
	importer->SetShapeHints(true,true,true);

	// Do the actual object import
	vrmlBody = new B3dFaceBody("VRML object",World(),importer);

	// Delete the importer; we're done with it
	delete importer;

	// Place it somewhere an set it spinning
	vrmlBody->SetOrigin(9.0*0.5,5.0*0.5,-5.5*0.5);
	matrix.Set(0,0,PI/2.0);
	vrmlBody->SetRotation(&matrix);
	v.Set(0.0, 0.0, 1.0);
	vrmlBody->LinkTo(new B3dAxisLink(0.4, &v));

	// Create an offscreen view to render into
	offView = new B3dOffView("fake", 
		BRect(0,0,(1<<pict->size_h)-1,(1<<pict->size_v)-1),
		World(),pict);
	
	// Instatiate an importer
	B3dTeapotImporter *teapotImporter = 
		new B3dTeapotImporter("/boot/beos/etc/teapot.data");

	// Do the actual object import
	teapot = new B3dFaceBody("VRML object",World(),teapotImporter);

	// Delete the importer object; we're done with it
	delete teapotImporter;

	// Make this object mirror the rendered view	
	material.SetMirror(offView,0.6);
	material.SetProperties(material.Properties() & 
		~B_MATERIAL_IS_LIT);
	teapot->WriteFaceMaterials(B_OBJECT_SELECTION_ID,
		&material,B_ONE_TO_ALL_FACES);

	// Set it somewhere not far from the first object,
	// and spin it in the opposite direction
	teapot->SetOrigin(12.4*0.75, 15.7*0.75, -6.75*0.75);
	teapot->LinkTo(new B3dAxisLink(-0.71, &v));

	// Place the offcreen view's camera in between the
	// two objects
	v.Set(12.2*0.75, 15.4*0.75, -6.4*0.75);
	v.x = (teapot->origin.x + vrmlBody->origin.x) / 2;
	v.y = (teapot->origin.y + vrmlBody->origin.y) / 2;
	v.z = (teapot->origin.z + vrmlBody->origin.z) / 2;
	offView->Camera()->SetViewPoint(v,World());

	// Point it at the first object
	v.Set(7.0, 5.1, -3.8);
	offView->Camera()->ViewPoint()->LookAt(vrmlBody);

	// "Focus" the camera
	offView->Camera()->Lens()->GetOptions(&opt_ptr);
	opt_ptr->zoom = 340.0;
	offView->Camera()->Lens()->SetOptions();

	// Make the mirroring look nice
	// Uncomment this if you're anal :)
//	SetRenderFlags(RenderFlags()|B_3D_ANTIALIAS_TEXTURES);

	// Point the view's camera in between the two objects
	v.Set(8.0, 7.1, -4.8);
	Camera()->ViewPoint()->LookAt(&v);

	// Create light
	color.Set(0.9, 0.9, 1.0, 0.0);
	specColor.Set(1.0, 1.0, 1.0, 0.0);
	v.Set(-2.0, 4.0, 3.0);
	new B3dParallelLight("Demo Light 1", World(), 0.9, &color, 
		&specColor, &v);
 	new B3dAmbientLight("Demo Light 2", World(), 0.4, &color);
	
	// Optimize the objects for fast rendering
	vrmlBody->Optimize(B_OPTIMIZE_LIGHT);
	teapot->Optimize(B_OPTIMIZE_LIGHT);
	
	// End of universe modification
	Universe()->Unlock();	
}

wView::~wView()
{
	delete offView;
	delete channel[0];
	if (channel[0] != channel[1])
		delete channel[1];
}

#endif

#if (DEMO != 3) && (DEMO != 5)

void wView::MessageReceived(BMessage *msg)
{
	inherited::MessageReceived(msg);
};

#endif
