//*****************************************************************************
//
//	File:		 3dmovApp.cpp
//
//	Description: Demo application using the 3d Kit.
//
//	Copyright 1996, Be Incorporated
//
//*****************************************************************************

#include <Debug.h>

#ifndef _3D_MOV_APP_H
#include "3dmovApp.h"
#endif
#ifndef _3D_CINEPAK_CHANNEL_H
#include "3dCinepakChannel.h"
#endif
#ifndef _3D_MATERIAL_H
#include "3dMaterial.h"
#endif
#ifndef _ALERT_H
#include "Alert.h"
#endif

#include <StorageKit.h>
#include <Path.h>
#include <NodeInfo.h>
#include <Debug.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

// prototype for the TIFF loader.
extern BBitmap *LoadTiff(char *filename);

//*****************************************************************************

static B3dBitmap *tiff_loader(entry_ref *ref, bool *lock);
long bitmap_thread(void *data);
B3dBitmap *default_texture, *default_texture2;

//*****************************************************************************

// standard main() function.
main() {	
	Z3dApplication  *app;
	
	app = new Z3dApplication();
	app->ActiveWindow(0, 1);
	app->Run();
	delete app;
	return(0);
}

Z3dApplication::Z3dApplication() : BApplication('3DMO') {
	long        i;
	BBitmap     *bitmap;

	// init the shared universe
	shared_uni = new B3dUniverse("Shared universe");
	shared_uni->SetFrameRate(25.0);
	
	// Default state for animation and all rotation (universe flow of time)
	Pause = FALSE;

	// create a semaphore to allow modification of active window
	window_sem = create_sem(1,"3dmov window sem");
	for (i=0;i<4;i++)
		myWindow[i] = 0L;

	// initialise the default texture (White and Black)
	bitmap = new BBitmap(BRect(0.0, 0.0, 0.0, 0.0), B_RGB_32_BIT);
	default_texture = new B3dBitmap("default texture", bitmap);
	bitmap = new BBitmap(BRect(0.0, 0.0, 0.0, 0.0), B_RGB_32_BIT);
	*((int32*)bitmap->Bits()) = 0L;
	default_texture2 = new B3dBitmap("default texture2", bitmap);
	
	// init Video Channel dynamic management.	
	for (i=0; i<13; i++) {
		channel_locks[i] = FALSE;	// locks used for init synchro.
		channel_dev[i] = 0;			// ids used for unique identification
		channel_ino[i] = 0;			// of a drag&drop file.
		channels[i] = 0L;			// pointer to the B3dChannel.
	}
}

Z3dApplication::~Z3dApplication() {
	long     i;

	// close all demo windows
	for (i=0;i<4;i++)
		ActiveWindow(i, -1);

	// free the default texture
	delete default_texture;
	delete default_texture2;
	
	// delete semaphores
	delete_sem(window_sem);

	// release the shared universe
	delete shared_uni;
}

void Z3dApplication::AboutRequested() {
	BAlert				*my_alert;

	// poor about alert
	my_alert = new BAlert("",
						  "3D Kit demo\n"
						  "by Pierre Raynaud-Richard,\n"
						  "(C) 1997 - Be, Inc.\n",
						  "Big deal");
	my_alert->Go();
}

void Z3dApplication::MessageReceived(BMessage* msg)
{
	switch (msg->what) {
	case FIRST_WINDOW :
	case SECOND_WINDOW :
	case THIRD_WINDOW :
	case FOURTH_WINDOW :
		// active demo window if not activated 
		if (myWindow[msg->what-FIRST_WINDOW] == 0L)
			ActiveWindow(msg->what-FIRST_WINDOW, 1);
		break;
	case PAUSE_WINDOWS :
		// switch the pause state by setting the flow of time type of
		// the shared universe.
		Pause = ~Pause;
		shared_uni->Lock();
		shared_uni->SetTimeMode((Pause)?B_FROZEN_TIME:B_REAL_TIME, 0.0);
		shared_uni->Unlock();
		break;
	}
}

void Z3dApplication::ActiveWindow(long index, long state) {
	int    i;
	
	if ((index<0) || (index>3)) return;
	
	acquire_sem(window_sem);
	// never mess with the state of components of the universe without
	// locking it first.
	shared_uni->Lock();
	// create a new demo window
	if (state == 1) {
		if (myWindow[index] == 0L) {
			// create the new window, and show it
			myWindow[index] = new Z3dWindow(index, shared_uni);
			myWindow[index]->Show();
		}
	}
	// free the window from the window active list (the window quit itself).
	else if (state == 0) {
		myWindow[index] = 0L;
		// clear all the channels used by the specific demos.
	simple_quit:
		if (index == 0)
			for (i=0; i<6; i++)
				ClearChannel(i);		
		else if (index == 1)
			ClearChannel(6);		
		else if (index == 2)
			for (i=7; i<9; i++)
				ClearChannel(i);		
		else
			for (i=9; i<13; i++)
				ClearChannel(i);		
	}	
	// free the window from the window active list and quit the window.
	else if (state == -1) {
		if (myWindow[index] != 0L) {
			myWindow[index]->Quit();
			goto simple_quit;
		}
	}
	shared_uni->Unlock();
	// Close the application when the last window is closed
	if ((myWindow[0] == 0L) &&
		(myWindow[1] == 0L) &&
		(myWindow[2] == 0L) &&
		(myWindow[3] == 0L))
		PostMessage(B_QUIT_REQUESTED);
	release_sem(window_sem);
}

void Z3dApplication::SelectChannel(entry_ref *tref, B3dFaceBody *obj,
								   selection_id touch, int index) {
	int         i;
	char		mtype[256];
	BNode	    *tnode;
	BEntry	    *eInfo;
	BNodeInfo   *ninfo;
	struct stat st;
	B3dMaterial material;

	// Are we trying to bother a busy channel ?	
	if (channel_locks[index])
		return;
	// Are we trying to select the same channel again ?
	eInfo = new BEntry(tref);
	eInfo->GetStat(&st);
	delete eInfo;
	if ((channel_dev[index] == st.st_dev) && (channel_ino[index] == st.st_ino))
		return;
	// Is this channel already used somewhere ? Just get a copy...
	for (i=0; i<13; i++)
		if ((channel_dev[i] == st.st_dev) && (channel_ino[i] == st.st_ino)) {
			ClearChannel(index);
			channels[index] = channels[i];
			goto set_new_channel;
		}
	// create a new channel if possible
	tnode = new BNode(tref);
	ninfo = new BNodeInfo(tnode);
	ninfo->GetType(mtype);
	delete ninfo;
	
	if (strcmp(mtype, "video/quicktime") == 0) {
		// new quicktime channel
		ClearChannel(index);
		channels[index] = new B3dCinepakChannel("", obj->Universe(), tref);
		goto set_new_channel;
	}
	else if (strcmp(mtype, "image/tiff") == 0) {
		// new tiff channel
		ClearChannel(index);
		channels[index] = tiff_loader(tref, channel_locks+index);
		goto set_new_channel;
	}
	// unknow file mimetype. Do nothing.
	delete tref;
	return;
	
 set_new_channel:
 	// keep a copy of the unique identifier of the file used by
 	// the new channel.
	channel_dev[index] = st.st_dev;
	channel_ino[index] = st.st_ino;
	// Set the new channel as the texture of the appropriate selection.
	material.SetTexture(channels[index]);
	// selection 0 to 5 are used on the cube, and so are not smooth
	if (index < 6)
		material.SetProperties(material.Properties() & ~B_MATERIAL_IS_SMOOTH);
	obj->WriteFaceMaterials(touch, &material, B_ONE_TO_ALL_FACES);
}

void Z3dApplication::ClearChannel(int index) {
	int    i;

	// Do we need to kill the old channel ? Is it still used somewhere else
	for (i=0; i<13; i++)
		if (i != index)
			if (channels[i] == channels[index])
				goto still_used;
	// kill the previous channel
	delete channels[index];			
 still_used:
 	// clean the channel descriptor
	channels[index] = 0L;
	channel_dev[index] = 0;
	channel_ino[index] = 0;
}

// create a new B3dBitmap channel, and spawn a new thread to load the picture
// itself and clip it at the channel size.
typedef struct {
	bool        *lock;
	void        *buffer;
	entry_ref   *ref;
} my_struct;

B3dBitmap *tiff_loader(entry_ref *ref, bool *lock) {
	BBitmap     *bitmap;
	my_struct   *my_s;
	B3dBitmap   *channel;
	thread_id   poller;

	my_s = (my_struct*)malloc(sizeof(my_struct));
	bitmap = new BBitmap(BRect(0.0, 0.0, 255.0, 255.0), B_RGB_32_BIT);
	my_s->buffer = bitmap->Bits();
	channel = new B3dBitmap("", bitmap);	
	memset(my_s->buffer, 0x20, 4*256*256);

	my_s->ref = ref;
	*lock = TRUE;
	my_s->lock = lock;
	poller = spawn_thread((thread_entry)bitmap_thread, "tiff loader",
						  B_NORMAL_PRIORITY+3, (void*)my_s);
	resume_thread(poller);
	
	return channel;
}

long bitmap_thread(void *data) {
	int			i, j, imin, imax, hs, vs;
	long		size;
	char		path[1024];
	BPath		pInfo;
	ulong		*src, *dst;
	BEntry		*eInfo;
	BBitmap		*bitmap;
	my_struct	*s;

	s = (my_struct*)data;

	eInfo = new BEntry(s->ref);
	eInfo->GetPath(&pInfo);
	strcpy(path, pInfo.Path());
	delete eInfo;

	if ((bitmap = LoadTiff(path)) != 0L) {
		hs = bitmap->Bounds().Width()+1;
		vs = bitmap->Bounds().Height()+1;

		src = (ulong*)bitmap->Bits();
		dst = (ulong*)s->buffer;
		if (vs >= 256) {
			imin = 0;
			imax = 256;
			src += hs*((vs-256)/2);
		}
		else {
			imin = (256-vs)/2;
			imax = imin+vs;
			dst += 256*imin;
		}
		if (hs >= 256) {
			size = 256;
			src += (hs-256)/2;
		}
		else {
			size = hs;
			dst += (256-hs)/2;
		}
		for (i=imin; i<imax; i++) {
			for (j=0; j<size; j++)
				*dst++ = *src++;
			dst += 256-size;
			src += hs-size;
		}
		delete bitmap;
	}
	*(s->lock) = FALSE;
	free(s);
	return 0L;
}













