[comment {-*- tcl -*- doctools manpage}]
[manpage_begin websocket n 1.3]
[see_also http]
[keywords http]
[keywords internet]
[keywords net]
[keywords {rfc 6455}]
[moddesc {websocket client and server}]
[titledesc {Tcl implementation of the websocket protocol}]
[category Networking]
[require Tcl 8.4]
[require http 2.7]
[require logger]
[require sha1]
[require base64]
[require websocket 1.3]
[description]

[para]

NOTE: THIS DOCUMENTATION IS WORK IN PROGRESS...

[para]

The websocket library is a pure Tcl implementation of the WebSocket
specification covering the needs of both clients and servers.
Websockets provide a way to upgrade a regular HTTP connection into a
long-lived and continuous binary or text communication between the
client and the server.  The library offers a high-level interface to
receive and send data as specified in RFC 6455 (v. 13 of the
protocol), relieving callers from all necessary protocol framing and
reassembly.  It implements the ping facility specified by the
standard, together with levers to control it.  Pings are server-driven
and ensure the liveness of the connection across home (NAT) networks.
The library has a number of introspection facilities to inquire about
the current state of the connection, but also to receive notifications
of incoming pings, if necessary.  Finally, the library contains a
number of helper procedures to facilitate the upgrading handshaking in
existing web servers.

[para]

Central to the library is the procedure [cmd websocket::takeover] that
will take over a regular socket and treat it as a WebSocket, thus
performing all necessary protocol framing, packetisation and
reassembly in servers and clients.  The procedure also takes a
handler, a command that will be called back each time a (possibly
reassembled) packet from the remote end is ready for delivery at the
original caller.  While exported by the package, the command 
[cmd websocket::takeover] is seldom called in applications, since the
package provides other commands that are specifically tuned for the
needs of clients and servers.

[para]

Typically, clients will open a connection to a remote server by
providing a WebSocket URL ([term ws:] or [term wss:] schemes) and the
handler described above to the command [cmd websocket::open]. The
opening procedure is a wrapper around the latest http::geturl
implementations: it arranges to keep the socket created within the
http library opened for reuse, but confiscates it from its (internal)
map of known sockets for its own use.

[para]

Servers will start by registering themselves through the command 
[cmd ::websocket::server] and a number of handlers for paths using the
command [cmd ::websocket::live].  Then for each incoming client
connection, they should test the incoming request to detect if it is
an upgrade request using [cmd ::websocket::test] and perform the final
handshake to place the socket connection under the control of the
websocket library and its central procedure using [cmd ::websocket::upgrade].

[para]

Apart from these main commands, the package provides a number of
commands for introspection and basic operations on the websockets that
it has under its control.  As WebSockets connections are long-lived,
most remaining communication with the library will be by way of
callbacks, i.e. commands that are triggered whenever important events
within the library have occur, but mostly whenever data has been
received on a WebSocket.

[section Callbacks]

A number of commands of the library take a handler handler command as
an argument, a command which will be called back upon reception of
data, but also upon important events within the library or events
resulting from control messages sent by the remote end.  For each
callback being performed, the following arguments will be appended:

[list_begin definitions]

[def [arg "sock"]]

The identifier of the WebSocket, as returned for example by 
[cmd ::websocket::open]

[def [arg "type"]]

A textual type describing the event or message content, can be one of
the following
[list_begin definitions]
[def [const "text"]] Complete text message
[def [const "binary"]] Complete binary message
[def [const "ping"]] Incoming ping message
[def [const "connect"]] Notification of successful connection to server
[def [const "disconnect"]] Disconnection from remote end
[def [const "close"]] Pending closure of connection
[list_end]

[def [arg "msg"]]

Will contain the data of the message, whenever this is relevant,
i.e. when the [arg "type"] is [const "text"], [const "binary"] or
[const "ping"] and whenever there is data available.

[list_end]

[section API]

[list_begin definitions]

[call [cmd ::websocket::open] [arg url] [arg handler] [opt [arg options]]]

This command is used in clients to open a WebSocket to a remote
websocket-enabled HTTP server.  The URL provided as an argument in
[arg url] should start with ws: or wss:, which are the WebSockets
counterpart of http: and https:. The [arg handler] is a command that
will be called back on data reception or whenever important events
occur during the life of the websocket.

[cmd ::websocket::open] will return a socket which serves as both the
identifier of the websocket and of the physical low-level socket to
the server.  This socket can be used in a number of other commands for
introspection or for controlling the behaviour of the library.

Being essentially a wrapper around the [cmd ::http::geturl] command,
this command provides mostly the same set of dash-led options than
[cmd ::http::geturl].  Documented below are the options that differ
from [cmd ::http::geturl] and which are specific to the WebSocket
library.

[list_begin definitions]

[def "-headers"]

This option is supported, knowing that a number of headers will be
automatically added internally in the library in order to be able to
handshake the upgrading of the socket from a regular HTTP socket to a
WebSocket with the server.

[def "-validate"]

This option is not supported as it has no real point for WebSockets.

[def "-handler"]

This option is used internally by the websocket library and cannot be
used.

[def "-command"]

This option is used internally by the websocket library and cannot be
used.

[def "-protocol"]

This option specifies a list of application protocols to handshake
with the server.  This protocols might help the server triggering
application specific features.

[def "-timeout"]

This option is supported, but will implemented as part of the library
to enable a number of finalising cleanups.

[list_end]

[call [cmd ::websocket::send] [arg sock] [arg type] [opt [arg msg]] [opt [arg final]]]

This command will send a fragment or a control message to the remote
end of the WebSocket identified by [arg sock].  The type of the
message specified in [arg type] can either be an integer according to
the specification or (preferrably) one of the following case
insensitive strings: "text", "binary" or "ping".  The content of the
message to send to the remote end is contained in [arg msg] and
message fragmentation is made possible by the setting the argument
[arg final] to non-true, knowing that the type of each fragment has
then to be the same.

The command returns the number of bytes that were effectively sent, or
-1 on errors.  Serious errors, such as when [arg sock] does not
identify a known WebSocket or when the connection is not stable yet
will generate errors that must be catched.

[call [cmd ::websocket::server] [arg sock]]

This command registers the (accept) socket [arg sock] as the
identifier fo an HTTP server that is capable of doing WebSockets.
Paths onto which this server will listen for incoming connections
should be declared using [cmd ::websocket::live].

[call [cmd ::websocket::live] [arg sock] [arg path] [arg cb] [opt [arg proto]]]

This procedure registers callbacks that will be performed on a
WebSocket compliant server registered with [cmd ::websocket::server]]
whenever a client connects to a matching path and protocol. 
[arg sock] is the listening socket of the websocket compliant server
declared using [cmd ::websocket::server].  [arg path] is a glob-style
path to match in client request, whenever this will occur.  [arg cb]
is the command to callback (see Callbacks).  [arg proto] is a
glob-style protocol name matcher.

[call [cmd ::websocket::test] [arg srvSock] [arg cliSock] [arg path] [opt [arg hdrs]] [opt [arg qry]]]

This procedure will test if the connection from an incoming client on
socket [arg cliSock] and on the path [arg path] is the opening of a
WebSocket stream within a known server [arg srvSock].  The incoming
request is not upgraded at once, instead a (temporary) context for the
incoming connection is created.  This allows server code to perform a
number of actions, if necessary, before the WebSocket stream
connection goes live.  The text is made by analysing the content of
the headers [arg hdrs] which should contain a dictionary list of the
HTTP headers of the incoming client connection.

The command will return [const 1] if this is an incoming WebSocket
upgrade request and [const 0] otherwise.

[call [cmd ::websocket::upgrade] [arg sock]]

Upgrade the socket [arg sock] that had been deemed by
[cmd ::websocket::test] to be a WebSocket connection request to a true
WebSocket as recognised by this library. As a result, the necessary
connection handshake will be sent to the client, and the command will
arrange for relevant callbacks to be made during the life of the
WebSocket, notably using the specifications described by 
[cmd ::websocket::live].

[call [cmd ::websocket::takeover] [arg sock] [arg handler] [opt [arg server]]]

Take over the existing opened socket [arg sock] to implement sending
and receiving WebSocket framing on top of the socket.  The procedure
arranges for [arg handler] to be called back whenever messages,
control messages or other important internal events are received or
occured.  [arg server] defaults to [const 0] and can be set to 
[const 1] (or a boolean that evaluates to true) to specify that this is a
WebSocket within a server.  Apart from specificities in the protocol,
servers should ping their clients at regular intervals in order to
keep the connection opened at all time.  When [arg server] is set to
true, the library will arrange to send these pings automatically.

[call [cmd ::websocket::conninfo] [arg sock] [arg what]]

Provides callers with some introspection facilities in order to get
some semi-internal information about an existing websocket
connection. Depending on the value of the [arg what] argument, the
procedure returns the following piece of information:

[list_begin definitions]
[def [const "peername"]] Name (preferred) or IP of remote end.
[def [const "sockname"]] or [const "name"] Name or IP of local end.
[def [const "closed"]] [const 1] if the connection is closed, [const 0] otherwise
[def [const "client"]] [const 1] if the connection is a client websocket, [const 0] otherwise
[def [const "server"]] [const 1] if the connection is a server websocket, [const 0] otherwise
[def [const "type"]] [const "server"] if the connection is a server websocket, [const "client"] otherwise.
[def [const "handler"]] The handler command associated to the websocket
[def [const "state"]] The state of the websocket, which can be one of:
[list_begin definitions]

[def [const "CONNECTING"]] Connection to remote end is in progress.
[def [const "CONNECTED"]] Connection is connected to remote end.
[def [const "CLOSED"]] Connection is closed.
[list_end]
[list_end]

[call [cmd ::websocket::find] [opt [arg host]] [opt [arg port]]]

Look among existing websocket connections for the ones that match the
hostname and port number filters passed as parameters.  This lookup
takes the remote end into account and the [arg host] argument is
matched both against the hostname (whenever available) and the IP
address of the remote end.  Both the [arg host] and [arg port]
arguments are glob-style string matching filters and default to 
[const "*"], i.e. will match any host and/or port number.

[call [cmd ::websocket::configure] [arg sock] [arg args]]

This command takes a number of dash-led options (and their values) to
configure the behaviour of an existing websocket connection.  The
recognised options are the following (they can be shortened to the
lowest common denominator):

[list_begin definitions]
[def [const "-keepalive"]] is the number of seconds between each
keepalive pings being sent along the connection.  A zero or negative
number will effectively turn off the feature.  In servers, 
[const "-keepalive"] defaults to 30 seconds, and in clients, no pings 
are initiated.
[def [const "-ping"]] is the text that is used during the automated
pings.  This text defaults to the empty string, leading to an empty ping.
[list_end]

[list_end]

[section Examples]

[para]

The following example opens a websocket connection to the echo
service, waits 400ms to ensure that the connection is really
established and sends a single textual message which should be echoed
back by the echo service.  A real example would probably use the
[const "connect"] callback to know when connection to the remote
server has been establish and would only send data at that time.

[example_begin]
package require websocket
::websocket::loglevel debug

proc handler { sock type msg } {
    switch -glob -nocase -- $type {
	co* {
	    puts "Connected on $sock"
	}
	te* {
	    puts "RECEIVED: $msg"
	}
	cl* -
	dis* {
	}
    }
    
}

proc test { sock } {
    puts "[lb]::websocket::conninfo $sock type[rb] from [lb]::websocket::conninfo $sock sockname[rb] to [lb]::websocket::conninfo $sock peername[rb]"
    
    ::websocket::send $sock text "Testing, testing..."
}

set sock [lb]::websocket::open ws://echo.websocket.org/ handler[rb]
after 400 test $sock
vwait forever
[example_end]

[manpage_end]
